#include "ColumbusWaypointModel.h"

#include "ColumbusController.h"

#include <QDebug>
#include <QSettings>

#include <QSqlQuery>
#include <QSqlError>
#include <QSqlRecord>
#include <QSqlTableModel>

#define CONFIG_KEY_WAYPOINT_CURRENT "navigation/waypoints/current"

#define CREATE_WAYPOINT_TABLE_QUERY "CREATE TABLE IF NOT EXISTS waypoints (id INTEGER PRIMARY KEY AUTOINCREMENT, name VARCHAR NOT NULL, latitude REAL NOT NULL, longitude REAL NOT NULL, altitude REAL NOT NULL);"

class ColumbusWaypointModelPrivate
{
public:
    ColumbusController  *controller;
    QModelIndex          destination;
};

ColumbusWaypointModel::ColumbusWaypointModel(ColumbusController *controller, QSqlDatabase db)
    : QSqlTableModel(controller, db)
{
    this->d = new ColumbusWaypointModelPrivate;

    d->controller  = controller;
    d->destination = QModelIndex();

    QObject::connect(d->controller, SIGNAL(positionUpdated(QGeoPositionInfo)),
                     this, SLOT(onPositionUpdate(QGeoPositionInfo)));

    qDebug() << "ColumbusWaypointModel: Checking and updating database schema if appropriate...";
    QSqlQuery query;
    if(!query.exec(CREATE_WAYPOINT_TABLE_QUERY))
    {
        qDebug() << "WaypointModel: Failed to execute SQL CREATE TABLE query:" << this->lastError().text();
        return;
    }

    qDebug() << "ColumbusWaypointModel: done.";
    this->setTable("waypoints");
    this->setEditStrategy(QSqlTableModel::OnRowChange);

    if(!this->select())
    {
        qDebug() << "ColumbusWaypointModel: Failed to execute SQL SELECT query:" << this->lastError().text();
        return;
    }

    qDebug() << "ColumbusWaypointModel: Found" << this->rowCount() << "records";

    this->setHeaderData(ColumbusWaypointModel::Id, Qt::Horizontal, tr("Id"));
    this->setHeaderData(ColumbusWaypointModel::Name, Qt::Horizontal, tr("Name"));
    this->setHeaderData(ColumbusWaypointModel::Latitude, Qt::Horizontal, tr("Latitude"));
    this->setHeaderData(ColumbusWaypointModel::Longitude, Qt::Horizontal, tr("Longitude"));
    this->setHeaderData(ColumbusWaypointModel::Altitude, Qt::Horizontal, tr("Altitude"));
    this->setHeaderData(ColumbusWaypointModel::Distance, Qt::Horizontal, tr("Distance"));
    this->setHeaderData(ColumbusWaypointModel::Bearing, Qt::Horizontal, tr("Bearing"));

    int destId = QSettings().value(CONFIG_KEY_WAYPOINT_CURRENT, -1).toInt();

    if(destId != 0)
    {
        for(int i = 0; i < this->rowCount(); i++)
        {
            QModelIndex index = this->index(i, ColumbusWaypointModel::Id);
            int id = this->data(index, Qt::DisplayRole).toInt();

            if(id == destId)
            {
                qDebug() << "ColumbusWaypointModel: Using saved destination waypoint with id:" << id;
                d->destination = index;
            }
        }
    }
}

ColumbusWaypointModel::~ColumbusWaypointModel()
{
    delete this->d;
}

QModelIndex ColumbusWaypointModel::destination() const
{
    return d->destination;
}

void ColumbusWaypointModel::setDestination(const QModelIndex &index)
{
    QModelIndex mi = this->index(index.row(), ColumbusWaypointModel::Id);
    int id = this->data(index, Qt::DisplayRole).toInt();

    d->destination = index;

    qDebug() << "ColumbusWaypointModel: Saving destination waypoint.";
    QSettings().setValue(CONFIG_KEY_WAYPOINT_CURRENT, id);
    emit this->destinationChanged(index);
}

QVariant ColumbusWaypointModel::data(const QModelIndex &idx, int role) const
{
    QVariant result = QSqlTableModel::data(idx, role);

    if(role == Qt::DisplayRole && idx.column() >= ColumbusWaypointModel::Distance)
    {
        ColumbusWaypoint waypoint = this->waypoint(idx);
        switch(idx.column())
        {
        case ColumbusWaypointModel::Distance:
            result = QVariant(d->controller->model()->currentPosition().distanceTo(waypoint.coordinate()));
            break;

        case ColumbusWaypointModel::Bearing:
            result = QVariant(d->controller->model()->currentPosition().azimuthTo(waypoint.coordinate()));
            break;

        default:
            result = QVariant();
            break;
        }
    }

    return result;
}

void ColumbusWaypointModel::onPositionUpdate(const QGeoPositionInfo&)
{
    emit this->dataChanged(this->index(0, 0), this->index(this->rowCount(), ColumbusWaypointModel::ColumnCount));
}

ColumbusWaypoint ColumbusWaypointModel::waypoint(const QModelIndex &index) const
{
    QString name        = this->data(this->index(index.row(), ColumbusWaypointModel::Name), Qt::DisplayRole).toString();
    double  latitude    = this->data(this->index(index.row(), ColumbusWaypointModel::Latitude), Qt::DisplayRole).toDouble();
    double  longitude   = this->data(this->index(index.row(), ColumbusWaypointModel::Longitude), Qt::DisplayRole).toDouble();

    return ColumbusWaypoint(name, QGeoCoordinate(latitude, longitude));
}

bool ColumbusWaypointModel::addWaypoint(const ColumbusWaypoint &waypoint)
{
    QSqlRecord record = this->record();

    record.setGenerated(ColumbusWaypointModel::Id, false);
    record.setValue(ColumbusWaypointModel::Name, waypoint.name());
    record.setValue(ColumbusWaypointModel::Latitude, waypoint.coordinate().latitude());
    record.setValue(ColumbusWaypointModel::Longitude, waypoint.coordinate().longitude());
    record.setValue(ColumbusWaypointModel::Altitude, 0.0f);

    bool result = this->insertRecord(-1, record);

    if(!result)
    {
        qDebug() << "ColumbusWaypointModel: Failed to execute SQL INSERT query:" << this->lastError().text();
    }

    return result;
}

bool ColumbusWaypointModel::addWaypoint(const QString &name, const QGeoCoordinate &coord)
{
    return this->addWaypoint(ColumbusWaypoint(name, coord));
}

bool ColumbusWaypointModel::setWaypoint(int row, const ColumbusWaypoint &waypoint)
{
    QSqlRecord record = this->record();

    record.setGenerated(ColumbusWaypointModel::Id, false);
    record.setValue(ColumbusWaypointModel::Name, waypoint.name());
    record.setValue(ColumbusWaypointModel::Latitude, waypoint.coordinate().latitude());
    record.setValue(ColumbusWaypointModel::Longitude, waypoint.coordinate().longitude());
    record.setValue(ColumbusWaypointModel::Altitude, 0.0f);

    bool result = this->setRecord(row, record);

    if(!result)
    {
        qDebug() << "ColumbusWaypointModel: Failed to execute SQL UPDATE query:" << this->lastError().text();
    }

    return result;
}

bool ColumbusWaypointModel::setWaypoint(int row, const QString &name, const QGeoCoordinate &coord)
{
    return this->setWaypoint(row, ColumbusWaypoint(name, coord));
}
