#include "ColumbusModel.h"
#include "ColumbusWaypoint.h"

#include "units.h"

#include <qnumeric.h>
#include <QDateTime>
#include <QStringList>
#include <QGeoCoordinate>

class ColumbusModelPrivate
{
public:
    QString name;
    QString comment;

    ColumbusModel::DisplayUnits displayUnits;

    qreal distanceCounter;

    qreal currentSpeed;
    qreal sumSpeed;
    qreal maximumSpeed;

    qreal currentAltitude;
    qreal minimumAltitude;
    qreal maximumAltitude;

    QGeoCoordinate::CoordinateFormat positionFormat;

    int travelTime;

    QDateTime tripStart;
    QDateTime tripEnd;

    QList<QGeoPositionInfo> datalog;
    QList<QGeoCoordinate>   tracklog;
};

ColumbusModel::ColumbusModel(QObject *parent)
    : QObject(parent)
{
    this->d = new ColumbusModelPrivate;
    d->name = "";
    d->comment = "";

    d->displayUnits = ColumbusModel::Metric;

    d->distanceCounter = 0;

    d->currentSpeed = 0;
    d->sumSpeed     = 0;
    d->maximumSpeed = 0;

    d->currentAltitude = 0;
    d->minimumAltitude = 100000;
    d->maximumAltitude = 0;

    d->travelTime = 0;
    d->tripStart = QDateTime::currentDateTimeUtc();
    d->tripEnd = QDateTime::currentDateTimeUtc();
}

ColumbusModel::~ColumbusModel()
{
    delete d;
}

QString ColumbusModel::name() const
{
    return d->name;
}

void ColumbusModel::setName(const QString &name)
{
    d->name = name;
    emit this->nameChanged(name);
    emit this->updated();
}

QString ColumbusModel::comment() const
{
    return d->comment;
}

void ColumbusModel::setComment(const QString &comment)
{
    d->comment = comment;
    emit this->commentChanged(comment);
    emit this->updated();
}

const QList<QGeoPositionInfo>* ColumbusModel::datalog() const
{
    return &d->datalog;
}

const QList<QGeoCoordinate>* ColumbusModel::tracklog() const
{
    return &d->tracklog;
}

qreal ColumbusModel::convertDistanceToUnits(DisplayUnits units, qreal distance)
{
    switch(units)
    {
    case ColumbusModel::Metric:
        return CONVERT_TO_KI(distance);
        break;

    case ColumbusModel::Imperial:
        return CONVERT_TO_MI(distance);
        break;

    case ColumbusModel::Nautical:
        return CONVERT_TO_NM(distance);
        break;
    }

    return distance;
}

qreal ColumbusModel::convertSpeedToUnits(DisplayUnits units, qreal speed)
{
    switch(units)
    {
    case ColumbusModel::Metric:
        return CONVERT_TO_KPH(speed);
        break;

    case ColumbusModel::Imperial:
        return CONVERT_TO_MPH(speed);
        break;

    case ColumbusModel::Nautical:
        return CONVERT_TO_KN(speed);
        break;
    }

    return speed;
}

qreal ColumbusModel::convertAltitudeToUnits(DisplayUnits units, qreal altitude)
{
    switch(units)
    {
    case ColumbusModel::Metric:
        break;

    case ColumbusModel::Imperial:
    case ColumbusModel::Nautical:
        return CONVERT_TO_FT(altitude);
        break;
    }

    return altitude;
}

ColumbusModel::DisplayUnits ColumbusModel::displayUnits() const
{
    return d->displayUnits;
}

QGeoCoordinate::CoordinateFormat ColumbusModel::positionFormat() const
{
    return this->positionFormat();
}

void ColumbusModel::setPositionFormat(QGeoCoordinate::CoordinateFormat format)
{
    d->positionFormat = format;
    emit this->positionFormatChanged(format);
}

void ColumbusModel::setDisplayUnits(DisplayUnits units)
{
    d->displayUnits = units;
    emit this->displayUnitsChanged(d->displayUnits);

    emit this->distanceUpdated(this->distance());

    emit this->currentSpeedUpdated(this->currentSpeed());
    emit this->averageSpeedUpdated(this->averageSpeed());
    emit this->maximumSpeedUpdated(this->maximumSpeed());

    emit this->currentAltitudeUpdated(this->currentAltitude());
    emit this->minimumAltitudeUpdated(this->minimumAltitude());
    emit this->maximumAltitudeUpdated(this->maximumAltitude());

    emit this->updated();
}

QString ColumbusModel::distanceUnits() const
{
    QString units;

    switch(d->displayUnits)
    {
    case ColumbusModel::Metric:
        units = "km";
        break;
    case ColumbusModel::Imperial:
        units = "mi";
        break;
    case ColumbusModel::Nautical:
        units = "nm";
        break;
    }

    return units;
}

QString ColumbusModel::speedUnits() const
{
    QString units;

    switch(d->displayUnits)
    {
    case ColumbusModel::Metric:
        units = "kph";
        break;
    case ColumbusModel::Imperial:
        units = "mph";
        break;
    case ColumbusModel::Nautical:
        units = "kn";
        break;
    }

    return units;
}

QString ColumbusModel::altitudeUnits() const
{
    QString units;

    switch(d->displayUnits)
    {
    case ColumbusModel::Metric:
        units = "m";
        break;

    case ColumbusModel::Imperial:
    case ColumbusModel::Nautical:
        units = "ft";
        break;
    }

    return units;
}

qreal ColumbusModel::distance() const
{
    return this->convertDistanceToUnits(d->displayUnits, d->distanceCounter);
}

qreal ColumbusModel::currentSpeed() const
{
    return this->convertSpeedToUnits(d->displayUnits, d->currentSpeed);
}

qreal ColumbusModel::averageSpeed() const
{
    return this->convertSpeedToUnits(d->displayUnits, d->sumSpeed / d->tracklog.count());
}

qreal ColumbusModel::maximumSpeed() const
{
    return this->convertSpeedToUnits(d->displayUnits, d->maximumSpeed);
}

qreal ColumbusModel::currentAltitude() const
{
    return this->convertAltitudeToUnits(d->displayUnits, d->currentAltitude);
}

qreal ColumbusModel::minimumAltitude() const
{
    return this->convertAltitudeToUnits(d->displayUnits, d->minimumAltitude);
}

qreal ColumbusModel::maximumAltitude() const
{
    return this->convertAltitudeToUnits(d->displayUnits, d->maximumAltitude);
}

qreal ColumbusModel::currentHeading() const
{
    if(d->datalog.length() > 0)
    {
        if(d->datalog.last().hasAttribute(QGeoPositionInfo::Direction))
        {
            qreal heading = d->datalog.last().attribute(QGeoPositionInfo::Direction);
            if(!qIsNaN(heading)) return heading;
        }
    }

    return 0.0;
}

QGeoCoordinate ColumbusModel::currentPosition() const
{
    return d->tracklog.last();
}

QString ColumbusModel::currentLatitude() const
{
    QGeoCoordinate coord = this->currentPosition();
    QStringList format = coord.toString((QGeoCoordinate::CoordinateFormat)d->positionFormat).split(',');
    return format.at(0).trimmed();
}

QString ColumbusModel::currentLongitude() const
{
    QGeoCoordinate coord = this->currentPosition();
    QStringList format = coord.toString((QGeoCoordinate::CoordinateFormat)d->positionFormat).split(',');
    return format.at(1).trimmed();
}

qreal ColumbusModel::currentHorizontalAccuracy() const
{
    if(d->datalog.last().hasAttribute(QGeoPositionInfo::HorizontalAccuracy))
    {
        qreal hacc = d->datalog.last().attribute(QGeoPositionInfo::HorizontalAccuracy);
        return hacc;
    }

    return -1;
}

qreal ColumbusModel::currentVerticalAccuracy() const
{
    if(d->datalog.last().hasAttribute(QGeoPositionInfo::VerticalAccuracy))
    {
        qreal vacc = d->datalog.last().attribute(QGeoPositionInfo::VerticalAccuracy);
        return vacc;
    }

    return -1;
}

int ColumbusModel::travelTime() const
{
    return 0;
}

int ColumbusModel::elapsedTime() const
{
    return d->tripStart.secsTo(d->tripEnd);
}

void ColumbusModel::onPositionUpdated(const QGeoPositionInfo &position)
{
    QGeoCoordinate coord = position.coordinate();

    if(position.hasAttribute(QGeoPositionInfo::Direction))
    {
        qreal direction = position.attribute(QGeoPositionInfo::Direction);
        if(!qIsNaN(direction))
        {
            emit this->currentHeadingUpdated(direction);
        }
    }

    if(position.hasAttribute(QGeoPositionInfo::GroundSpeed))
    {
        qreal speed = position.attribute(QGeoPositionInfo::GroundSpeed);

        d->currentSpeed = speed;
        emit this->currentSpeedUpdated(this->currentSpeed());

        if(speed > d->maximumSpeed)
        {
            d->maximumSpeed = speed;
            emit this->maximumSpeedUpdated(this->maximumSpeed());
        }

        d->sumSpeed += speed;
        emit this->averageSpeedUpdated(this->averageSpeed());
    }

    if(coord.type() == QGeoCoordinate::Coordinate3D)
    {
        d->currentAltitude = coord.altitude();
        emit this->currentAltitudeUpdated(this->currentAltitude());

        if(coord.altitude() < d->minimumAltitude)
        {
            d->minimumAltitude = coord.altitude();
            emit this->minimumAltitudeUpdated(this->minimumAltitude());
        }
        else if(coord.altitude() > d->maximumAltitude)
        {
            d->maximumAltitude = coord.altitude();
            emit this->maximumAltitudeUpdated(this->maximumAltitude());
        }
    }

    if(d->tracklog.count() == 0)
    {
        d->tripStart = QDateTime::currentDateTimeUtc();
    }
    else
    {
        d->distanceCounter += d->tracklog.last().distanceTo(coord);
        emit this->distanceUpdated(this->distance());
    }

    d->datalog.append(position);
    d->tracklog.append(coord);

    d->tripEnd = QDateTime::currentDateTimeUtc();
    emit this->elapsedTimeUpdated(this->elapsedTime());

    emit this->updated();
}
