/*
 * This file is part of the Liquid project.
 * It is header file of class Painter, which realizes a Core Module
 * of project. It provides drawing and all tools of editor.
 *
 * Copyright (C) 2009 Kirpichonock K.N. <kirpiche@cs.karelia.ru>
 * Copyright (C) 2009 Volkov A.A. <volkov@cs.karelia.ru>
 * Copyright (C) 2009 Dmitriev V.V. <vdmitrie@cs.karelia.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
//! Core Module.
/*!
  This module like a draw engine for graphic editor.
  It contains methods for diferrent drawing tools and implements drawing on
  workspace using accelerometer or touchscreen of a tablet.
*/

#ifndef PAINTER_H
#define PAINTER_H

#define BRUSH 1
#define DROP 2
#define ICE 3
#define BLUR_EFFECT 4
#define NO_EFFECT 0

#include "accelerometer/accelerometer.h"
#include "error_handler/errorhandler.h"
#include "filters/filters.h"
#include <QWidget>
#include <QVector3D>
#include <QPainter>
#include <QMouseEvent>
#include <QColor>
#include <math.h>
#include <QTimer>
#include <QContextMenuEvent>
#include "mce.h"

class Painter : public QWidget
{
Q_OBJECT
public:
    //! A constructor.
    Painter(QWidget *parent = 0, ErrorHandler *errorHandler = 0);
    //! An instance of Accelerometer class.
    Accelerometer *acc;    
    //! An instance of Error Handler class.
    ErrorHandler *eh;
    //! An instance of Filters class.
    Filters *filter;
    MCE* mce;


    //! A current color.
    QColor m_color;
    //! A current position of drawing tool.
    QPointF m_pos;
    //! A pixmap, on which pearl is drawing before it appears on workspace.
    QPixmap m_pearl_pixmap;
    //! A color of brush tool.
    QColor m_brushColor;
    //! A current size of brush.
    float m_size;
    float currentSize;
    float size;
    float vz;
    bool isClean;

    QPointF oldPos; /*!< Additional var for keeping old position. */

    bool isPainting; /*!< Flag, true when painting is active. */
    bool activeAcc; /*!< Flag, true when accelerometer mode is active. */
    bool isIce; /*!< Flag, true if user choose ice tool. */
    bool isModified; /*!< Flag, true if current image is modified and not saved. */
    bool activated;
    bool bg_mode;

    int currentEffect;

    int workspaceWidth;    

    //! Opens local image.
    /*!
      \param fileName - a name of graphic file.
    */
    bool openImage(const QString &fileName);
    //! Saves an image on tablet.
    /*!
      \param fileName - a name of graphic file.
      \param fileFormat - format in which it will be saved.
    */
    bool saveImage(const QString &fileName, const char *fileFormat);
    //! Loads an image to workspace.
    /*!
      Loads an image, which you are opening to workspace and converts
      it in a suitable format.
    */
    void setImage(const QImage &image);
    //! Create a new image.
    void createImage();    
    //! Sets current color for brush.
    void setColor(QColor color);
    //! Sets current tool.
    void setTool(int tool);
    //! Sets the position of brush.
    void setPosition(QPointF pos, bool isPainting);
    //! Sets the thickness for Brush or Eraser.
    void setThickness(float size);
    //! Draws a pearl on workspace.
    void drawPearl();
    //! Controls a movement of drop.
    bool checkBoundaries();
    //! Activates accelerometer mode.
    bool setAccMode(bool);
    //! Sets current effect.
    void setEffect(int effect);
    //! Sets height of workspace to full screen value or normal
    void setFullHeight(bool);
    //! Clear workspace.
    void clearWorkspace();

protected:
    //! Handler for mouse pressing.
    void mousePressEvent(QMouseEvent *e);
    //! Handler for mouse movement.
    void mouseMoveEvent(QMouseEvent *e);
    //! Handler for mouse released event.
    void mouseReleaseEvent(QMouseEvent *e);
    //! Overloading of standart QWidget function
    /*!
       All drawing is implemented in this function.
    */
    void paintEvent(QPaintEvent* e);    
    //! Overloading of standart ContextMenuEvent
    void contextMenuEvent(QContextMenuEvent *event);

private:
    //! Last position on X axis.
    int lastX;
    //! Last position on Y axis.
    int lastY;
    //! It uses for workspace. All drawing is implementing on it.
    QImage *theImage;    
    //! "Small" entry in brush size menu
    QAction *smallBrush;
    //! "Medium" entry in brush size menu
    QAction *mediumBrush;
    //! "Large" entry in brush size menu
    QAction *largeBrush;
    //! "Huge" entry in brush size menu
    QAction *hugeBrush;
    //! A list of actions for changing brush size
    QList<QAction* > sizeActions;
    void checkBrushSize(QAction *action);
    //! Creates entries for brush size menu
    void createActions();
    //! Timer, needed for delay of context menu appearance.
    QTimer *touchTimer;
    //! Height of workspace
    int workspaceHeight;

private slots:
    //! Sets brush position.
    /*!
      This slot triggers by Accelerometer::deviceOrientationChanged() signal.
      \param accVector3 - a vector with values of offset on 3 axises.
    */
    void setPosition(QVector3D accVector3);    
    //! Sets size of brush to small (30)
    void setSmallSize();
    //! Sets size of brush to medium (50)
    void setMediumSize();    
    //! Sets size of brush to large (75)
    void setLargeSize();
    //! Sets size of brush to huge (100)
    void setHugeSize();
    //! Shows context menu for brush size
    void showContextMenu();
    void rollHandler();
    void changeModeByShaking();
   
public slots:
    void changeState(const QString &);

signals:
    void disableAccelerometer();
    void activateAccelerometer();
};

#endif // PAINTER_H

