/*
* Copyright (C) 2005-2006  Koos Vriezen <koos.vriezen@gmail.com>
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef _KMPLAYER_TYPES_H_
#define _KMPLAYER_TYPES_H_

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>

#include <glib.h>
#include <gdk/gdk.h>
#include <gconf/gconf-client.h>

#include "io.h"
#include "log.h"

namespace KMPlayer {

/**
 * Type meant for screen coordinates
 */
class KMPLAYER_NO_EXPORT Single {
    int value;
    friend Single operator + (const Single s1, const Single s2);
    friend Single operator - (const Single s1, const Single s2);
    friend Single operator * (const Single s1, const Single s2);
    friend Single operator / (const Single s1, const Single s2);
    friend Single operator + (const Single s1, const int i);
    friend Single operator - (const Single s1, const int i);
    friend float operator * (const Single s, const float f);
    friend double operator * (const Single s, const double f);
    friend Single operator * (const int i, const Single s);
    friend float operator * (const float f, const Single s);
    friend double operator * (const double d, const Single s);
    friend Single operator / (const Single s, const int i);
    friend float operator / (const Single s, const float f);
    friend double operator / (const Single s, const double d);
    friend double operator / (const double d, const Single s);
    friend bool operator > (const Single s1, const Single s2);
    friend bool operator > (const Single s, const int i);
    friend bool operator > (const int i, const Single s);
    friend bool operator >= (const Single s1, const Single s2);
    friend bool operator == (const Single s1, const Single s2);
    friend bool operator != (const Single s1, const Single s2);
    friend bool operator < (const Single s1, const Single s2);
    friend bool operator < (const Single s, const int i);
    friend bool operator < (const int i, const Single s);
    friend bool operator <= (const Single s1, const Single s2);
    friend bool operator <= (const Single s, const int i);
    friend Single operator - (const Single s);
public:
    Single () : value (0) {}
    Single (const int v) : value (v << 8) {}
    Single (const float v) : value (int (256 * v)) {}
    Single (const double v) : value (int (256 * v)) {}
    Single & operator = (const Single s) { value = s.value; return *this; }
    Single & operator = (const int v) { value = v << 8; return *this; }
    Single & operator = (const float v) { value = int (256 * v); return *this; }
    Single & operator = (const double v) { value = int(256 * v); return *this; }
    Single & operator += (const Single s) { value += s.value; return *this; }
    Single & operator += (const int i) { value += (i << 8); return *this; }
    Single & operator -= (const Single s) { value -= s.value; return *this; }
    Single & operator -= (const int i) { value -= (i << 8); return *this; }
    Single & operator *= (const Single s);
    Single & operator *= (const float f) { value = int(value*f); return *this; }
    Single & operator /= (const int i) { value /= i; return *this; }
    Single & operator /= (const float f);
    operator int () const { return value >> 8; }
    operator double () const { return 1.0 * value / 256; }
    operator float () const { return 1.0 * value / 256; }
};

template <class T>
class KMPLAYER_NO_EXPORT Point {
public:
    Point ();
    Point (T _x, T _y);

    bool operator == (const Point<T> &p) const;
    bool operator != (const Point<T> &p) const;

    T x;
    T y;
};

template <class T>
class KMPLAYER_NO_EXPORT Size {
public:
    Size ();
    Size (T w, T h);

    bool isEmpty () const;
    bool operator == (const Size<T> &s) const;
    bool operator != (const Size<T> &s) const;

    T width;
    T height;
};

template <class T>
class KMPLAYER_NO_EXPORT Rect {
public:
    Rect ();
    Rect (T a, T b, T w, T h);
    Rect (T a, T b, const Size<T> &s);
    Rect (const Point<T> &point, const Size<T> &s);
    T x () const;
    T y () const;
    T width () const;
    T height () const;
    Rect<T> unite (const Rect<T> &r) const;
    Rect<T> intersect (const Rect<T> &r) const;
    bool operator == (const Rect<T> &r) const;
    bool operator != (const Rect<T> &r) const;
    bool isEmpty () const;

    Point<T> point;
    Size<T> size;
};

typedef Size<Single> SSize;
typedef Rect<Single> SRect;

typedef Size<int> ISize;
typedef Point<int> IPoint;
typedef Rect<int> IRect;
template <> Rect<int> Rect<int>::intersect (const Rect<int> &r) const;

Log & operator << (Log &, const Single &s);
Log & operator << (Log &, const SRect &s);
Log & operator << (Log &, const IRect &s);


class GLibTextStream {
    GString * m_gstring;
    gchar * m_utf8_str;
    const gchar * m_iter;
public:
    GLibTextStream ();
    GLibTextStream (const String *buffer); // for reading from buffer only
    GLibTextStream (const ByteArray *buffer); // for reading from buffer only
    ~GLibTextStream ();
    GLibTextStream & operator << (const String & str);
    GLibTextStream & operator << (const char * str);
    GLibTextStream & operator << (const Char & ch);
    GLibTextStream & operator << (int i);
    GLibTextStream & operator >> (Char & ch);
    //GLibTextStream & readRawBytes (char * b, unsigned length);
    bool atEnd () const;
    bool isEmpty () const;
    const gchar *at () const;
    String release ();
    String readLine ();
private:
    GLibTextStream (const GLibTextStream & other);
    GLibTextStream & operator = (const GLibTextStream & other);
};

typedef GLibTextStream TextStream;


class GMimeType {
public:
    static String findByContent (const char * data, int size);
    static String findByURL (const URL & url);
    static bool isBufferBinaryData (const char * data, int size);
};

typedef GMimeType MimeType;


class GConf {
    GConfClient * client;
public:
    GConf ();
    ~GConf ();
    String readEntry (const String & key, const String & def);
    int readNumEntry (const String & key, int def);
    double readNumEntry (const String & key, double def);
    void writeEntry (const String & key, const String & val);
    void writeEntry (const String & key, int val);
    void writeEntry (const String & key, double val);
    operator GConfClient* ();
};

typedef GConf Config;


class GColor {
    GdkColor * m_color;
public:
    GColor ();
    GColor (const String & desc);
    GColor (int rgb);
    GColor (const GColor & other);
    ~GColor ();
    void clear ();
    GColor & operator = (const GColor & other);
    //GColor & operator = (const String & desc);
    //GColor & operator = (int rgb);
    operator GdkColor * () const;
    unsigned rgb () const;
};

typedef GColor Color;
typedef unsigned Rgb;


class GPixbuf {
    GdkPixbuf * m_pixbuf;
public:
    GPixbuf ();
    GPixbuf (int w, int h);
    GPixbuf (const ByteArray & data);
    GPixbuf (GdkPixbuf * pix, bool skip_addref=false);
    GPixbuf (const GPixbuf & other); // only a g_object_ref on other.m_pixmap
    ~GPixbuf ();
    bool isNull () const;
    int width () const;
    int height () const;
    void fill (const Color & c = Color (0));
    void reset ();
    GPixbuf scale (int w, int h);
    operator GdkPixbuf * () const;
    GPixbuf & operator = (const GPixbuf & other);
    //GPixbuf & operator = (const Pixmap & pixmap);
    GPixbuf & operator = (GdkPixbuf * pixbuf);
    GdkPixbuf * pixbuf () const;
};

typedef GPixbuf Image;

//-----------------------------------------------------------------------------

inline Single & Single::operator *= (const Single s) {
    value = (((int64_t)value) * s.value) >> 8;
    return *this;
}

inline Single & Single::operator /= (const float f) {
    value = (int) (value / f);
    return *this;
}

inline Single operator + (const Single s1, const Single s2) {
    Single s;
    s.value = s1.value + s2.value;
    return s;
}

inline Single operator - (const Single s1, const Single s2) {
    Single s;
    s.value = s1.value - s2.value;
    return s;
}

inline Single operator * (const Single s1, const Single s2) {
    Single s;
    s.value = (((int64_t)s1.value) * s2.value) >> 8;
    return s;
}

inline Single operator / (const Single s1, const Single s2) {
    Single s;
    s.value = ((int64_t)s1.value << 8) / s2.value;
    return s;
}

inline Single operator + (const Single s, const int i) {
    return s + Single (i);
}

inline Single operator - (const Single s, const int i) {
    return s - Single (i);
}

inline Single operator * (const int i, const Single s) {
    Single s1;
    s1.value = s.value * i;
    return s1;
}

inline Single operator * (const Single s, const int i) {
    return i * s;
}
inline float operator * (const Single s, const float f) {
    return s.value * f / 256;
}

inline double operator * (const Single s, const double d) {
    return s.value * d / 256;
}

inline float operator * (const float f, const Single s) {
    return s.value * f / 256;
}

inline double operator * (const double d, const Single s) {
    return s.value * d / 256;
}

inline Single operator / (const Single s, const int i) {
    Single s1;
    s1.value = s.value / i;
    return s1;
}

inline float operator / (const Single s, const float f) {
    return (s.value / f ) / 256;
}

inline double operator / (const Single s, const double d) {
    return (s.value / d ) / 256;
}

inline double operator / (const double d, const Single s) {
    return (d * 256 / s.value);
}

inline bool
operator > (const Single s1, const Single s2) { return s1.value > s2.value; }

inline bool
operator > (const Single s, const int i) { return s > Single (i); }

inline bool
operator > (const int i, const Single s) { return Single (i) > s; }

inline bool
operator >= (const Single s1, const Single s2) { return s1.value >= s2.value; }

inline bool
operator == (const Single s1, const Single s2) { return s1.value == s2.value; }

inline bool
operator != (const Single s1, const Single s2) { return s1.value != s2.value; }

inline bool
operator < (const Single s1, const Single s2) { return s1.value < s2.value; }

inline bool
operator < (const Single s, const int i) { return s < Single (i); }

inline bool
operator < (const int i, const Single s) { return Single (i) < s; }

inline bool
operator <= (const Single s1, const Single s2) { return s1.value <= s2.value; }

inline bool
operator <= (const Single s, const int i) { return s <= Single (i); }

inline Single operator - (const Single s) {
    Single s1;
    s1.value = -s.value;
    return s1;
}

//-----------------------------------------------------------------------------

template <class T> inline Point<T>::Point () : x (0), y (0) {}

template <class T> inline Point<T>::Point (T _x, T _y) : x (_x), y (_y) {}

template <class T>
inline bool Point<T>::Point::operator == (const Point<T> &p) const {
    return x == p.x && y == p.y;
}

template <class T>
inline bool Point<T>::Point::operator != (const Point<T> &p) const {
    return !(*this == p);
}

//-----------------------------------------------------------------------------

template <class T> inline Size<T>::Size () : width (0), height (0) {}

template <class T> inline Size<T>::Size (T w, T h) : width (w), height (h) {}

template <class T> inline bool Size<T>::isEmpty () const {
    return width <= 0 || height <= 0;
}

template <class T>
inline bool Size<T>::Size::operator == (const Size<T> &s) const {
    return width == s.width && height == s.height;
}

template <class T>
inline bool Size<T>::Size::operator != (const Size<T> &s) const {
    return !(*this == s);
}

//-----------------------------------------------------------------------------

template <class T> inline Rect<T>::Rect () {}

template <class T> inline  Rect<T>::Rect (T a, T b, T w, T h)
 : point (a, b), size (w, h) {}

template <class T> inline Rect<T>::Rect (T a, T b, const Size<T> &s)
 : point (a, b), size (s) {}

template <class T> inline Rect<T>::Rect (const Point<T> &pnt, const Size<T> &s)
 : point (pnt), size (s) {}

template <class T> inline T Rect<T>::x () const { return point.x; }

template <class T> inline T Rect<T>::y () const { return point.y; }

template <class T> inline T Rect<T>::width () const { return size.width; }

template <class T> inline T Rect<T>::height () const { return size.height; }

template <class T> inline bool Rect<T>::operator == (const Rect<T> &r) const {
    return point == r.point && size == r.size;
}

template <class T> inline bool Rect<T>::operator != (const Rect<T> &r) const {
    return !(*this == r);
}

template <class T> inline bool Rect<T>::isEmpty () const {
    return size.isEmpty ();
}

template <class T> inline Rect<T> Rect<T>::unite (const Rect<T> &r) const {
    if (size.isEmpty ())
        return r;
    if (r.size.isEmpty ())
        return *this;
    T a (point.x < r.point.x ? point.x : r.point.x);
    T b (point.y < r.point.y ? point.y : r.point.y);
    return Rect<T> (a, b,
            ((point.x + size.width < r.point.x + r.size.width)
             ? r.point.x + r.size.width : point.x + size.width) - a,
            ((point.y + size.height < r.point.y + r.size.height)
             ? r.point.y + r.size.height : point.y + size.height) - b);
}

template <class T> inline Rect<T> Rect<T>::intersect (const Rect<T> &r) const {
    T a (point.x < r.point.x ? r.point.x : point.x);
    T b (point.y < r.point.y ? r.point.y : point.y);
    return Rect<T> (a, b,
            ((point.x + size.width < r.point.x + r.size.width)
             ? point.x + size.width : r.point.x + r.size.width) - a,
            ((point.y + size.height < r.point.y + r.size.height)
             ? point.y + size.height : r.point.y + r.size.height) - b);
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline GLibTextStream::GLibTextStream ()
    //: m_gstring (g_string_new ("")), m_utf8_str (0L), m_iter (0L) {}
    : m_gstring (g_string_sized_new (128)), m_utf8_str (0L), m_iter (0L) {}

KMPLAYER_NO_CDTOR_EXPORT
inline GLibTextStream::GLibTextStream (const ByteArray *buffer)
    : m_gstring (NULL),
      m_utf8_str (NULL),
      m_iter (buffer->size () ? buffer->data () : NULL) {}

KMPLAYER_NO_CDTOR_EXPORT inline GLibTextStream::~GLibTextStream () {
    if (m_gstring)
        g_string_free (m_gstring, true);
    else if (m_utf8_str)
        g_free (m_utf8_str);
}

KMPLAYER_NO_EXPORT inline GLibTextStream & GLibTextStream::operator << (int i) {
    return *this << String::number (i);
}

KMPLAYER_NO_EXPORT inline GLibTextStream & GLibTextStream::operator << (const char * str) {
    if (m_gstring)
        g_string_append (m_gstring, (const char *) str);
    return *this;
}

KMPLAYER_NO_EXPORT inline GLibTextStream & GLibTextStream::operator << (const String & str) {
    return *this << (const char *) str;
}

KMPLAYER_NO_EXPORT inline GLibTextStream & GLibTextStream::operator << (const Char & ch) {
    if (m_gstring)
        g_string_append_unichar (m_gstring, (gunichar) ch.unicode ());
    return *this;
}

/*KMPLAYER_NO_EXPORT inline
GLibTextStream & GLibTextStream::readRawBytes (char * b, unsigned len) {
    if (m_gstring)
        g_string_append_len (m_gstring, b, len);
    return *this;
}*/

KMPLAYER_NO_EXPORT inline bool GLibTextStream::atEnd () const {
    return (m_iter && !m_iter[0]) || (!m_gstring && !m_iter);
}

KMPLAYER_NO_EXPORT inline bool GLibTextStream::isEmpty () const {
    return (!m_gstring && (!m_utf8_str || !m_utf8_str [0]) ||
            (m_gstring && !m_gstring->len));
}

KMPLAYER_NO_EXPORT inline const gchar *GLibTextStream::at () const {
    return m_iter;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_EXPORT
inline void GConf::writeEntry (const String & key, const String & val) {
    gconf_client_set_string (client, (const char *) key, (const char *)val, 0L);
}

KMPLAYER_NO_EXPORT
inline void GConf::writeEntry (const String & key, int val) {
    gconf_client_set_int (client, (const char *) key, val, 0L);
}

KMPLAYER_NO_EXPORT
inline void GConf::writeEntry (const String & key, double val) {
    gconf_client_set_float (client, (const char *) key, val, 0L);
}

KMPLAYER_NO_EXPORT inline GConf::operator GConfClient * () {
    return client;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline GColor::GColor () : m_color (0L) {}

KMPLAYER_NO_CDTOR_EXPORT inline GColor::GColor (const GColor & c)
    : m_color (c.m_color ? gdk_color_copy (c.m_color) : 0L) {}

KMPLAYER_NO_EXPORT inline void GColor::clear () {
    if (m_color) {
        //gdk_colormap_free_colors (gdk_colormap_get_system (), m_color, 1);
        gdk_color_free (m_color); // ??
        m_color = 0L;
    }
}

KMPLAYER_NO_CDTOR_EXPORT inline GColor::~GColor () {
    clear ();
}

KMPLAYER_NO_EXPORT inline GColor & GColor::operator = (const GColor & c) {
    clear ();
    m_color = c.m_color ? gdk_color_copy (c.m_color) : 0L;
    return *this;
}

KMPLAYER_NO_EXPORT inline GColor::operator GdkColor * () const {
    return m_color;
}

KMPLAYER_NO_EXPORT inline unsigned GColor::rgb () const {
    return m_color ? ((m_color->red >> 8) << 16) | ((m_color->green >> 8) << 8) | (m_color->blue >> 8) : 0L;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_EXPORT inline GPixbuf::GPixbuf () : m_pixbuf (0L) {}

KMPLAYER_NO_EXPORT inline bool GPixbuf::isNull () const { return !m_pixbuf; }

KMPLAYER_NO_EXPORT inline
GPixbuf::operator GdkPixbuf * () const { return m_pixbuf; }

KMPLAYER_NO_EXPORT
inline GdkPixbuf * GPixbuf::pixbuf () const { return m_pixbuf; }

KMPLAYER_NO_EXPORT inline int GPixbuf::width () const {
    return m_pixbuf ? gdk_pixbuf_get_width (m_pixbuf) : 0;
}

KMPLAYER_NO_EXPORT inline int GPixbuf::height () const {
    return m_pixbuf ? gdk_pixbuf_get_height (m_pixbuf) : 0;
}

} // namespace KMPlayer

#include "triestring.h"

#endif  // _KMPLAYER_TYPES_H_

