/* This file is part of the KMPlayer project
 *
 * Copyright (C) 2005 Koos Vriezen <koos.vriezen@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * until boost gets common, a more or less compatable one ..
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <gtk/gtkmain.h>
#include <gio/gio.h>

#include "kmplayertypes.h"

using namespace KMPlayer;


KMPLAYER_NO_EXPORT Log &operator << (Log &log, const Single &s) {
    fprintf (stderr, "%.3f", (double)s);
    return log;
}

KMPLAYER_NO_EXPORT Log &operator << (Log &log, const SRect &r) {
    fprintf (stderr, "SRect(%.1f, %.1f, %.1f, %.1f)",
           (double)r.x(), (double)r.y(), (double)r.width(), (double)r.height());
    return log;
}

KMPLAYER_NO_EXPORT Log &operator << (Log &log, const IRect &r) {
    fprintf (stderr, "SRect(%d, %d, %d, %d)",
            r.x (), r.y (), r.width (), r.height ());
    return log;
}

//-----------------------------------------------------------------------------

bool GMimeType::isBufferBinaryData (const char *data, int size)
{
    const gchar *end;
    if (g_utf8_validate ((gchar *) data, size, &end))
        return false;
    int valid = end - data;
    if (size - valid < 6)
        return g_utf8_get_char_validated (end, size - valid) != (gunichar)-2;
    return true;
}

String GMimeType::findByContent (const char *data, int nr) {
    if (nr <= 0)
        return String();
    gboolean pb;
    return String (g_content_type_guess (NULL, (const guchar *)data, nr, &pb));
}

String GMimeType::findByURL (const URL &url) {
    gboolean pb;
    if (url.isLocalFile ())
        return String (g_content_type_guess (url.path (), NULL, 200, &pb));
    warningLog () << "GMimeType::findByURL not supported on remote links\n";
    return String ();
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT
GLibTextStream::GLibTextStream (const String *buffer)
    : m_gstring (NULL), m_utf8_str (NULL), m_iter ((const char *) *buffer) {
}

GLibTextStream & GLibTextStream::operator >> (Char & ch) {
    if (m_gstring) {
        m_utf8_str = g_string_free (m_gstring, false);
        m_gstring = 0L;
        m_iter = m_utf8_str;
    }
    if (m_iter) {
        ch = Char (g_utf8_get_char (m_iter));
        m_iter = g_utf8_next_char (m_iter);
    }
}

#if 0
KMPLAYER_NO_EXPORT
GLibTextStream & GLibTextStream::operator = (const GLibTextStream & s) {
    if (m_gstring)
        g_string_free (m_gstring, true);
    else if (m_utf8_str)
        g_free (m_utf8_str);
    m_utf8_str = 0L;
    m_iter = 0L;
    m_gstring = g_string_new (s.m_gstring ? s.m_gstring->str :
            (s.m_utf8_str ? s.m_utf8_str : ""));
    return *this;
}
#endif

KMPLAYER_NO_EXPORT String GLibTextStream::release () {
    GLibString str;
    if (m_gstring) {
        str.takeUtf8 (g_string_free (m_gstring, false));
        m_gstring = 0L;
    }
    return str;
}

String GLibTextStream::readLine () {
    String str;
    Char ch;
    while (!atEnd ()) {
        *this >> ch;
        if (ch == '\r' || ch == '\n')
            break;
        str += ch;
    }
    const gchar *iter = m_iter;
    while (!atEnd ()) { // FIXME: current is skipping emty lines as well
        *this >> ch;
        if (ch != '\r' && ch != '\n') {
            m_iter = iter;
            break;
        }
        iter = m_iter;
    }
    return str;
}

//-----------------------------------------------------------------------------
#if 0
namespace KMPlayer {

    class GnomeVFSGetJob : public IOJob {
    public:
        GnomeVFSGetJob (IOJobListener * rec, const String & u);
        ~GnomeVFSGetJob ();
        void kill (bool quiet = true);
        void finished ();
        bool error ();

        GnomeVFSAsyncHandle * network_handle;
        IOJobListener * m_receiver;
        bool job_closed;
        bool job_error;
        bool keep_quiet;
        bool continuing;
        char read_buf [2048];
    };

    IOJob * asyncGet (IOJobListener * receiver, const String & url) {
        return new GnomeVFSGetJob (receiver, url);
    }
}

//static GTree *uri_info;

//void getRedirection (gpointer data, gpointer d) {
//    debugLog() << " getRedirection: " << (char *)data << endl;
//}

//static void globalVFSAsyncModuleCallback (gconstpointer in,
//        gsize in_size,
//        gpointer out,
//        gsize out_size,
//        gpointer callback_data/*,
//        GnomeVFSModuleCallbackResponse response,
//        gpointer response_data*/) {
//    GnomeVFSModuleCallbackReceivedHeadersIn *in_args =
//        (GnomeVFSModuleCallbackReceivedHeadersIn *)in;
//    debugLog() << "globalVFSAsyncModuleCallback " << 
//        in_args->uri->text << endl;
//    g_list_foreach (in_args->headers, getRedirection, NULL);
//    //response (response_data);
//}


static void cb_async_get_close (GnomeVFSAsyncHandle *handle, GnomeVFSResult result, gpointer p) {
    //debugLog () << "cb_async_get_close " << endl;
    if (!p)
        return;
    GnomeVFSGetJob * get = static_cast <GnomeVFSGetJob *> (p);
    get->finished ();
}

static void cb_async_get_read (GnomeVFSAsyncHandle *handle, GnomeVFSResult result, gpointer buffer, GnomeVFSFileSize bytes_requested, GnomeVFSFileSize bytes_read, gpointer p) {
    GnomeVFSGetJob * get = static_cast <GnomeVFSGetJob *> (p);
    if (get->job_closed)
        return;
    get->continuing = true;
    if (bytes_read > 0) {
    debugLog () << "cb_async_get_read " << handle << endl;
        ByteArray data (bytes_read);
        memcpy (data.data (), (const char *) buffer, bytes_read);
        get->m_receiver->jobData (get, data);
    }
    //debugLog () << "cb_async_get_read " << int (bytes_read) << " canceled:" << get->job_closed << " not ok:" << (result != GNOME_VFS_OK) << " less:" << (bytes_read < bytes_requested) << " killed:" << !get->continuing << endl;
    if (result != GNOME_VFS_OK || bytes_read <= 0 || !get->continuing) {
        // || bytes_read < bytes_requested 
        get->job_closed = true;
        get->job_error = result != GNOME_VFS_ERROR_EOF;
        gnome_vfs_async_cancel (handle);
        gnome_vfs_async_close (handle, cb_async_get_close, p);
    } else {
        gnome_vfs_async_read (handle, get->read_buf, sizeof (get->read_buf) -1, cb_async_get_read, p);
    }
    get->continuing = false;
}

static void cb_async_get_open (GnomeVFSAsyncHandle *handle, GnomeVFSResult result, gpointer p) {
    debugLog () << "cb_async_get_open " << handle << endl;
    GnomeVFSGetJob * get = static_cast <GnomeVFSGetJob *> (p);
    if (get->job_closed)
        return;
    if (result != GNOME_VFS_OK)
        get->finished ();  // kills get
    else
        gnome_vfs_async_read (handle, get->read_buf, sizeof (get->read_buf) -1, cb_async_get_read, p);
}

GnomeVFSGetJob::GnomeVFSGetJob (IOJobListener * rec, const String &url)
 : m_receiver (rec), network_handle (0L),
   job_closed (false), job_error (false),
   keep_quiet (false), continuing (false) {

    //if (!uri_info) {
    //    uri_info = g_tree_new ((GCompareFunc)::strcmp);
    //    gnome_vfs_module_callback_set_default
    //        (GNOME_VFS_MODULE_CALLBACK_HTTP_RECEIVED_HEADERS,
    //         globalVFSAsyncModuleCallback,
    //         uri_info,
    //         NULL);
    //}
    gnome_vfs_async_open (&network_handle, (const char *) url, GNOME_VFS_OPEN_READ, 0, cb_async_get_open , static_cast <void *> (this));
    debugLog() << "GnomeVFSGetJob " << network_handle << " " << url << endl;
}

GnomeVFSGetJob::~GnomeVFSGetJob () {
    if (network_handle && !job_closed) {
        gnome_vfs_async_cancel (network_handle);
        gnome_vfs_async_close (network_handle, cb_async_get_close, 0L);
    }
}

void GnomeVFSGetJob::kill (bool quiet) {
    keep_quiet = quiet;
    if (network_handle && !job_closed) {
        if (!continuing) {
            job_closed = true;
            gnome_vfs_async_cancel (network_handle);
            gnome_vfs_async_close (network_handle, cb_async_get_close, static_cast <void *> (this));//
        } else
            continuing = false;  // killed from jobData receiver callback
    } else if (!quiet)
        m_receiver->jobResult (this);
}

void GnomeVFSGetJob::finished () {
    network_handle = 0L;
    if (!keep_quiet)
        m_receiver->jobResult (this);
    delete this;
}

bool GnomeVFSGetJob::error () {
    return job_error;
}
#endif

static bool gconf_initialized = false;

KMPLAYER_NO_CDTOR_EXPORT GConf::GConf () {
    if (!gconf_initialized) {
        gconf_initialized = true;
        g_type_init ();
    }
    client = gconf_client_get_default ();
}

KMPLAYER_NO_CDTOR_EXPORT GConf::~GConf () {
    gconf_client_clear_cache (client);
    g_object_unref (client);
}

KMPLAYER_NO_EXPORT
String GConf::readEntry (const String & key, const String & def) {
    gchar *val = gconf_client_get_string (client, (const char *) key, 0L);
    if (!val)
        return def;
    String s;
    s.take (val);
    return s;
}

KMPLAYER_NO_EXPORT int GConf::readNumEntry (const String & key, int def) {
    GConfValue *val = gconf_client_get (client, (const char *) key, 0L);
    if (val) {
        int i = gconf_value_get_int (val);
        gconf_value_free (val);
        return i;
    }
    return def;
}

KMPLAYER_NO_EXPORT double GConf::readNumEntry (const String & key, double def) {
    GError * err;
    double val = gconf_client_get_float (client, (const char *) key, &err);
    bool has_err = !!err;
    g_error_free (err);
    return has_err ? def : val;
}

//-----------------------------------------------------------------------------


#include <glib/gprintf.h>

static GdkColor * colorFromRGB (int rgb) {
    GdkColormap * cmap = gdk_colormap_get_system ();
    GdkColor *color = g_new (GdkColor, 1);
    GdkColor *rcolor = NULL;
    char buf [16];
    g_sprintf (buf, "#%02X%02X%02X", (rgb & 0x00FF0000) >> 16, (rgb & 0x0000FF00) >> 8, (rgb & 0x000000FF));
    if (gdk_color_parse (buf, color) &&
                (gdk_colormap_alloc_color (cmap, color, 0, 0) ||
                 gdk_colormap_alloc_color (cmap, color, 0, 1)))
        rcolor = gdk_color_copy (color);
    g_free (color);
    return rcolor;
}

KMPLAYER_NO_CDTOR_EXPORT GColor::GColor (const String & desc) : m_color (NULL) {
    GdkColormap * cmap = gdk_colormap_get_system ();
    GdkColor *color = g_new (GdkColor, 1);
    if (!(gdk_color_parse ((const char *) desc, color) &&
                gdk_colormap_alloc_color (cmap, color, 0, 1))) {
        if (desc.lower () == "silver")
            m_color = colorFromRGB (0xbebebe);
        // TODO add more unsupported real live colors
        if (!m_color)
            errorLog () << "Could not parse " << desc << " color" << endl;
    } else {
        m_color = gdk_color_copy (color);
    }
    g_free (color);
    //debugLog () << desc << " -> " << int (m_color ? m_color->pixel : 0) << endl;
}

KMPLAYER_NO_CDTOR_EXPORT GColor::GColor (int rgb)
    : m_color (colorFromRGB(rgb)) {
    //debugLog () << "GColor::GColor (int " << rgb << ") -> " << int (m_color ? m_color->pixel : -1) << " str:" << buf << endl;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::GPixbuf (int w, int h) {
    GdkVisual * visual = gdk_visual_get_system ();//gdk_rgb_get_visual
    m_pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, false, visual->depth, w, h);
}

KMPLAYER_NO_CDTOR_EXPORT
GPixbuf::GPixbuf (const ByteArray & data) : m_pixbuf (0L) {
    GdkPixbufLoader * pix_loader = gdk_pixbuf_loader_new ();
    GError *gerror = 0L;
    if (gdk_pixbuf_loader_write (pix_loader, (const guchar *) data.data (), data.size (), &gerror) && gdk_pixbuf_loader_close (pix_loader, 0L)) {
        m_pixbuf = gdk_pixbuf_loader_get_pixbuf (pix_loader);
        g_object_ref (m_pixbuf);
    } else
        g_free (gerror);
    g_object_unref (pix_loader);
}

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::GPixbuf (GdkPixbuf * pb, bool skip_addref)
  : m_pixbuf (pb) {
    if (!skip_addref && m_pixbuf)
        g_object_ref (m_pixbuf);
}

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::GPixbuf (const GPixbuf & other)
 : m_pixbuf (other.m_pixbuf) {
    if (m_pixbuf)
        g_object_ref (m_pixbuf);
}

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::~GPixbuf () {
    reset ();
}

KMPLAYER_NO_EXPORT GPixbuf & GPixbuf::operator = (const GPixbuf & other) {
    if (&other != this) {
        reset ();
        m_pixbuf = other.m_pixbuf;
        if (m_pixbuf)
            g_object_ref (m_pixbuf);
    }
    return *this;
}

KMPLAYER_NO_EXPORT void GPixbuf::reset () {
    if (m_pixbuf) {
        g_object_unref (m_pixbuf);
        m_pixbuf = 0L;
    }
}

KMPLAYER_NO_EXPORT void GPixbuf::fill (const Color & c) {
    if (m_pixbuf)
        gdk_pixbuf_fill (m_pixbuf, c.rgb ());
}

KMPLAYER_NO_EXPORT GPixbuf GPixbuf::scale (int w, int h) {
    if (m_pixbuf)
        return GPixbuf (gdk_pixbuf_scale_simple (m_pixbuf, w, h, w > width () ? GDK_INTERP_NEAREST : GDK_INTERP_BILINEAR), true);
}


/*
void        gdk_pixbuf_xlib_init            (Display *display,
                                             int screen_num);


GdkPixbuf*  gdk_pixbuf_new_from_file        (const char *filename,
                                             GError **error);
// gdk_pixbuf_new_from_data gdk_pixbuf_new_from_xpm_data
void        gdk_pixbuf_copy_area            (const GdkPixbuf *src_pixbuf,
                                             int src_x,
                                             int src_y,
                                             int width,
                                             int height,
                                             GdkPixbuf *dest_pixbuf,
                                             int dest_x,
                                             int dest_y);

void        gdk_pixbuf_scale                (const GdkPixbuf *src,
                                             GdkPixbuf *dest,
                                             int dest_x,
                                             int dest_y,
                                             int dest_width,
                                             int dest_height,
                                             double offset_x,
                                             double offset_y,
                                             double scale_x,
                                             double scale_y,
                                             GdkInterpType interp_type);

g_object_unref() 

// http://developer.gnome.org/doc/API/2.0/gdk-pixbuf/gdk-pixbuf-animation.html

GdkPixbuf*  gdk_pixbuf_add_alpha            (const GdkPixbuf *pixbuf,
                                             gboolean substitute_color,
                                             guchar r,
                                             guchar g,
                                             guchar b);


typedef struct {
  gint x;
  gint y;
  gint width;
  gint height;
} GdkRectangle;

void        gdk_rectangle_union             (GdkRectangle *src1,
                                             GdkRectangle *src2,
                                             GdkRectangle *dest);

*/
