/*
  Kandis - A tea steepming timer.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <Lum/Audio/PlaySound.h>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Def/Menu.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Integer.h>

#include <Lum/OS/Main.h>

#include <Lum/OS/Manager/Repository.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Dialog.h>
#include <Lum/Panel.h>
#include <Lum/Toggle.h>

#include "Configuration.h"
#include "Selector.h"
#include "StopWatch.h"

#include "config.h"

static Lum::Def::AppInfo info;

class MainDialog : public Lum::Dialog
{
private:
  enum State {
    stateUninitialized,
    stateRunning,
    statePaused,
    stateStoped,
    stateAlerting
  };

private:
  Lum::Model::ActionRef  aboutAction;
  Lum::Model::ActionRef  startAction;
  Lum::Model::BooleanRef paused;
  Lum::Model::ActionRef  stopAction;
  Lum::Model::ActionRef  timer;
  Lum::Model::SizeTRef   seconds;

  size_t                 initialSeconds;
  Lum::Base::Timer       time;

  State                  state;
  StopWatch              *value;
  Lum::Base::SystemTime  alertTime;
  Lum::Audio::PlaySound  *beepSound;

public:
  MainDialog()
  : aboutAction(new Lum::Model::Action()),
    startAction(new Lum::Model::Action()),
    paused(new Lum::Model::Boolean(false)),
    stopAction(new Lum::Model::Action()),
    timer(new Lum::Model::Action()),
    seconds(new Lum::Model::SizeT()),
    initialSeconds(0),
    state(stateUninitialized),
    value(new StopWatch()),
    beepSound(NULL)
  {
    Lum::Base::Path soundPath;

    soundPath.SetNativeDir(Lum::Base::Path::GetCWD());
    soundPath.AppendDir(L"data");
    soundPath.AppendDir(L"sounds");
    soundPath.SetBaseName(L"alarm.wav");

#if defined(APP_DATADIR)
    if (!soundPath.Exists()) {
      soundPath.SetNativeDir(Lum::Base::StringToWString(APP_DATADIR));
      soundPath.AppendDir(L"sounds");
      soundPath.SetBaseName(L"alarm.wav");
    }
#endif

    if (soundPath.Exists()) {
      beepSound=Lum::Audio::GetPlaySound();
      if (beepSound!=NULL) {
        beepSound->SetFilename(soundPath.GetPath(),Lum::Audio::PlaySound::typeWav);
      }
    }

    Observe(GetOpenedAction());
    Observe(GetClosedAction());

    Observe(seconds);

    Observe(value->GetClickAction());

    Observe(aboutAction);
    Observe(startAction);
    Observe(paused);
    Observe(stopAction);
    Observe(timer);

    // Settings for initial state
    value->GetClickAction()->Enable();
    startAction->Disable();
    paused->Disable();
    stopAction->Disable();
  }

  ~MainDialog()
  {
    delete beepSound;
  }

  void SetValue()
  {
    if (!seconds.Valid() ||
        seconds->IsNull()) {
      value->SetText(L"-:--");
    }
    else {
      value->SetText(Lum::Base::TimeSpanToWString(seconds->Get()));
    }
  }

  void PlayAlert()
  {
    if (beepSound!=NULL) {
      beepSound->Play();
    }
    else {
      Lum::OS::display->Beep();
    }
  }

  void AllowScreenBlanking(bool allow)
  {
    Lum::OS::Manager::DisplayManager *displayManager=Lum::OS::Manager::repository->GetDisplayManager();

    if (displayManager!=NULL) {
      displayManager->AllowScreenBlanking(allow);
    }
  }

  void SetState(State state)
  {
    if (this->state==state) {
      return;
    }

    this->state=state;

    // Visual representation

    switch (state) {
    case stateUninitialized:
      assert(false);
      break;
    case stateRunning:
      value->GetClickAction()->Disable();
      value->SetAlarm(false);
      startAction->Enable();
      paused->Enable();
      stopAction->Enable();
      AllowScreenBlanking(false);
    case statePaused:
      value->GetClickAction()->Disable();
      value->SetAlarm(false);
      startAction->Enable();
      paused->Enable();
      stopAction->Enable();
      AllowScreenBlanking(false);
      break;
    case stateStoped:
      value->GetClickAction()->Enable();
      value->SetAlarm(false);
      startAction->Enable();
      paused->Disable();
      stopAction->Disable();
      AllowScreenBlanking(true);
      break;
    case stateAlerting:
      value->GetClickAction()->Disable();
      value->SetAlarm(true);
      startAction->Enable();
      paused->Disable();
      stopAction->Enable();
      AllowScreenBlanking(false);
      break;
    }

    // Timer & state

    switch (state) {
    case stateUninitialized:
      assert(false);
      break;
    case stateRunning:
      paused->Set(false);
      if (time.IsPausing()) {
        time.Resume();
      }
      else {
        time.Start();
      }
      seconds->Set(initialSeconds-time.GetTime());
      alertTime.SetToNow();
      alertTime.Add(seconds->Get());
      Lum::OS::display->AddTimer(0,100000,timer);
      break;
    case statePaused:
      seconds->Set(initialSeconds-time.GetTime());
      time.Pause();
      Lum::OS::display->RemoveTimer(timer);
      break;
    case stateStoped:
      if (time.IsStarted()) {
        seconds->Set(initialSeconds-time.GetTime());
      }
      time.Reset();
      paused->Set(false);
      Lum::OS::display->RemoveTimer(timer);
      break;
    case stateAlerting:
      seconds->Set(initialSeconds-time.GetTime());
      time.Reset();
      paused->Set(false);
      PlayAlert();
      alertTime.SetToNow();
      alertTime.Add(3);
      Lum::OS::display->RemoveTimer(timer);
      Lum::OS::display->AddTimer(0,100000,timer);
      break;
    }
  }

  virtual void PreInit()
  {
    value->SetFlex(true,true);

    SetMain(Lum::VPanel::Create(true,true)
            ->Add(value)
            ->AddSpace()
            ->Add(Lum::ButtonRow::Create(true,false)
                  ->Add(Lum::Button::Create(_(L"BUTTON_SETTINGS",L"S_ettings..."),value->GetClickAction(),true,true))
                  ->Add(Lum::Button::Create(_(L"BUTTON_START",L"_Start"),startAction,true,true))
                  ->Add(Lum::Toggle::Create(_(L"BUTTON_PAUSE",L"Pause"),paused,true,true))
                  ->Add(Lum::Button::Create(_(L"BUTTON_STOP",L"Sto_p"),stopAction,true,true))));

    Lum::Def::Menu *menu=Lum::Def::Menu::Create();

    menu
      ->GroupProject()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->GroupEdit()
        ->ActionSettings(value->GetClickAction())
      ->End()
      ->GroupHelp()
        //->AddMenuItemAction(_ld(menuHelpHelp),NULL)
        ->ActionAbout(aboutAction)
      ->End();

    SetMenu(menu);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      LoadConfig();
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      if (configurationChanged && !SaveConfig()) {
        Lum::Dlg::Msg::ShowOk(this,
                              _(L"ERR_CONFIG_SAVE_CAPTION",
                                L"Error while storing configuration..."),
                              _(L"ERR_CONFIG_SAVE_TEXT",
                                L"There was an error while storing the configuration!"));
      }
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==value->GetClickAction() &&
             value->GetClickAction()->IsFinished() &&
             (state==stateStoped || state==stateUninitialized)) {
      Selector *selector;

      selector=new Selector(seconds);
      selector->SetParent(GetWindow());
      if (selector->Open()) {
        selector->EventLoop();
        selector->Close();
      }

      if (selector->Success()) {
        initialSeconds=seconds->Get();
        SetState(stateStoped);
      }
      delete selector;

    }
    else if (model==startAction &&
             startAction->IsFinished()) {
      SetState(stateStoped);
      SetState(stateRunning);
    }
    else if (model==paused) {
      if (state==stateRunning &&
          paused->Get()) {
        SetState(statePaused);
      }
      else if (state==statePaused &&
               !paused->Get()) {
        SetState(stateRunning);
      }
    }
    else if (model==stopAction &&
             stopAction->IsFinished()) {
      SetState(stateStoped);
    }
    else if (model==timer &&
             timer->IsFinished()) {
      if (state==stateRunning) {
        Lum::Base::SystemTime now;

        if (now>=alertTime) {
          SetState(stateAlerting);
        }
        else {
          seconds->Set(initialSeconds-time.GetTime());
          Lum::OS::display->AddTimer(0,100000,timer);
        }
      }
      else if (state==stateAlerting) {
        Lum::Base::SystemTime now;

        if (now.GetTime()>alertTime) {
          PlayAlert();
          alertTime.SetToNow();
          alertTime.Add(3);
        }

        Lum::OS::display->AddTimer(0,100000,timer);
      }
    }
    else if (model==seconds) {
      SetValue();
    }

    Dialog::Resync(model,msg);
  }

};

class Main : public Lum::OS::MainDialog<MainDialog>
{
public:
  bool Prepare()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"A tea steeping timer"));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2007, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    return Lum::OS::MainDialog<MainDialog>::Prepare();
  }

  void Cleanup()
  {
    FreeConfig();

    Lum::OS::MainDialog<MainDialog>::Cleanup();
  }
};

LUM_MAIN(Main,L"Kandis")
