/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtGui/QGraphicsPixmapItem>
#include <QtCore/QDebug>
#include <QtCore/QTimeLine>
#include <math.h>
#include "pointer.h"
#include "reader.h"
#include "graphicsscene.h"
#include "odometer.h"

namespace
{
    const GraphicsElement::AttributeDetails ATTRIBUTES[Pointer::ATTRIBUTE_COUNT] =
    {
     {"xpos", true},
     {"ypos", true},
     {"zpos", true},
     {"visiblewhen", false},
     {"src", false},
     {"zeroangle", true},
     {"fullangle", true},
     {"zerospeed", true},
     {"fullspeed", true},
     {"xrotationpoint", true},
     {"yrotationpoint", true}
    };

    int const ANIMATION_TIME = 500;
    int const ANIMATION_FRAMES = 50;
    int const ANIMATION_UPDATEINTERVAL = 20;
}

Pointer::Pointer(Reader* reader, bool animate): GraphicsElement(reader),
xRotationPoint_(0), yRotationPoint_(0), x_(0), y_(0), zeroAngle_(0),
fullAngle_(180), zeroSpeed_(0), fullSpeed_(220), targetAngle_(0),
startAngle_(0), angle_(-1), imageSet_(false), animate_(animate), timer_(0)
{
    element_ = new QGraphicsPixmapItem();

    if(animate_)
    {
        timer_ = new QTimeLine(ANIMATION_TIME);
        timer_->setFrameRange(0, ANIMATION_FRAMES);
        timer_->setUpdateInterval(ANIMATION_UPDATEINTERVAL);
        connect(timer_, SIGNAL(frameChanged(int)), this, SLOT(setFrame(int)));
    }

}

Pointer::~Pointer()
{
    delete timer_;
}

bool Pointer::setAttribute(QString const& name, QString const& value)
{
    int intVal = 0;
    int attrId = -1;

    if((attrId = getAttribute(name, value, ATTRIBUTES, ATTRIBUTE_COUNT, intVal)) != -1)
    {
        Attribute attr = static_cast<Attribute>(attrId);

        switch(attr)
        {
        case XPOS:
            x_ = intVal;
            break;
        case YPOS:
            y_ = intVal;
            break;
        case ZPOS:
            element_->setZValue(intVal);
            break;
        case VISIBLEWHEN:
            setVisibleWhen(strToVisibleWhen(value));
            break;
        case SRC:
            return loadImage(value);
            break;
        case ZEROANGLE:
            zeroAngle_ = intVal;
            break;
        case FULLANGLE:
            fullAngle_ = intVal;
            break;
        case ZEROSPEED:
            zeroSpeed_ = intVal;
            break;
        case FULLSPEED:
            fullSpeed_ = intVal;
            break;
        case XROTATIONPOINT:
            xRotationPoint_ = intVal;
            break;
        case YROTATIONPOINT:
            yRotationPoint_ = intVal;
            break;
        default:
            qDebug() << "Unknown attribute: " << attr;
            return false;
        }

        return true;
    }
    else
    {
        return false;
    }
}

void Pointer::addToScene(GraphicsScene* scene)
{
    if(!imageSet_)
    {
        return;
    }

    element_->setTransformOriginPoint(xRotationPoint_, yRotationPoint_);
    element_->setX(x_ - xRotationPoint_);
    element_->setY(y_ - yRotationPoint_);

    bool animateVal = animate_;
    animate_ = false;
    update();
    animate_ = animateVal;

    scene->addItem(element_);
}

void Pointer::update()
{
    double speed = Odometer::instance().getLatestFix().kmSpeed;

    if(speed < zeroSpeed_)
    {
        speed = static_cast<double>(zeroSpeed_);
    }
    else if(speed > fullSpeed_)
    {
        speed = static_cast<double>(fullSpeed_);
    }

    double multiplier = static_cast<double>(speed - zeroSpeed_) / (fullSpeed_ - zeroSpeed_);
    double angle = static_cast<double>(zeroAngle_) + ((fullAngle_ - zeroAngle_) * multiplier);

    int rounded = static_cast<int>(round(angle));

    if(rounded == angle_)
    {
        return;
    }

    angle_ = rounded;

    if(!animate_)
    {
        element_->setRotation(rounded);
    }
    else
    {
        targetAngle_ = rounded;
        startAngle_ = element_->rotation();

        if(timer_->state() == QTimeLine::Running)
        {
            timer_->stop();
        }

        timer_->start();
    }
}

void Pointer::setFrame(int frame)
{
    element_->setRotation(startAngle_ + ((static_cast<double>(frame) / ANIMATION_FRAMES) * (targetAngle_ - startAngle_)));
}

bool Pointer::loadImage(QString const& name)
{
    QPixmap pixmap;
    QByteArray data;

    if(!readFile(name, data))
    {
        return false;
    }

    if(!pixmap.loadFromData(data))
    {
        setError("Invalid image file: " + name);
        return false;
    }

    element_->setPixmap(pixmap);
    imageSet_ = true;

    return true;
}

QGraphicsItem* Pointer::getElement() const
{
    return element_;
}

