/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtGui/QGraphicsTextItem>
#include <QtGui/QGraphicsLineItem>
#include <QtGui/QGraphicsPixmapItem>
#include <QtGui/QPixmap>
#include <QtCore/QString>
#include <QtCore/QDebug>
#include <math.h>
#include "detailscreen.h"
#include "odometer.h"
#include "graphicsscene.h"

namespace
{
    QString const FONT = "Tahoma";
    QString const DEFAULT_BACKGROUND_COLOR = "#000";
    QString const DEFAULT_COLOR = "#fff";
    int const FONT_SIZE = 20;
    int const START_HEIGHT = 74;
    int const ITEM_WIDTH = 130;

}

DetailScreen::DetailScreen(QWidget* parent): ThemeScreen(parent)
{
    QString unit = Odometer::getUnit();
    QString speedUnit = Odometer::getSpeedUnit();
    QString meterUnit = Odometer::getMeterUnit();

    getScene()->setBackgroundBrush(QBrush(QColor(DEFAULT_BACKGROUND_COLOR)));

    tripLabel_ = createItem(tr("Trip"));
    trip_ = createItem(roundDouble(Odometer::instance().getTrip()));
    tripUnit_ = createItem(unit);

    totalLabel_ = createItem(tr("Total"));
    total_ = createItem(roundDouble(Odometer::instance().getTotal()));
    totalUnit_ = createItem(unit);

    altitudeLabel_ = createItem(tr("Altitude"));
    altitude_ = createItem(roundDouble(Odometer::instance().getLatestFix().altitude));
    altitudeUnit_ = createItem(meterUnit);

    speedLabel_ = createItem(tr("Speed"));
    speed_ = createItem(roundDouble(0.0));
    speedUnit_ = createItem(speedUnit);

    avgSpeedLabel_ = createItem(tr("Average speed"));
    avgSpeed_ = createItem(roundDouble(Odometer::instance().getAverageSpeed()));
    avgSpeedUnit_ = createItem(speedUnit);

    maxSpeedLabel_ = createItem(tr("Max speed"));
    maxSpeed_ = createItem(roundDouble(Odometer::instance().getMaxSpeed()));
    maxSpeedUnit_ = createItem(speedUnit);

    tripTimeLabel_ = createItem(tr("Trip time"));
    tripTime_ = createItem(formatTime(Odometer::instance().getTripTime()));

    totalTimeLabel_ = createItem(tr("Total time"));
    totalTime_ = createItem(formatTime(Odometer::instance().getTotalTime()));

    line1_ = createLine();
    line2_ = createLine();

    strength_ = new QGraphicsPixmapItem;
    currentStrength_ = getStrength();
    strength_->setPixmap(QPixmap(":/resources/signal_" + QString::number(currentStrength_) + ".png"));
    strength_->setZValue(999);
    getScene()->addItem(strength_);

    connect(&(Odometer::instance()), SIGNAL(timeUpdated()), this, SLOT(updateTime()));
    connect(&(Odometer::instance()), SIGNAL(unitChanged()), this, SLOT(updateUnits()));

    reArrange();
}

void DetailScreen::update()
{
    ThemeScreen::update();

    speed_->setPlainText(roundDouble(Odometer::instance().getLatestFix().speed));
    trip_->setPlainText(roundDouble(Odometer::instance().getTrip()));
    altitude_->setPlainText(roundDouble(Odometer::instance().getLatestFix().altitude));
    total_->setPlainText(roundDouble(Odometer::instance().getTotal()));
    avgSpeed_->setPlainText(roundDouble(Odometer::instance().getAverageSpeed()));
    maxSpeed_->setPlainText(roundDouble(Odometer::instance().getMaxSpeed()));

    int strength = getStrength();

    if(strength != currentStrength_)
    {
        currentStrength_ = strength;
        strength_->setPixmap(QPixmap(":/resources/signal_" + QString::number(strength) + ".png"));
    }
}

void DetailScreen::updateTime()
{
    tripTime_->setPlainText(formatTime(Odometer::instance().getTripTime()));
    totalTime_->setPlainText(formatTime(Odometer::instance().getTotalTime()));
}

void DetailScreen::updateUnits()
{
    QString unit = Odometer::getUnit();
    QString speedUnit = Odometer::getSpeedUnit();
    QString meterUnit = Odometer::getMeterUnit();

    tripUnit_->setPlainText(unit);
    totalUnit_->setPlainText(unit);
    altitudeUnit_->setPlainText(meterUnit);
    speedUnit_->setPlainText(speedUnit);
    avgSpeedUnit_->setPlainText(speedUnit);
    maxSpeedUnit_->setPlainText(speedUnit);

    update();
}

void DetailScreen::reArrange()
{
    ThemeScreen::reArrange();

    int width = getScene()->width();
    int height = getScene()->height();

    int lineHeight = (height - START_HEIGHT) / 8;

    int padding = width / 27;
    int area1 = (width / 3) + ITEM_WIDTH / 2 + 20;
    int area2 = area1 + ITEM_WIDTH;

    tripLabel_->setPos(padding, START_HEIGHT);
    trip_->setPos(area1, START_HEIGHT);
    tripUnit_->setPos(area2, START_HEIGHT);

    totalLabel_->setPos(padding, START_HEIGHT + lineHeight);
    total_->setPos(area1, START_HEIGHT + lineHeight);
    totalUnit_->setPos(area2, START_HEIGHT + lineHeight);

    altitudeLabel_->setPos(padding, START_HEIGHT + 2 * lineHeight);
    altitude_->setPos(area1, START_HEIGHT + 2 * lineHeight);
    altitudeUnit_->setPos(area2, START_HEIGHT + 2 * lineHeight);

    speedLabel_->setPos(padding, START_HEIGHT + 3 * lineHeight);
    speed_->setPos(area1, START_HEIGHT + 3 * lineHeight);
    speedUnit_->setPos(area2, START_HEIGHT + 3 * lineHeight);

    avgSpeedLabel_->setPos(padding, START_HEIGHT + 4 * lineHeight);
    avgSpeed_->setPos(area1, START_HEIGHT + 4 * lineHeight);
    avgSpeedUnit_->setPos(area2, START_HEIGHT + 4 * lineHeight);

    maxSpeedLabel_->setPos(padding, START_HEIGHT + 5 * lineHeight);
    maxSpeed_->setPos(area1, START_HEIGHT + 5 * lineHeight);
    maxSpeedUnit_->setPos(area2, START_HEIGHT + 5 * lineHeight);

    tripTimeLabel_->setPos(padding, START_HEIGHT + 6 * lineHeight);
    tripTime_->setPos(area1, START_HEIGHT + 6 * lineHeight);

    totalTimeLabel_->setPos(padding, START_HEIGHT + 7 * lineHeight);
    totalTime_->setPos(area1, START_HEIGHT + 7 * lineHeight);

    int y1 = START_HEIGHT + 3 * lineHeight - lineHeight / 2 + FONT_SIZE + 2;
    int y2 = START_HEIGHT + 6 * lineHeight - lineHeight / 2 + FONT_SIZE + 2;
    int x = width - padding;
    line1_->setLine(padding + 2, y1, x, y1);
    line2_->setLine(padding + 2, y2, x, y2);

    strength_->setPos(width - padding - 64, height - padding - 41);
}

QGraphicsTextItem* DetailScreen::createItem(QString const& text)
{
    QGraphicsTextItem* item = new QGraphicsTextItem(text);
    QFont font(FONT, FONT_SIZE);
    font.setBold(false);
    item->setFont(font);
    item->setDefaultTextColor(QColor(DEFAULT_COLOR));
    item->setZValue(999);
    getScene()->addItem(item);
    return item;
}

QGraphicsLineItem* DetailScreen::createLine()
{
    QGraphicsLineItem* item = new QGraphicsLineItem;
    QColor pColor(DEFAULT_COLOR);
    QPen p(pColor);
    p.setWidth(1);
    item->setPen(p);
    item->setZValue(999);
    getScene()->addItem(item);
    return item;
}

QString DetailScreen::roundDouble(double number)
{
    QString result;
    result.sprintf("%.2lf", number);
    return result;
}

QString DetailScreen::formatTime(qulonglong time)
{
    qulonglong secs = time % 60;
    qulonglong mins_tmp = time / 60;
    qulonglong mins = mins_tmp % 60;
    qulonglong hours = mins_tmp / 60;

    static char format[] = "%02d";

    return QString().sprintf(format, hours) + ":" +
           QString().sprintf(format, mins) + ":" +
           QString().sprintf(format, secs);
}

int DetailScreen::getStrength()
{
    double strength = round(Odometer::instance().getSignalStrength() / 25.0);
    return static_cast<int>(strength);
}

void DetailScreen::setColor(QString const& color)
{
    QColor c(color);

    tripLabel_->setDefaultTextColor(c);
    trip_->setDefaultTextColor(c);
    tripUnit_->setDefaultTextColor(c);
    totalLabel_->setDefaultTextColor(c);
    total_->setDefaultTextColor(c);
    totalUnit_->setDefaultTextColor(c);
    altitudeLabel_->setDefaultTextColor(c);
    altitude_->setDefaultTextColor(c);
    altitudeUnit_->setDefaultTextColor(c);
    speedLabel_->setDefaultTextColor(c);
    speed_->setDefaultTextColor(c);
    speedUnit_->setDefaultTextColor(c);
    avgSpeedLabel_->setDefaultTextColor(c);
    avgSpeed_->setDefaultTextColor(c);
    avgSpeedUnit_->setDefaultTextColor(c);
    maxSpeedLabel_->setDefaultTextColor(c);
    maxSpeed_->setDefaultTextColor(c);
    maxSpeedUnit_->setDefaultTextColor(c);
    tripTimeLabel_->setDefaultTextColor(c);
    tripTime_->setDefaultTextColor(c);
    totalTimeLabel_->setDefaultTextColor(c);
    totalTime_->setDefaultTextColor(c);

    QPen p(c);
    p.setWidth(1);

    line1_->setPen(p);
    line2_->setPen(p);
}
