#include <SDL.h>
#include <SDL_image.h>
#include "graphics.h"
#include "board.h"
#include "sge_core.h"
#include "gweled.h"

SDL_Surface *screen;

gint gi_gems_pixbuf[7];
gint gi_tiles_bg_pixbuf[2];
gint gi_cursor_pixbuf;
gint gi_small_charset_pixbuf[50];
gint gi_medium_charset_pixbuf[50];
gint gi_big_charset_pixbuf[50];
gint gi_not_pixbuf;
gint gi_background_pixbuf;
gint gi_power_pixbuf;

signed char gpc_font_glyphs[256];

static void gfx_init_glyphs (void);

static SDL_Surface *gfx_load_image(char *filename, int transparent)
{

	SDL_Surface *img, *img2;
//	fprintf(stderr, "Loading: %s\n", filename);

	img = IMG_Load(filename);
	if (!img) {
		fprintf(stderr, "File not found: %s\n", filename);
		return NULL;
	}
	if (transparent && transparent != 2) {
		SDL_SetColorKey(img,  SDL_RLEACCEL,  img->format->colorkey);
		return img;
	} 
    // Create hardware surface
    	img2 = SDL_CreateRGBSurface(SDL_HWSURFACE | (transparent)?SDL_SRCCOLORKEY:0, 
		img->w, img->h, 16, 0xF800, 0x7E0, 0x1F, 0);
	if (!img2) {
		SDL_FreeSurface(img);
		fprintf(stderr, "Error creating surface!\n");
		return NULL;
	}
	
	if (transparent)
		SDL_SetColorKey(img2,  SDL_SRCCOLORKEY | SDL_RLEACCEL, 0xF81F);

	SDL_SetAlpha(img2, 0, 0);
	SDL_BlitSurface(img, NULL, img2, NULL);
	SDL_FreeSurface(img);
	return img2;
}

static int gfx_load_font(char *font, int w, int h, gint *indexes)
{
	int i;
	SDL_Surface *pixbuf;
	
	pixbuf = gfx_load_image (font, 1);
	if (!pixbuf)
		return -1;
	for (i = 0; i < 50; i++) {
		SDL_Rect src;
		SDL_Surface *chr;
		src.w = w;
		src.h = h;
		src.y = 0;
		src.x = i * w;
		
	    	chr = SDL_CreateRGBSurface(SDL_HWSURFACE | SDL_SRCCOLORKEY, 
			w, h, 16, 0xF800, 0x7E0, 0x1F, 0);
		if (!chr) {
			fprintf(stderr, "Error while loading font: %s\n", font);
			return -1;	
		}
		SDL_SetColorKey(chr, SDL_SRCCOLORKEY | SDL_RLEACCEL, pixbuf->format->colorkey);
		SDL_SetAlpha(chr, 0, 0);
		SDL_BlitSurface(pixbuf, &src, chr, NULL);
		indexes[i] = sge_register_pixbuf (chr, -1);
	}
	SDL_FreeSurface(pixbuf);
	return 0;
}

static int gfx_load_images(void)
{
	char fname[255];
	SDL_Surface *pixbuf;

	int i;
	/* handles 0-7 are diamonds */
	pixbuf = gfx_load_image(DATADIR"pixmaps/board.png", 0);
	if (!pixbuf)
		return -1;
	gi_background_pixbuf = sge_register_pixbuf (pixbuf, -1);
	pixbuf = gfx_load_image(DATADIR"pixmaps/power.png", 0);
	if (!pixbuf)
		return -1;
	gi_power_pixbuf = sge_register_pixbuf (pixbuf, -1);
	for (i = 0; i < 7; i++) {
		snprintf(fname, sizeof(fname),DATADIR"pixmaps/diamond%d.bmp", i);
		pixbuf = gfx_load_image(fname, 2);
		if (!pixbuf)
			return -1;
		gi_gems_pixbuf[i] = sge_register_pixbuf(pixbuf, -1);
	}

	pixbuf = gfx_load_image (DATADIR"pixmaps/tile_odd.png", 0);
	if (!pixbuf)
		return -1;
	gi_tiles_bg_pixbuf[0] = sge_register_pixbuf (pixbuf, -1);

	pixbuf = gfx_load_image (DATADIR"pixmaps/tile_even.png", 0);
	if (!pixbuf)
		return -1;
	gi_tiles_bg_pixbuf[1] = sge_register_pixbuf (pixbuf, -1);

	pixbuf = gfx_load_image (DATADIR"pixmaps/cursor.png", 1);
	if (!pixbuf)
		return -1;
	gi_cursor_pixbuf = sge_register_pixbuf (pixbuf, -1);
	gfx_init_glyphs ();
	if (gfx_load_font(DATADIR"pixmaps/font_24_20.png", SMALL_FONT_WIDTH, SMALL_FONT_HEIGHT, gi_small_charset_pixbuf))
		return -1;
	if (gfx_load_font(DATADIR"pixmaps/font_36_30.png", MEDIUM_FONT_WIDTH, MEDIUM_FONT_HEIGHT, gi_medium_charset_pixbuf))
		return -1;
	if (gfx_load_font(DATADIR"pixmaps/font_48_40.png", BIG_FONT_WIDTH, BIG_FONT_HEIGHT, gi_big_charset_pixbuf))
		return -1;
	pixbuf = gfx_load_image (DATADIR"pixmaps/music_off.png", 1);
	if (!pixbuf)
		return -1;
	gi_not_pixbuf = sge_register_pixbuf (pixbuf, -1);
	return 0;
}

static void gfx_draw_character (gint *chars, int x, int y, char character)
{
	int index = (unsigned char)character;
	SDL_Rect dst;
	
	if (gpc_font_glyphs[index] == -1)
		return;
	dst.x = x;
	dst.y = y;
	dst.w = 0;
	dst.h = 0;	
	SDL_BlitSurface(sge_get_pixbuf(chars[gpc_font_glyphs[index]]), NULL, screen, &dst);
}

void gfx_draw_text(const char *txt, int x, int y, int size)
{
	int i;
	int sz = strlen(txt);
	gint *chars;
	int fn_width, fn_height;
	switch (size){
		case SMALL_FONT:
			chars = gi_small_charset_pixbuf;
			fn_width = SMALL_FONT_WIDTH;
			fn_height = SMALL_FONT_HEIGHT;
			break;
		case MEDIUM_FONT:
			chars = gi_medium_charset_pixbuf;
			fn_width = MEDIUM_FONT_WIDTH;
			fn_height = MEDIUM_FONT_HEIGHT;
			break;
		case BIG_FONT:
			chars = gi_medium_charset_pixbuf;
			fn_width = BIG_FONT_WIDTH;
			fn_height = BIG_FONT_HEIGHT;
			break;
		default:
			return;
	}
	gfx_clear(x, y, sz * fn_width, fn_height);
	for (i = 0; i<sz; i++)
		gfx_draw_character(chars, x + i * fn_width, y, txt[i]);
	gfx_update(x, y, sz * fn_width, fn_height);
	return;
}

static void gfx_init_glyphs (void)
{
	int i;

	for (i = 0; i < 127; i++)
		if (g_ascii_isupper (i))
			gpc_font_glyphs[i] = i - 'A';
		else if (g_ascii_isdigit (i))
			gpc_font_glyphs[i] = 29 + i - '0';
		else
			switch (i) {
			case '(':
				gpc_font_glyphs[i] = 26;
				break;
			case ')':
				gpc_font_glyphs[i] = 27;
				break;
			case '-':
				gpc_font_glyphs[i] = 28;
				break;
			case '.':
				gpc_font_glyphs[i] = 39;
				break;
			case ':':
				gpc_font_glyphs[i] = 40;
				break;
			case ',':
				gpc_font_glyphs[i] = 41;
				break;
			case '\'':
				gpc_font_glyphs[i] = 42;
				break;
			case '"':
				gpc_font_glyphs[i] = 43;
				break;
			case '?':
				gpc_font_glyphs[i] = 44;
				break;
			case '!':
				gpc_font_glyphs[i] = 45;
				break;
			case '#':	// "!!!"
				gpc_font_glyphs[i] = 46;
				break;
			case '_':	// "..."
				gpc_font_glyphs[i] = 47;
				break;
			case '*':	// "ang"
				gpc_font_glyphs[i] = 48;
				break;
			default:
				gpc_font_glyphs[i] = -1;
				break;
			}
}



void gfx_draw(SDL_Surface *pixbuf, int x, int y)
{
	SDL_Rect dest;
	dest.x = x;
	dest.y = y; 
	dest.w = pixbuf->w; 
	dest.h = pixbuf->h;
//	gfx_clear(x, y, dest.w, dest.h);
	SDL_BlitSurface(pixbuf, NULL, screen, &dest);
//	gfx_update(x, y, dest.w, dest.h);
}

void gfx_clear(int x, int y, int w, int h)
{
	SDL_Rect dest;
	dest.x = x;
	dest.y = y; 
	dest.w = w; 
	dest.h = h;
	SDL_BlitSurface(sge_get_pixbuf(gi_background_pixbuf), &dest, screen, &dest);
//	SDL_FillRect(screen, &dest, SDL_MapRGB(screen->format, 0, 0, 0));
}

int gfx_init(void)
{
	// Initialize SDL
	if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_TIMER | SDL_INIT_AUDIO) < 0) {
		fprintf(stderr, "Couldn't initialize SDL: %s\n", SDL_GetError());
		return -1;
  	}
#ifdef MAEMO	
	screen = SDL_SetVideoMode(800, 480, 16, SDL_DOUBLEBUF | SDL_HWSURFACE | SDL_FULLSCREEN);
#else
	screen = SDL_SetVideoMode(800, 480, 16, SDL_DOUBLEBUF | SDL_HWSURFACE);
//	SDL_WM_SetCaption("SDL Gweled", NULL); //HACK to fix focus bug--Flandry
#endif	
	if (screen == NULL) {
		fprintf(stderr, "Unable to set 800x480 video: %s\n", SDL_GetError());
		return -1;
	}
#ifdef MAEMO
	SDL_ShowCursor(SDL_DISABLE);
#endif	
	sge_init();
	sge_set_drawing_area(screen, BOARD_X, BOARD_Y, TILE_WIDTH * BOARD_WIDTH, 
					TILE_HEIGHT * BOARD_HEIGHT);
	if (gfx_load_images())
		return -1;
	gfx_clear(0, 0, 800, 480);
	return 0;
}
void gfx_update(int x, int y, int w, int h) {
	SDL_UpdateRect(screen, x, y, w, h);
}

void gfx_done(void)
{
//	destroy_all_alignments ();
	sge_destroy_all_objects ();
	sge_destroy();
	SDL_Quit();
}
