#include <QtWebKit>
#include <QApplication>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QNetworkCookie>
#include <QBuffer>
#include <QTimer>
#include <QDateTime>
#include <QDebug>

#include <qjson/parser.h>

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>

#include "googlereader.h"

void Feed::updateSubscription(Feed *feed) {
	title = feed->title;
	sortid = feed->sortid;
	firstitemmsec = feed->firstitemmsec;
	cat_id = feed->cat_id;
	cat_label = feed->cat_label;
	subscription_updated = true;
}

void Feed::fetch(bool cont) {
	QNetworkRequest request;
	QByteArray ba = "http://www.google.com/reader/api/0/stream/contents/";
	ba.append(QUrl::toPercentEncoding(id));
	QUrl url = QUrl::fromEncoded(ba);

	if(continuation != "" && cont)
		url.addEncodedQueryItem("c", continuation.toUtf8());

	if(!cont && updated) {
		/* Add 1 to the timestamp, otherwise we get the latest item
		 * again. Also the order has to be reversed for this to work. */
		url.addEncodedQueryItem("ot", QByteArray::number(updated + 1));
		url.addEncodedQueryItem("r", "o");
	}

	request.setUrl(url);
	reply = reader->getManager()->get(request);
	connect(reply, SIGNAL(finished()), SLOT(fetchFinished()));
}

void Feed::fetchFinished() {
	if (reply->error()) {
		qDebug() << "Download of" << reply->url() << "failed:" << qPrintable(reply->errorString());
		return;
	}

	QJson::Parser parser;
	bool ok;
	QVariantMap result = parser.parse(reply->readAll(), &ok).toMap();
	QString continuation;

	continuation = result["continuation"].toString();
	updated = result["updated"].toUInt();

	foreach(QVariant l, result["items"].toList()) {
		QVariantMap e = l.toMap();
		Entry *entry = new Entry();;
		QString content, summary;

		entry->id = e["id"].toString();
		entry->published = QDateTime::fromTime_t(e["published"].toUInt());
		entry->author = e["author"].toString();
		entry->source = (e["origin"].toMap())["streamId"].toString();
		entry->link = (e["alternate"].toMap())["href"].toString();

		content = (e["content"].toMap())["content"].toString();
		summary = (e["summary"].toMap())["content"].toString();
		if(content != "") entry->content = content; else entry->content = summary;

		if(e["isReadStateLocked"].toBool())
			entry->flags |= ENTRY_FLAG_LOCKED | ENTRY_FLAG_READ;

		QWebPage p;
		p.mainFrame()->setHtml(e["title"].toString());
		entry->title = p.mainFrame()->toPlainText();

		foreach(QVariant c, e["categories"].toList()) {
			QString cat = c.toString();
			if(cat.endsWith("/state/com.google/read"))
				entry->flags |= ENTRY_FLAG_READ;
			else if(cat.endsWith("/state/com.google/starred"))
				entry->flags |= ENTRY_FLAG_STARRED;
			else if(cat.endsWith("/state/com.google/broadcast"))
				entry->flags |= ENTRY_FLAG_SHARED;
		}

		entry->feed = this;
		addEntry(entry);
	}

	lastUpdated = QDateTime::currentDateTime();

	emit updateFeedComplete();

	reply->deleteLater();
}

GoogleReader::GoogleReader() {
	connect(&manager, SIGNAL(finished(QNetworkReply*)),
		SLOT(downloadFinished(QNetworkReply*)));

	SID = NULL;
	updateSubscriptionsPending = false;
	updateUnreadPending = false;
	SIDPending = false;

	login_url.setUrl("https://www.google.com/accounts/ClientLogin");
	subscriptions_url.setUrl("http://www.google.com/reader/api/0/subscription/list?output=json");
	unread_url.setUrl("http://www.google.com/reader/api/0/unread-count?output=json");
	edittag_url.setUrl("http://www.google.com/reader/api/0/edit-tag?client=-");
	token_url.setUrl("http://www.google.com/reader/api/0/token");
	markallread_url.setUrl("http://www.google.com/reader/api/0/mark-all-as-read?client=-");

	/* Add the virtual 'Starred items' feed */
	Feed *feed = new Feed(this);
	feed->id = "user/-/state/com.google/starred";
	feed->title = "Starred items";
	feed->special = 2;
	feeds.insert(feed->id, feed);
	connect(feed, SIGNAL(allReadChanged()), SIGNAL(allReadChanged()));

	/* Add the virtual 'Shared items' feed */
	feed = new Feed(this);
	feed->id = "user/-/state/com.google/broadcast";
	feed->title = "Shared items";
	feed->special = 1;
	feeds.insert(feed->id, feed);
	connect(feed, SIGNAL(allReadChanged()), SIGNAL(allReadChanged()));
}

void GoogleReader::downloadFinished(QNetworkReply *reply) {
	QUrl url = reply->url();

	/* TODO: Instead of comparing against the url, use the signal from the
	 * QNetworkReply... */

	if (reply->error()) {
		qDebug() << "Download of" << url << "failed:" << qPrintable(reply->errorString());
		if(url == login_url) {
			SIDPending = false;
			emit loginFailed("Incorrect username or password");
		}
		else if(url == edittag_url)
			getToken();
		return;
	}
	else if(url == login_url) {
		QByteArray data = reply->readAll();
		data.remove(0, data.indexOf("SID=", 0) + 4);
		data.remove(data.indexOf("\n", 0), 1024);
		SID = strdup(data.data());

		qDebug() << "SID:" << SID;

		manager.cookieJar()->setCookiesFromUrl(
			QList<QNetworkCookie>() << QNetworkCookie("SID", SID),
			QUrl("http://www.google.com"));

		SIDPending = false;

		getToken();

		/* TODO: Replace this with a proper state machine */
		if(updateSubscriptionsPending) {
			updateSubscriptionsPending = false;
			updateSubscriptions();
		}
	}
	else if(url == token_url) {
		token = reply->readAll();
		qDebug() << "token:" << token;
	}
	else if(url == subscriptions_url) {
		parseSubscriptions(reply->readAll());

		/* TODO: Replace this with a proper state machine */
		if(updateUnreadPending) {
			updateUnreadPending = false;
			updateUnread();
		}
	}
	else if(url == unread_url) {
		parseUnread(reply->readAll());
	}
	else if(url == edittag_url) {
		QByteArray data = reply->readAll();
		//qDebug() << "Result:" << data;
	}
	else if(url == markallread_url) {
		QByteArray data = reply->readAll();
		//qDebug() << "Result:" << data;
	}

	reply->deleteLater();
}

void GoogleReader::parseSubscriptions(QByteArray data) {
	QJson::Parser parser;
	bool ok;
	QVariantMap result = parser.parse(data, &ok).toMap();

	/* Clear the subscription updated flag */
	QHash<QString, Feed *>::iterator i;
	for(i = feeds.begin(); i != feeds.end(); ++i)
		i.value()->subscription_updated = false;

	foreach(QVariant l, result["subscriptions"].toList()) {
		QVariantMap subscription = l.toMap();
		Feed *feed = new Feed(this);
		Feed *existing_feed;

		feed->id = subscription["id"].toString();
		feed->title = subscription["title"].toString();
		feed->sortid = subscription["sortid"].toString();
		feed->firstitemmsec = subscription["firstitemmsec"].toString();

		foreach(QVariant c, subscription["categories"].toList()) {
			QVariantMap cat = c.toMap();
			feed->cat_id = cat["id"].toString();
			feed->cat_label = cat["label"].toString();
		}

		existing_feed = feeds.value(feed->id);
		if(existing_feed) {
			existing_feed->updateSubscription(feed);
			delete(feed);
			feed = existing_feed;

		}
		else {
			feed->subscription_updated = true;
			feeds.insert(feed->id, feed);
			connect(feed, SIGNAL(allReadChanged()), SIGNAL(allReadChanged()));
		}
	}

	/* Delete feeds no longer subscribed to  */
	for(i = feeds.begin(); i != feeds.end(); ++i) {
		if(i.value()->subscription_updated == false && i.value()->special == 0) {
			printf("DELETED: %s\n", i.value()->title.toLatin1().data());
			i = feeds.erase(i);
		}
	}

	lastUpdated = QDateTime::currentDateTime();
	emit updateSubscriptionsComplete();
}

void GoogleReader::parseUnread(QByteArray data) {
	QJson::Parser parser;
	bool ok;
	QVariantMap result = parser.parse(data, &ok).toMap();

	foreach(QVariant l, result["unreadcounts"].toList()) {
		QVariantMap unread = l.toMap();
		QString id = unread["id"].toString();
		int count = unread["count"].toInt();
		ulong newestitem = unread["newestitem"].toUInt();

		Feed *f = feeds.value(id);
		if(f) {
			f->unread = count;
			f->newestitem = newestitem;

			/* Not a good idea if it doesn't happen sequentially. */
			/* Pre-fetch feeds with unread items */
			/* f->fetch(false); */
		}
	}

	lastUpdated = QDateTime::currentDateTime();
	emit updateUnreadComplete();
}

void GoogleReader::getSID() {

	if(SIDPending)
		return;

	SIDPending = true;

	QNetworkRequest request;
	request.setUrl(login_url);

	buffer.open(QBuffer::ReadWrite | QBuffer::Truncate);
	buffer.write("Email=");
	buffer.write(QUrl::toPercentEncoding(login));
	buffer.write("&Passwd=");
	buffer.write(QUrl::toPercentEncoding(passwd));
	buffer.write("&service=reader&source=grms&continue=http://www.google.com");

	//buffer.seek(0);
	//qDebug() << buffer.readAll();

	buffer.seek(0);
	manager.post(request, &buffer);
}

void GoogleReader::getToken() {
	QNetworkRequest request;
	request.setUrl(token_url);
	manager.get(request);
}

void GoogleReader::updateSubscriptions() {
	QNetworkRequest request;

	if(updateSubscriptionsPending)
		return;

	if(!SID) {
		updateSubscriptionsPending = true;
		getSID();
		return;
	}

	request.setUrl(subscriptions_url);
	manager.get(request);
}

void GoogleReader::updateUnread() {
	QNetworkRequest request;

	if(updateUnreadPending)
		return;

	if(!SID) {
		updateUnreadPending = true;
		getSID();
		return;
	}

	request.setUrl(unread_url);
	manager.get(request);
}

static bool compareFeedItems(const Feed *f1, const Feed *f2) {
	if(f1->special || f2->special)
		return f1->special > f2->special;

	if(f1->cat_label == f2->cat_label)
		return f1->title.toLower() < f2->title.toLower();

	if(f1->cat_label.length() == 0)
		return false;

	if(f2->cat_label.length() == 0)
		return true;

	return f1->cat_label.toLower() < f2->cat_label.toLower();
}

QList<Feed *> GoogleReader::getFeeds() {
	QList<Feed *> list = feeds.values();
	qSort(list.begin(), list.end(), compareFeedItems);
	return list;
}

void Feed::addEntry(Entry *entry) {
	entries.insert(entry->id, entry);
}

void Feed::delEntry(Entry *entry) {
	entries.remove(entry->id);
}

void Feed::updateUnread(int i) {
	bool allRead = (unread == 0);

	unread += i;
	if(unread <= 0) unread = 0;

	if(allRead != (unread == 0))
		emit allReadChanged();
}

void Feed::markRead() {
	if(unread > 0) {
		/* Mark all the remaining items read */

		QNetworkRequest request;
		request.setUrl(reader->markallread_url);

		buffer.open(QBuffer::ReadWrite | QBuffer::Truncate);
		buffer.write("s=");
		buffer.write(QUrl::toPercentEncoding(id));
		//buffer.write("&ts=");
		//buffer.write(QByteArray::number(oldest));
		buffer.write("&T=");
		buffer.write(QUrl::toPercentEncoding(reader->token));

		//buffer.seek(0);
		//qDebug() << buffer.readAll();

		buffer.seek(0);
		reader->manager.post(request, &buffer);

		unread = 0;

		/* Go over all the entries and mark them read */
		QHash<QString, Entry *>::iterator i;
		for(i = entries.begin(); i != entries.end(); ++i)
			i.value()->flags |= ENTRY_FLAG_READ | ENTRY_FLAG_LOCKED;
	}

	emit allReadChanged();
}

static bool compareEntryItems(const Entry *e1, const Entry *e2) {
	return e1->published > e2->published;
}

QList<Entry *> Feed::getEntries() {
	QList<Entry *> list = entries.values();
	qSort(list.begin(), list.end(), compareEntryItems);
	return list;
}

/* TODO: Remove the duplicate code in changing stated */

void Entry::markRead(bool mark_read) {
	/* Check if the read flag differs from the requested state */
	if(((flags & ENTRY_FLAG_READ) != 0) == mark_read)
		return;

	/* Cannot mark an item unread if it's locked */
	if((flags & ENTRY_FLAG_LOCKED) && !mark_read)
		return;

	QNetworkRequest request;
	request.setUrl(feed->reader->edittag_url);

	postread.open(QBuffer::ReadWrite | QBuffer::Truncate);
	postread.write("i=");
	postread.write(QUrl::toPercentEncoding(id));
	if(mark_read)
		postread.write("&a=");
	else
		postread.write("&r=");
	postread.write(QUrl::toPercentEncoding("user/-/state/com.google/read"));
	postread.write("&ac=edit-tags&T=");
	postread.write(QUrl::toPercentEncoding(feed->reader->token));
	postread.seek(0);
	feed->reader->manager.post(request, &postread);

	feed->updateUnread(mark_read ? -1 : 1);

	if(mark_read)
		flags |= ENTRY_FLAG_READ;
	else
		flags &= ~ENTRY_FLAG_READ;
}

void Entry::markStar(bool mark_star) {
	/* Check if the starred flag differs from the requested state */
	if(((flags & ENTRY_FLAG_STARRED) != 0) == mark_star)
		return;

	QNetworkRequest request;
	request.setUrl(feed->reader->edittag_url);

	poststar.open(QBuffer::ReadWrite | QBuffer::Truncate);
	poststar.write("i=");
	poststar.write(QUrl::toPercentEncoding(id));
	if(mark_star)
		poststar.write("&a=");
	else
		poststar.write("&r=");
	poststar.write(QUrl::toPercentEncoding("user/-/state/com.google/starred"));
	poststar.write("&ac=edit-tags&T=");
	poststar.write(QUrl::toPercentEncoding(feed->reader->token));
	poststar.seek(0);
	feed->reader->manager.post(request, &poststar);

	Feed *starred = feed->reader->feeds.value("user/-/state/com.google/starred");

	if(mark_star) {
		starred->addEntry(this);
		flags |= ENTRY_FLAG_STARRED;
	}
	else {
		starred->delEntry(this);
		flags &= ~ENTRY_FLAG_STARRED;
	}
}

void Entry::markShared(bool mark_shared) {
	/* Check if the shared flag differs from the requested state */
	if(((flags & ENTRY_FLAG_SHARED) != 0) == mark_shared)
		return;

	QNetworkRequest request;
	request.setUrl(feed->reader->edittag_url);

	postshared.open(QBuffer::ReadWrite | QBuffer::Truncate);
	postshared.write("i=");
	postshared.write(QUrl::toPercentEncoding(id));
	if(mark_shared)
		postshared.write("&a=");
	else
		postshared.write("&r=");
	postshared.write(QUrl::toPercentEncoding("user/-/state/com.google/broadcast"));
	postshared.write("&ac=edit-tags&T=");
	postshared.write(QUrl::toPercentEncoding(feed->reader->token));
	postshared.seek(0);
	feed->reader->manager.post(request, &postshared);

	Feed *shared = feed->reader->feeds.value("user/-/state/com.google/broadcast");

	if(mark_shared) {
		shared->addEntry(this);
		flags |= ENTRY_FLAG_SHARED;
	}
	else {
		shared->delEntry(this);
		flags &= ~ENTRY_FLAG_SHARED;
	}
}
