/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"
#include "custom_rating_renderer.h"

/* http://scentric.net/tutorial/sec-custom-cell-renderers.html */
/* https://stage.maemo.org/svn/maemo/projects/haf/trunk/gtk+/gtk/gtkcellrendererpixbuf.c */
/* https://stage.maemo.org/svn/maemo/projects/haf/trunk/gtk+/gtk/gtkcellrenderertext.c */

/* This is based mainly on GtkCellRendererProgress
 *  in GAIM, written and (c) 2002 by Sean Egan
 *  (Licensed under the GPL), which in turn is
 *  based on Gtk's GtkCellRenderer[Text|Toggle|Pixbuf]
 *  implementation by Jonathan Blandford */

/* Some boring function declarations: GObject type system stuff */

static void     custom_cell_renderer_rating_init       (CustomCellRendererRating      *cellrating);
static void     custom_cell_renderer_rating_class_init (CustomCellRendererRatingClass *klass);
static void     custom_cell_renderer_rating_get_property  (GObject                    *object,
                                                             guint                       param_id,
                                                             GValue                     *value,
                                                             GParamSpec                 *pspec);
static void     custom_cell_renderer_rating_set_property  (GObject                    *object,
                                                             guint                       param_id,
                                                             const GValue               *value,
                                                             GParamSpec                 *pspec);
static void     custom_cell_renderer_rating_finalize (GObject *gobject);


/* These functions are the heart of our custom cell renderer: */

static void     custom_cell_renderer_rating_get_size   (GtkCellRenderer            *cell,
                                                          GtkWidget                  *widget,
                                                          GdkRectangle               *cell_area,
                                                          gint                       *x_offset,
                                                          gint                       *y_offset,
                                                          gint                       *width,
                                                          gint                       *height);

static void     custom_cell_renderer_rating_render     (GtkCellRenderer            *cell,
                                                          GdkWindow                  *window,
                                                          GtkWidget                  *widget,
                                                          GdkRectangle               *background_area,
                                                          GdkRectangle               *cell_area,
                                                          GdkRectangle               *expose_area,
                                                          guint                       flags);


enum { PROP_RATING = 1, };

static gpointer parent_class;

GType custom_cell_renderer_rating_get_type (void) {
  static GType cell_rating_type = 0;

  if (cell_rating_type == 0)
  {
    static const GTypeInfo cell_rating_info =
    {
      sizeof (CustomCellRendererRatingClass),
      NULL,                                                     /* base_init */
      NULL,                                                     /* base_finalize */
      (GClassInitFunc) custom_cell_renderer_rating_class_init,
      NULL,                                                     /* class_finalize */
      NULL,                                                     /* class_data */
      sizeof (CustomCellRendererRating),
      0,                                                        /* n_preallocs */
      (GInstanceInitFunc) custom_cell_renderer_rating_init,
    };

    /* Derive from GtkCellRenderer */
    cell_rating_type = g_type_register_static (GTK_TYPE_CELL_RENDERER,
                                                 "CustomCellRendererRating",
                                                  &cell_rating_info,
                                                  0);
  }
  return cell_rating_type;
}


/***************************************************************************
 *
 *  custom_cell_renderer_rating_init: set some default properties of the
 *                                      parent (GtkCellRenderer).
 *
 ***************************************************************************/

static void
custom_cell_renderer_rating_init (CustomCellRendererRating *cellrendererrating)
{
  GTK_CELL_RENDERER(cellrendererrating)->mode = GTK_CELL_RENDERER_MODE_INERT;
  GTK_CELL_RENDERER(cellrendererrating)->xpad = 0;
  GTK_CELL_RENDERER(cellrendererrating)->ypad = 0;
}


/***************************************************************************
 *
 *  custom_cell_renderer_rating_class_init:
 *
 *  set up our own get_property and set_property functions, and
 *  override the parent's functions that we need to implement.
 *  And make our new "size" property known to the type system.
 *  If you want cells that can be activated on their own (ie. not
 *  just the whole row selected) or cells that are editable, you
 *  will need to override 'activate' and 'start_editing' as well.
 *
 ***************************************************************************/

static void
custom_cell_renderer_rating_class_init (CustomCellRendererRatingClass *klass) {
  GtkCellRendererClass *cell_class   = GTK_CELL_RENDERER_CLASS(klass);
  GObjectClass         *object_class = G_OBJECT_CLASS(klass);

  parent_class           = g_type_class_peek_parent (klass);
  object_class->finalize = custom_cell_renderer_rating_finalize;

  /* Hook up functions to set and get our
   *   custom cell renderer properties */
  object_class->get_property = custom_cell_renderer_rating_get_property;
  object_class->set_property = custom_cell_renderer_rating_set_property;

  /* Override the two crucial functions that are the heart
   *   of a cell renderer in the parent class */
  cell_class->get_size = custom_cell_renderer_rating_get_size;
  cell_class->render   = custom_cell_renderer_rating_render;

  /* Install our very own properties */
  g_object_class_install_property (object_class, PROP_RATING,
    g_param_spec_int ("rating", "Rating", "Cache rating", 0, 1600, 0,
		      G_PARAM_READWRITE));
}


/***************************************************************************
 *
 *  custom_cell_renderer_rating_finalize: free any resources here
 *
 ***************************************************************************/

static void custom_cell_renderer_rating_finalize (GObject *object) {
  (* G_OBJECT_CLASS (parent_class)->finalize) (object);
}


/***************************************************************************
 *
 *  custom_cell_renderer_rating_get_property: as it says
 *
 ***************************************************************************/

static void
custom_cell_renderer_rating_get_property (GObject    *object,
                                            guint       param_id,
                                            GValue     *value,
                                            GParamSpec *psec) {
  CustomCellRendererRating  *cellrating = CUSTOM_CELL_RENDERER_RATING(object);

  switch (param_id) {
    case PROP_RATING:
      g_value_set_int(value, 100 * cellrating->difficulty + cellrating->terrain);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, psec);
      break;
  }
}


/***************************************************************************
 *
 *  custom_cell_renderer_rating_set_property: as it says
 *
 ***************************************************************************/

static void
custom_cell_renderer_rating_set_property (GObject      *object,
					  guint         param_id,
					  const GValue *value,
					  GParamSpec   *pspec) {
  CustomCellRendererRating *cellrating = CUSTOM_CELL_RENDERER_RATING (object);

  switch (param_id) {
    case PROP_RATING:
      {
	int i =  g_value_get_int(value);
	cellrating->difficulty = i / 100;
	cellrating->terrain = i % 100;
      }
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, param_id, pspec);
      break;
  }
}

/***************************************************************************
 *
 *  custom_cell_renderer_rating_new: return a new cell renderer instance
 *
 ***************************************************************************/

GtkCellRenderer *custom_cell_renderer_rating_new (void) {
  return g_object_new(CUSTOM_TYPE_CELL_RENDERER_RATING, NULL);
}

static PangoLayout *new_layout(GtkWidget *widget, const char *str) {
#ifndef USE_MAEMO
  return gtk_widget_create_pango_layout(widget, str);
#else
  char tmp[48];
  snprintf(tmp, sizeof(tmp), "<span size=\"xx-small\">%s</span>", str);
  PangoLayout *layout = gtk_widget_create_pango_layout(widget, NULL);
  pango_layout_set_markup(layout, tmp, strlen(tmp));
  return layout;
#endif
}

static void
custom_cell_renderer_rating_get_size (GtkCellRenderer *cell,
	      GtkWidget *widget, GdkRectangle *cell_area, gint *x_offset, 
	      gint *y_offset, gint *width, gint *height) {
  GdkPixbuf *pixbuf = icon_get(ICON_STARS, 0);  // all icons have the same size
  /* do the text */
  PangoLayout *layout = new_layout(widget, "D:");
  PangoRectangle rect;
  pango_layout_get_pixel_extents(layout, NULL, &rect);
  g_object_unref (layout);

  /* width is text width + icon width */
  gint calc_width  = (gint)gdk_pixbuf_get_width(pixbuf) + rect.width;
  /* height is 2*MAX(icon-height, text-height) */
  gint calc_height  = (gint) 2 * MAX(gdk_pixbuf_get_height(pixbuf), rect.height);

  if (x_offset) *x_offset = 0;
  if (y_offset) *y_offset = 0;

  if (cell_area) {
    if (x_offset) 
      *x_offset = (cell_area->width - calc_width)/2;

    if (y_offset)
      *y_offset = (cell_area->height - calc_height)/2;
  }

  if (width)  *width = calc_width;
  if (height) *height = calc_height;
}


/***************************************************************************
 *
 *  custom_cell_renderer_rating_render: crucial - do the rendering.
 *
 ***************************************************************************/

static void
custom_cell_renderer_rating_render (GtkCellRenderer *cell,
                                      GdkWindow       *window,
                                      GtkWidget       *widget,
                                      GdkRectangle    *background_area,
                                      GdkRectangle    *cell_area,
                                      GdkRectangle    *expose_area,
                                      guint            flags)
{
  CustomCellRendererRating *cellrating = CUSTOM_CELL_RENDERER_RATING (cell);
  GdkPixbuf *dpixbuf = icon_get(ICON_STARS, cellrating->difficulty);
  GdkPixbuf *tpixbuf = icon_get(ICON_STARS, cellrating->terrain);
  GdkRectangle all_rect;
  GdkRectangle draw_rect;
 
  gint pix_width  = gdk_pixbuf_get_width(dpixbuf);
  gint pix_height  = gdk_pixbuf_get_height(dpixbuf);

  if ((!dpixbuf) || (!tpixbuf)) return;
  
  custom_cell_renderer_rating_get_size (cell, widget, cell_area,
				     &all_rect.x, &all_rect.y,
				     &all_rect.width, &all_rect.height);

  all_rect.x += cell_area->x;
  all_rect.y += cell_area->y;

  if (!gdk_rectangle_intersect (cell_area, &all_rect, &draw_rect) ||
      !gdk_rectangle_intersect (expose_area, &draw_rect, &draw_rect))
    return;

  /* do the text */
  PangoLayout *tlayout = new_layout(widget, "T:");
  PangoLayout *dlayout = new_layout(widget, "D:");
  PangoRectangle text_rect;
  pango_layout_get_pixel_extents(dlayout, NULL, &text_rect);

  int xoff = (all_rect.width - (text_rect.width + pix_width))/2;
  if(xoff < 0) xoff = 0;
  int yoff = all_rect.height/2 - MAX(text_rect.height, pix_height);

  /* if text is heigher, icon needs additional vertical offset and vice versa */
  int tyoff = 0, iyoff = 0;

  if(text_rect.height > pix_height) iyoff = (text_rect.height - pix_height)/2;
  else                              tyoff = (pix_height - text_rect.height)/2;

  gtk_paint_layout(widget->style, window, GTK_STATE_NORMAL,
	   TRUE, expose_area, widget, "cellrendererrating",
	   all_rect.x + xoff, all_rect.y + yoff + tyoff, dlayout);

  gtk_paint_layout(widget->style, window, GTK_STATE_NORMAL,
	   TRUE, expose_area, widget, "cellrendererrating",
	   all_rect.x + xoff, all_rect.y + MAX(text_rect.height, pix_height) + yoff + tyoff, tlayout);

  g_object_unref(dlayout);
  g_object_unref(tlayout);

  /* draw the bitmap */
  cairo_t *cr = gdk_cairo_create(window);
  gdk_cairo_set_source_pixbuf(cr, dpixbuf, 
	      all_rect.x + xoff + text_rect.width,
	      all_rect.y + yoff + iyoff);
  gdk_cairo_rectangle(cr, &draw_rect);
  cairo_fill(cr);

  gdk_cairo_set_source_pixbuf(cr, tpixbuf, 
	      all_rect.x + xoff + text_rect.width,
	      all_rect.y + MAX(text_rect.height, pix_height) + yoff + iyoff);
  gdk_cairo_rectangle(cr, &draw_rect);
  cairo_fill(cr);

  cairo_destroy(cr);
}
