/* appbar.h: statusbar/progress widget
 *
 * Based on gnome-appbar by Havoc Pennington
 * Based on MozStatusbar widget, by Chris Toshok
 * In turn based on GtkStatusbar widget, from Gtk+,
 * Copyright (C) 1995-1998 Peter Mattis, Spencer Kimball and Josh MacDonald
 * GtkStatusbar Copyright (C) 1998 Shawn T. Amundson
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Cambridge, MA 02139, USA.
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <gdk/gdkkeysyms.h>
/*This is just used to logically separate code out.*/
#define MAEMO (1)
/*The AppBar takes up a lot of screen real-estate, and in the default theme
   looks rather ugly.
  We can get all that back by using the banner interface instead for transient
   notifications.*/
#if defined(MAEMO)
# include <hildon/hildon-banner.h>
#endif

#include <string.h>
#include "appbar.h"

struct AppBarPrivate{
  /*Private; there's no guarantee on the type of these in the future.
    Statusbar could be a label, entry, GtkStatusbar, or something else;
     progress could be a label or progress bar; it's all up in the air for
     now.*/
  /*There is no reason for these not to be properties.*/
  GtkWidget *progress;
#if defined(MAEMO)
  GtkWidget *prog_banner;
#else
  GtkWidget *status;
#endif
  /*Keep it simple; no contexts.
    if(status_stack)display_top_of_stack;
    else if(default_status)display_default;
    else display_nothing;      */
  /*Also private by the way.*/
  GSList    *status_stack;
  gchar     *default_status;
  /*These are construct only currently.
    There is no reason this couldn't be changed.*/
  guint      has_progress:1;
  guint      has_status:1;
};

enum{
  PROP_0,
  PROP_HAS_PROGRESS,
  PROP_HAS_STATUS,
};

static void appbar_class_init(AppBarClass *_class);
static void appbar_instance_init(AppBar *_this);

static GtkHBoxClass *parent_class;

GType appbar_get_type(void){
  static GType appbar_type;
  if(!appbar_type){
    static const GTypeInfo appbar_info={
      sizeof(AppBarClass),
      NULL,
      NULL,
      (GClassInitFunc)appbar_class_init,
      NULL,
      NULL,
      sizeof(AppBar),
      0,
      (GInstanceInitFunc)appbar_instance_init
    };
    appbar_type=g_type_register_static(GTK_TYPE_HBOX,"appbar",&appbar_info,0);
  }
  return appbar_type;
}

static GSList *stringstack_push(GSList *_stringstack,const gchar *_s){
  return g_slist_prepend(_stringstack,g_strdup(_s));
}

static GSList *stringstack_pop(GSList *_stringstack){
  /*Could be optimized.*/
  if(_stringstack!=NULL){
    g_free(_stringstack->data);
    return g_slist_remove(_stringstack,_stringstack->data);
  }
  else return NULL;
}

static const gchar *stringstack_top(GSList *_stringstack){
  if(_stringstack)return _stringstack->data;
  else return NULL;
}

static void stringstack_free(GSList *_stringstack){
  GSList *tmp;
  tmp=_stringstack;
  while(tmp){
    g_free(tmp->data);
    tmp=g_slist_next(tmp);
  }
  g_slist_free(_stringstack);
}

static void appbar_instance_init(AppBar *_this){
  _this->priv=g_new0(AppBarPrivate,1);
  _this->priv->default_status=NULL;
  _this->priv->status_stack=NULL;
}


/** appbar_new
 * @has_progress: %TRUE if appbar needs progress bar widget, %FALSE if not.
 * @has_status: %TRUE if appbar needs status bar widget, %FALSE if not.
 * Description:
 * Create a new application status bar.
 * If @has_progress is %TRUE, a small progress bar widget will be created, and
 *  placed on the left side of the appbar.
 * If @has_status is %TRUE, a status bar, possibly an editable one, is created.
 * Returns: Pointer to new #AppBar widget.
 **/
GtkWidget* appbar_new(gboolean _has_progress,gboolean _has_status){
  return GTK_WIDGET(g_object_new(GJITEN_TYPE_APPBAR,
   "has_progress",_has_progress,"has_status",_has_status,NULL));
}

/** appbar_refresh
 * @_this: An #AppBar instance.
 * Description:
 * Refresh the status message bar by redrawing the item on the top of the
 *  stack, or the default value if the stack is empty.
 * Useful to force the message from a previous call to appbar_set_status() to
 *  disappear.
 **/
void appbar_refresh(AppBar *_this){
  g_return_if_fail(_this!=NULL);
  g_return_if_fail(GJITEN_IS_APPBAR(_this));
  if(_this->priv->status_stack){
    appbar_set_status(_this,stringstack_top(_this->priv->status_stack));
  }
  else if(_this->priv->default_status){
    appbar_set_status(_this,_this->priv->default_status);
  }
  else appbar_set_status(_this,"");
}

/** appbar_set_status
 * @_this: An #AppBar instance.
 * @_status: Text to which status label will be set.
 * Description:
 * Sets the status label without changing widget state; next call to
 *  appbar_set_status(), appbar_push(), appbar_pop(), appbar_clear_stack(),
 *  or appbar_set_default() will destroy this permanently.
 **/
void appbar_set_status(AppBar *_this,const gchar *_status){
  g_return_if_fail(_this!=NULL);
  g_return_if_fail(_status!=NULL);
  g_return_if_fail(GJITEN_IS_APPBAR(_this));
#if defined(MAEMO)
  if(_status!=NULL&&_status[0]!='\0'){
    hildon_banner_show_information(GTK_WIDGET(_this),"jiten",_status);
  }
#else
  gtk_label_set_text(GTK_LABEL(_this->priv->status),_status);
#endif
}

/** appbar_set_default
 * @_this: An #AppBar instance.
 * @_default_status: Text for status label.
 * Description:
 * What to show when showing nothing else is on the stack; defaults to the
 *  empty string.
 **/
void appbar_set_default(AppBar *_this,const gchar *_default_status){
  g_return_if_fail(_this!=NULL);
  g_return_if_fail(_default_status!=NULL);
  g_return_if_fail(GJITEN_IS_APPBAR(_this));
  g_free(_this->priv->default_status);
  _this->priv->default_status=g_strdup(_default_status);
#if defined(MAEMO)
  /*Prevent spurious messages from popping up.*/
  if(!_this->priv->status_stack)appbar_refresh(_this);
#else
  appbar_refresh(_this);
#endif
}

/** appbar_push
 * @_this: An #AppBar instance.
 * @_status: Text of status message.
 * Description:
 * Push a new status message onto the status bar stack and display it.
 **/
void appbar_push(AppBar *_this,const gchar *_status){
  g_return_if_fail(_this!=NULL);
  g_return_if_fail(_status!=NULL);
  g_return_if_fail(GJITEN_IS_APPBAR(_this));
  _this->priv->status_stack=
   stringstack_push(_this->priv->status_stack,_status);
  appbar_refresh(_this);
}

/** appbar_pop
 * @_this: An #AppBar instance.
 * Description:
 * Remove current status message, and display previous status message, if any.
 * It is fine to call this with an empty stack.
 **/
void appbar_pop(AppBar *_this){
  g_return_if_fail(_this!=NULL);
  g_return_if_fail(GJITEN_IS_APPBAR(_this));
  _this->priv->status_stack=stringstack_pop(_this->priv->status_stack);
  appbar_refresh(_this);
}

/** appbar_clear_stack
 * @_this: An #AppBar instance.
 * Description:
 * Remove all status messages from @appbar, and display default status message
 *  (if present).
 **/
void appbar_clear_stack(AppBar *_this){
  g_return_if_fail(_this!=NULL);
  g_return_if_fail(GJITEN_IS_APPBAR(_this));
  stringstack_free(_this->priv->status_stack);
  _this->priv->status_stack=NULL;
  appbar_refresh(_this);
}

/** appbar_set_progress_percentage
 * @_this: An #AppBar instance.
 * @_percentage: Percentage to which progress bar should be set.
 * Description:
 * Sets progress bar to @percentage.
 **/
void appbar_set_progress_percentage(AppBar *_this,gfloat _percentage){
  g_return_if_fail(_this!=NULL);
  g_return_if_fail(_this->priv->progress!=NULL);
  g_return_if_fail(GJITEN_IS_APPBAR(_this));
#if defined(MAEMO)
  if(_this->priv->prog_banner!=NULL){
    hildon_banner_set_fraction(HILDON_BANNER(_this->priv->prog_banner),
     _percentage);
    /*We don't have a good way of telling when to take down the progress
       display from this API.
      When we're done seems like a good idea, but relies on the application not
       doing anything clever (like counting down the progress from 1 to 0).*/
    if(_percentage>=1)gtk_widget_destroy(_this->priv->prog_banner);
  }
  else{
    /*We own the floating reference, so we have to take one for our parent.*/
    g_object_ref(_this->priv->progress);
    _this->priv->progress=hildon_banner_show_progress(GTK_WIDGET(_this),
     GTK_PROGRESS_BAR(_this->priv->progress),
     _this->priv->status_stack?stringstack_top(_this->priv->status_stack):
     _this->priv->default_status?_this->priv->default_status:"");
    /*Remove the progress bar so it is not destroyed.*/
    g_signal_connect_swapped(_this->priv->prog_banner,"destroy",
     G_CALLBACK(gtk_widget_unparent),_this->priv->progress);
    /*Set the banner reference to NULL on destruction, which can happen at any
       time.*/
    g_signal_connect(_this->priv->prog_banner,"destroy",
     G_CALLBACK(gtk_widget_destroyed),&_this->priv->prog_banner);
  }
#else
  gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(_this->priv->progress),
   _percentage);
#endif
}


/** appbar_get_progress
 * @_this: An #AppBar instance.
 * Description:
 * Retrieves the progress bar widget for further manipulation.
 * Returns: A #GtkProgressBar widget or %NULL if the #AppBar has no progress
 *  bar.
 **/
GtkProgressBar *appbar_get_progress(AppBar *_this){
  g_return_val_if_fail(_this!=NULL,NULL);
  g_return_val_if_fail(GJITEN_IS_APPBAR(_this),NULL);
  /*If this is documented behavior, why is it an error?*/
  g_return_val_if_fail(_this->priv->progress!=NULL,NULL);
  return GTK_PROGRESS_BAR(_this->priv->progress);
}

/** appbar_get_status
 * @_this: An #AppBar instance.
 * Description:
 * Retrieves the statusbar widget.
 * Returns:
 * A pointer to the statusbar widget.
 **/
GtkWidget *appbar_get_status(AppBar *_this){
  g_return_val_if_fail(_this!=NULL,NULL);
  g_return_val_if_fail(GJITEN_IS_APPBAR(_this),NULL);
#if defined(MAEMO)
  /*This is ugly, but we've no way of doing anything else (aside from making a
     dummy value to return), since the real one is transient, and can't be set
     explicitly like the progress bar.*/
  return NULL;
#else
  return _this->priv->status;
#endif
}

static void appbar_finalize(GObject *_object){
  AppBar *this;
  g_return_if_fail(_object!=NULL);
  g_return_if_fail(GJITEN_IS_APPBAR(_object));
  this=GJITEN_APPBAR(_object);
#if defined(MAEMO)
  if(this->priv->prog_banner!=NULL)gtk_widget_destroy(this->priv->prog_banner);
  if(this->priv->progress!=NULL){
    gtk_widget_destroy(this->priv->progress);
    g_object_unref(this->priv->progress);
  }
#endif
  stringstack_free(this->priv->status_stack);
  g_free(this->priv->default_status);
  g_free(this->priv);
  this->priv=NULL;
  if(G_OBJECT_CLASS(parent_class)->finalize!=NULL){
    (*G_OBJECT_CLASS(parent_class)->finalize)(_object);
  }
}

static GObject *appbar_constructor(GType _type,guint _nproperties,
 GObjectConstructParam *_properties){
  GObject  *object;
  AppBar   *this;
  GtkBox   *box;
  gboolean  has_status;
  gboolean  has_progress;
  object=G_OBJECT_CLASS(parent_class)->constructor(_type,
   _nproperties,_properties);
  this=GJITEN_APPBAR(object);
  has_status=this->priv->has_status;
  has_progress=this->priv->has_progress;
  box=GTK_BOX(this);
  box->spacing=4;
  box->homogeneous=FALSE;
  if(has_progress){
    this->priv->progress=gtk_progress_bar_new();
#if defined(MAEMO)
    /*We create the progress bar in advance so the user can customize it.
      Taking control of the floating reference here is needed to ensure it can
       survive successive (or no!) inclusions in a progress banner.*/
    g_object_ref(this->priv->progress);
    gtk_object_sink(GTK_OBJECT(this->priv->progress));
#else
    gtk_box_pack_start(box,this->priv->progress,FALSE,FALSE,0);
    gtk_widget_show(this->priv->progress);
  }
  if(has_status){
    GtkWidget *frame;
    frame=gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame),GTK_SHADOW_IN);
    this->priv->status=gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(this->priv->status),0.0,0.0);
    gtk_widget_set_size_request(this->priv->status,1,-1);
    gtk_box_pack_start(box,frame,TRUE,TRUE,0);
    gtk_container_add(GTK_CONTAINER(frame),this->priv->status);
    gtk_widget_show(frame);
    gtk_widget_show(this->priv->status);
#endif
  }
  return object;
}

static void appbar_set_property(GObject *_object,guint _prop_id,
 const GValue *_value,GParamSpec *_pspec){
  AppBarPrivate *priv;
  priv=GJITEN_APPBAR(_object)->priv;
  switch(_prop_id){
    case PROP_HAS_STATUS:{
      priv->has_status=g_value_get_boolean(_value);
    }break;
    case PROP_HAS_PROGRESS:{
      priv->has_progress=g_value_get_boolean(_value);
    }break;
    default:{
      G_OBJECT_WARN_INVALID_PROPERTY_ID(_object,_prop_id,_pspec);
    }
  }
}

static void appbar_get_property(GObject *_object,guint _prop_id,
 GValue *_value,GParamSpec *_pspec){
  AppBarPrivate *priv;
  priv=GJITEN_APPBAR(_object)->priv;
  switch(_prop_id){
    case PROP_HAS_STATUS:{
      g_value_set_boolean(_value,priv->has_status);
    }break;
    case PROP_HAS_PROGRESS:{
      g_value_set_boolean(_value,priv->has_progress);
    }break;
  }
}

static void appbar_class_init(AppBarClass *_class){
  GObjectClass *gobject_class;
  gobject_class=(GObjectClass *)_class;
  parent_class=(GtkHBoxClass *)g_type_class_peek_parent(_class);
  gobject_class->constructor=appbar_constructor;
  gobject_class->get_property=appbar_get_property;
  gobject_class->set_property=appbar_set_property;
  gobject_class->finalize=appbar_finalize;
  g_object_class_install_property(gobject_class,PROP_HAS_PROGRESS,
   g_param_spec_boolean("has_progress",_("Has Progress"),
   _("Create a progress widget."),FALSE,G_PARAM_READWRITE|G_PARAM_CONSTRUCT));
  g_object_class_install_property(gobject_class,PROP_HAS_STATUS,
   g_param_spec_boolean("has_status",_("Has Status"),
   _("Create a status widget."),FALSE,G_PARAM_READWRITE|G_PARAM_CONSTRUCT));
}
