/*****************************************************************************
 * Copyright: 2010-2011 Christian Fetzer <fetzer.ch@googlemail.com>          *
 * Copyright: 2010-2011 Michael Zanetti <mzanetti@kde.org>                   *
 *                                                                           *
 * This program is free software: you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation, either version 3 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.     *
 *                                                                           *
 ****************************************************************************/

#include "car2goengine.h"

#include "data/gmwparkingspot.h"
#include "data/gmwgasstation.h"
#include "data/gmwvehicle.h"

#include <QtCore/QDebug>
#include <QtCore/QStringList>
#include <QtCore/QFile>
#include <QtCore/QDir>
#include <QtCore/QDateTime>
#include <QtGui/QDesktopServices>

#include <QtXml/QXmlStreamReader>

Car2goEngine::Car2goEngine(const QString &cacheDir) :
    m_networkReplyParkingSpots(NULL),
    m_networkReplyGasStations(NULL),
    m_networkReplyVehicles(NULL),
    m_networkReplyParkingSpotsImage(NULL),
    m_networkReplyGasStationsImage(NULL),
    m_networkReplyVehiclesImage(NULL)
{
    connect(&m_network, SIGNAL(finished(QNetworkReply*)), this, SLOT(receivedData(QNetworkReply*)));

    // Create the cache directory if it does not exist
    m_cacheDir = cacheDir + "/car2go";
    QDir cacheDirectory(m_cacheDir);
    if(!cacheDirectory.exists()){
        qDebug() << "cachedir for car2go doesnt exist... creating" << m_cacheDir;
        cacheDirectory.mkpath(m_cacheDir);
    }

    connect(this, SIGNAL(locationChanged()), SLOT(loadBusinessArea()));
}

QStringList Car2goEngine::supportedLocations()
{
    return QStringList() << "Austin" << "Hamburg" << "Ulm" << "Vancouver";
}

void Car2goEngine::refreshStationary(bool useCache)
{
    refreshGasStations(useCache);
    refreshParkingSpots(useCache);
}

void Car2goEngine::refreshGasStations(bool useCache)
{
    QFile cacheFile(m_cacheDir + "/gasstations.dat");
    if (useCache && cacheFile.open(QFile::ReadOnly)) {
        qDebug() << "Using cache for gas stations.";
        QList<GMWItem*> items;
        QDataStream cacheStream(&cacheFile);
        QTime stats; stats.start();
        while (!cacheStream.atEnd()) {
            GMWGasStation *gasStation = new GMWGasStation(QString(), QString(), QGeoCoordinate(), QPixmap());
            cacheStream >> *gasStation;
            items.append(gasStation);
        }
        qDebug() << "Loading gas stations from cache took" << stats.elapsed() << "ms";
        if (!items.isEmpty()) {
            emit objectsReceived(items);
            emit loadedFromCache(GMWItem::TypeGasStation, QFileInfo(cacheFile).lastModified());
        }
        cacheFile.close();
    } else {
        qDebug() << "Not using cache for gas stations.";

        // Request GasStations
        if (hasDownloadError()) return;
        delete m_networkReplyGasStations;
        qDebug() << "Downloading gas stations";
        m_networkReplyGasStations = m_network.get(QNetworkRequest("http://www.car2go.com/api/v2.0/gasstations?loc=" + location()));
        //m_networkReplyGasStations = m_network.get(QNetworkRequest(QUrl("file:///home/fetzerc/gasstations")));
        m_downloads.append(m_networkReplyGasStations);
        if (m_downloads.size()) {
            qDebug() << "Download started!";
            emit downloadStarted();
        }
    }
}

void Car2goEngine::refreshParkingSpots(bool useCache) {
    QFile cacheFile(m_cacheDir + "/parkingspots.dat");
    if(useCache && cacheFile.open(QFile::ReadOnly)) {
        qDebug() << "Using cache for parking spots.";
        QList<GMWItem*> items;
        QDataStream cacheStream(&cacheFile);
        QTime stats; stats.start();
        while (!cacheStream.atEnd()) {
            GMWParkingSpot *parkingSpot = new GMWParkingSpot(QString(), QString(), QGeoCoordinate(), QPixmap(), 0, 0);
            cacheStream >> *parkingSpot;
            items.append(parkingSpot);
        }
        qDebug() << "Loading parking spots from cache took" << stats.elapsed() << "ms";
        if (!items.isEmpty()) {
            emit objectsReceived(items);
            emit loadedFromCache(GMWItem::TypeParkingLot, QFileInfo(cacheFile).lastModified());
        }
        cacheFile.close();
    } else {
        qDebug() << "Not using cache for parking spots.";

        // Request ParkingSpots
        if (hasDownloadError()) return;
        delete m_networkReplyParkingSpots;
        qDebug() << "Downloading parking spots";
        m_networkReplyParkingSpots = m_network.get(QNetworkRequest("http://www.car2go.com/api/v2.0/parkingspots?loc=" + location()));
        //m_networkReplyParkingSpots = m_network.get(QNetworkRequest(QUrl("file:///home/fetzerc/parkingspots")));
        m_downloads.append(m_networkReplyParkingSpots);
        if (m_downloads.size()) {
            emit downloadStarted();
        }
    }
}

void Car2goEngine::refreshVehicles(bool useCache)
{
    QFile cacheFile(m_cacheDir + "/vehicles.dat");
    if(useCache && cacheFile.open(QFile::ReadOnly) && (QFileInfo(cacheFile).lastModified().addSecs(3600) > QDateTime::currentDateTime())) {
        qDebug() << "Using cache vehicles.";
        QList<GMWItem*> items;
        QDataStream cacheStream(&cacheFile);
        QTime stats; stats.start();
        while (!cacheStream.atEnd()) {
            GMWVehicle *vehicle = new GMWVehicle(QString(), QString(), QGeoCoordinate(), QPixmap(), 0, GMWVehicle::StateUnknown, GMWVehicle::StateUnknown, QString());
            cacheStream >> *vehicle;
            items.append(vehicle);
        }
        qDebug() << "Loading vehicles from cache took" << stats.elapsed() << "ms";
        if (!items.isEmpty()) {
            emit objectsReceived(items);
            emit loadedFromCache(GMWItem::TypeVehicle, QFileInfo(cacheFile).lastModified());
        }
        cacheFile.close();
    } else {
        qDebug() << "Not using cache for vehicles.";

        // Request Vehicles
        if (hasDownloadError()) return;
        delete m_networkReplyVehicles;
        qDebug() << "Downloading vehicles";
        m_networkReplyVehicles = m_network.get(QNetworkRequest("http://www.car2go.com/api/v2.0/vehicles?loc=" + location()));
        //m_networkReplyVehicles = m_network.get(QNetworkRequest(QUrl("file:///home/fetzerc/vehicles")));
        m_downloads.append(m_networkReplyVehicles);
        if (m_downloads.size() > 1) {
            emit downloadStarted();
        }
    }
}

QGeoBoundingBox Car2goEngine::startingBounds()
{
    if (location() == "Austin") {
        return QGeoBoundingBox(QGeoCoordinate(30.29000,  -97.77000), QGeoCoordinate(30.25000, -97.70000));
    } else if (location() == "Hamburg") {
        return QGeoBoundingBox(QGeoCoordinate(53.72000, 9.84000), QGeoCoordinate(53.41300, 10.15500));
    } else if (location() == "Ulm") {
        return QGeoBoundingBox(QGeoCoordinate(48.41000, 9.96000), QGeoCoordinate(48.39000, 10.00000));
    } else if(location() == "Vancouver") {
        return QGeoBoundingBox(QGeoCoordinate(49.314789, -123.237269), QGeoCoordinate(49.230886, -123.062995));
    } else {
        return QGeoBoundingBox();
    }
}

GMWBusinessArea Car2goEngine::businessArea()
{
    return m_businessArea;
}

void Car2goEngine::receivedData(QNetworkReply *reply)
{
    //qDebug() << "Reply received: " << reply->request().url();
    m_downloads.removeAll(reply);

    if (reply->error() != QNetworkReply::NoError) {
        foreach (QNetworkReply *reply, m_downloads) {
            reply->abort();
            qDebug() << reply->request().url() << "aborted";
        }
        m_downloads.clear();
        emit downloadFinished(GMWItem::TypeUnknown, false, reply->errorString());
        return;
    }

    QByteArray replyData = reply->readAll();
    if (reply == m_networkReplyParkingSpots) {
        m_xmlParkingSpots = QString::fromUtf8(replyData);
        parseImage(ParkingSpotsReply, m_xmlParkingSpots);
    } else if (reply == m_networkReplyGasStations) {
        m_xmlGasStations = QString::fromUtf8(replyData);
        parseImage(GasStationReply, m_xmlGasStations);
    } else if (reply == m_networkReplyVehicles) {
        m_xmlVehicles = QString::fromUtf8(replyData);
        parseImage(VehiclesReply, m_xmlVehicles);
    } else if (reply == m_networkReplyParkingSpotsImage) {
        m_imageParkingSpots.loadFromData(replyData);
        parse(ParkingSpotsReply, m_xmlParkingSpots);
    } else if (reply == m_networkReplyGasStationsImage) {
        m_imageGasStations.loadFromData(replyData);
        parse(GasStationReply, m_xmlGasStations);
    } else if (reply == m_networkReplyVehiclesImage) {
        m_imageVehicles.loadFromData(replyData);
        parse(VehiclesReply, m_xmlVehicles);
    } else {
        qDebug() << "Error: Unknown data received.";
    }
}

void Car2goEngine::parseImage(NetworkReplyType type, const QString &xmlData)
{
    QXmlStreamReader xml(xmlData);
    while (!xml.atEnd()) {
        xml.readNext();
        //qDebug() << xml.tokenString() << " " << xml.name();
        if (xml.tokenType() != QXmlStreamReader::StartElement) continue;
        if (xml.name() == "IconStyle") {
            while (!(xml.tokenType() == QXmlStreamReader::EndElement && xml.name() == "IconStyle")) {
                xml.readNext();
                //qDebug() << xml.tokenString() << " " << xml.name();
                if (xml.tokenType() != QXmlStreamReader::StartElement) continue;
                if (xml.name() == "Icon") {
                    while (!(xml.tokenType() == QXmlStreamReader::EndElement && xml.name() == "Icon")) {
                        xml.readNext();
                        //qDebug() << xml.tokenString() << " " << xml.name();
                        if (xml.tokenType() != QXmlStreamReader::StartElement) continue;
                        if (xml.name() == "href") {
                            xml.readNext();
                            //qDebug() << xml.text().toString();
                            if (!hasDownloadError()) {
                                if (type == ParkingSpotsReply) {
                                    delete m_networkReplyParkingSpotsImage;
                                    m_networkReplyParkingSpotsImage = m_network.get(QNetworkRequest(xml.text().toString()));
                                    m_downloads.append(m_networkReplyParkingSpotsImage);
                                } else if (type == GasStationReply) {
                                    delete m_networkReplyGasStationsImage;
                                    m_networkReplyGasStationsImage = m_network.get(QNetworkRequest(xml.text().toString()));
                                    m_downloads.append(m_networkReplyGasStationsImage);
                                } else if (type == VehiclesReply) {
                                    delete m_networkReplyVehiclesImage;
                                    m_networkReplyVehiclesImage = m_network.get(QNetworkRequest(xml.text().toString()));
                                    m_downloads.append(m_networkReplyVehiclesImage);
                                } else {
                                    qDebug() << "Error: Unknown image requestet.";
                                }
                            }
                            return;
                        }
                    }
                }
            }
        }
    }
}

void Car2goEngine::parse(NetworkReplyType type, const QString &xmlData)
{
    QString name;
    QString description;
    QStringList coordinates;
    QGeoCoordinate location;
    QMap<QString, QString> extendedData;
    QString extendedDataName;

    QList<GMWItem*> parsedItems;
    QTime stats; stats.start();

    QXmlStreamReader xml(xmlData);
    while (!xml.atEnd()) {
        xml.readNext();
        //qDebug() << xml.tokenString() << " " << xml.name();

        // Placemark
        if (xml.name() == "Placemark") {

            // name, description, Point, ExtendedData
            while (!(xml.tokenType() == QXmlStreamReader::EndElement && xml.name() == "Placemark")) {
                xml.readNext();
                //qDebug() << xml.tokenString() << " " << xml.name();

                if (xml.tokenType() != QXmlStreamReader::StartElement) continue;
                if (xml.name() == "name") {
                    xml.readNext();
                    name = xml.text().toString();
                } else if (xml.name() == "description") {
                    xml.readNext();
                    description = xml.text().toString();
                } else if (xml.name() == "Point") {
                    while (!(xml.tokenType() == QXmlStreamReader::EndElement && xml.name() == "Point")) {
                        xml.readNext();
                        //qDebug() << xml.tokenString() << " " << xml.name();

                        if (xml.tokenType() != QXmlStreamReader::StartElement) continue;
                        if (xml.name() == "coordinates") {
                            xml.readNext();
                            coordinates = xml.text().toString().split(",");
                            location = QGeoCoordinate(coordinates[1].toDouble(), coordinates[0].toDouble(), coordinates[2].toDouble());
                        }
                    }
                } else if (xml.name() == "ExtendedData") {
                    while (!(xml.tokenType() == QXmlStreamReader::EndElement && xml.name() == "ExtendedData")) {
                        xml.readNext();
                        //qDebug() << xml.tokenString() << " " << xml.name();

                        if (xml.tokenType() != QXmlStreamReader::StartElement) continue;
                        if (xml.name() == "Data") {
                            extendedDataName = xml.attributes().value("name").toString();
                            while (!(xml.tokenType() == QXmlStreamReader::EndElement && xml.name() == "Data")) {
                                xml.readNext();
                                //qDebug() << xml.tokenString() << " " << xml.name();

                                if (xml.tokenType() != QXmlStreamReader::StartElement) continue;
                                if (xml.name() == "value") {
                                    xml.readNext();
                                    extendedData.insert(extendedDataName, xml.text().toString());
                                    //qDebug() << extendedDataName << xml.text().toString();
                                }
                            }
                        }
                    }
                }
            }

            //qDebug() << name << description << location;

            if (type == ParkingSpotsReply) {
                GMWParkingSpot *parkingSpot = new GMWParkingSpot(
                    "car2go Parking Spot",
                    name,
                    location,
                    m_imageParkingSpots,
                    extendedData.value("usedCapacity").toInt(),
                    extendedData.value("totalCapacity").toInt()
                );
                parsedItems.append(parkingSpot);
            } else if (type == GasStationReply) {
                GMWGasStation *gasStation = new GMWGasStation(
                    "Shell car2go Partner Gas Station",
                    name,
                    location,
                    m_imageGasStations
                );
                parsedItems.append(gasStation);
            } else if (type == VehiclesReply) {
                //qDebug() << description.section("<br/>", 0, 0);
                GMWVehicle *vehicle = new GMWVehicle(
                    name,
                    description.section("<br/>", 0, 0),
                    location,
                    m_imageVehicles,
                    extendedData.value("fuel").toInt(),
                    GMWVehicle::vehicleStateFromString(extendedData.value("interior")),
                    GMWVehicle::vehicleStateFromString(extendedData.value("exterior")),
                    extendedData.value("vin")
                );
                parsedItems.append(vehicle);
            } else {
                qDebug() << "Error: Unknown object received.";
            }
        }
    }

    if (!parsedItems.isEmpty()) {
        qDebug() << "Parsing of" << parsedItems.count() << "items took" << stats.restart() << "ms";

        // Write the parsed item to the cache
        QString cacheFileName;
        if (type == GasStationReply) {
            cacheFileName = "gasstations.dat";
        } else if (type == ParkingSpotsReply) {
            cacheFileName = "parkingspots.dat";
        } else if (type == VehiclesReply) {
            cacheFileName = "vehicles.dat";
        } else {
            qDebug() << "Error: Unknown object received.";
        }
        QFile cacheFile(m_cacheDir + "/" + cacheFileName);
        qDebug() << "Writing" << parsedItems.count() << "items to cachefile:" << cacheFile.fileName();
        cacheFile.open(QFile::WriteOnly);
        QDataStream cacheStream(&cacheFile);
        foreach (GMWItem* item, parsedItems) {
            if (type == GasStationReply) {
                cacheStream << dynamic_cast<GMWGasStation&>(*item);
            } else if (type == ParkingSpotsReply) {
                cacheStream << dynamic_cast<GMWParkingSpot&>(*item);
            } else if (type == VehiclesReply) {
                cacheStream << dynamic_cast<GMWVehicle&>(*item);
            }
        }
        cacheFile.close();
        qDebug() << "Writing to cache took" << stats.elapsed() << "ms";

        // Notify Map and List
        emit objectsReceived(parsedItems);
    }

    GMWItem::Type itemType = GMWItem::TypeUnknown;
    switch(type) {
    case ParkingSpotsReply:
        itemType = GMWItem::TypeParkingLot;
        break;
    case GasStationReply:
        itemType = GMWItem::TypeGasStation;
        break;
    case VehiclesReply:
        itemType = GMWItem::TypeVehicle;
        break;
    case IconReply:
        itemType = GMWItem::TypeUnknown;
    }

    if (xml.hasError()) {
        qDebug() << "Xml has errors!";
        emit downloadFinished(itemType, false, tr("Downloaded Data contains errors"));
    } else {
        // Emit the finished signal only if this is the last download in progress.
        if (m_downloads.size() == 0) {
            emit downloadFinished(itemType, true, "");
        }
    }
}

bool Car2goEngine::hasDownloadError() const
{
    foreach (QNetworkReply *reply, m_downloads) {
        if (reply->error() != QNetworkReply::NoError) {
            qDebug() << "Download has error" << reply->errorString();
            return true;
        }
    }
    return false;
}

void Car2goEngine::loadBusinessArea()
{
    qDebug() << "****************************** loading business area";
#ifdef Q_WS_MAEMO_5
    QFile file("/opt/usr/share/getmewheels/data/" + location() + ".gpx");
#else
    QFile file("data/" + location() + ".gpx");
#endif
    file.open(QIODevice::ReadOnly);
    QXmlStreamReader xml(&file);

    QList<Area> areaList;
    QList<Area> excludeList;
    while(!xml.atEnd()) {
        xml.readNext();
//        qDebug() << xml.tokenString() << " " << xml.name();

        // Track
        if (xml.name() == "trk") {
            xml.readNext();

            QList<QGeoCoordinate> path;

            bool exclude = false;
            while(!(xml.tokenType() == QXmlStreamReader::EndElement && xml.name() == "trk")) {
                  xml.readNext();
//                  qDebug() << xml.tokenString() << " " << xml.name();
                  if(xml.tokenType() == QXmlStreamReader::StartElement && xml.name() == "name") {
                      xml.readNext();
                      qDebug() << "got track" << xml.text();
                      if(xml.text().toString().startsWith("Exclude_")) {
                          exclude = true;
                      } else {
                          exclude = false;
                      }
                  }
                  if(xml.tokenType() == QXmlStreamReader::StartElement && xml.name() == "trkseg") {
                      xml.readNext();
                      qDebug() << "got track segment:" << xml.text();
                      while(!(xml.tokenType() == QXmlStreamReader::EndElement && xml.name() == "trkseg")) {
                          xml.readNext();
                          if(xml.tokenType() == QXmlStreamReader::StartElement && xml.name() == "trkpt") {
                              QGeoCoordinate coord;
                              coord.setLongitude(xml.attributes().value("lon").toString().toDouble());
                              coord.setLatitude(xml.attributes().value("lat").toString().toDouble());
                              path.append(coord);
                          }
                      }
                  }

            }

            Area area;
            area.setPath(path);
//            qDebug() << "appending path" << path;
            if(!exclude) {
                areaList.append(area);
            } else {
                excludeList.append(area);
            }
        }

    }
    m_businessArea.setAreaList(areaList);
    m_businessArea.setExcludes(excludeList);
}
