/*
 * Copyright 2010 Felipe Crochik <foss@crochik.com>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QDebug>
#include <QtWebKit/QWebView>
#include <QWebFrame>
#include <QMenuBar>

#include <QApplication>
#include <QMessageBox>
#include <QDir>
#include <QNetworkDiskCache>
#include <QToolButton>
#include <QResizeEvent>
#include <QTimer>

#include <QStringBuilder>

#include <QSettings>
#include <QShortcut>
#include <QWebInspector>

#include <pickselectoraction.h>

#ifdef Q_WS_MAEMO_5
    #include <QMaemo5InformationBox>
    #define BASECLASSWND Maemo5Wnd
#else
    #define BASECLASSWND QMainWindow
#endif

#include "geocoderequester.h"
#include "googledirections.h"
#include "fingerscrolleventfilter.h"

#include "finddlg.h"
#include "directionsdlg.h"
#include "currdirectionsdlg.h"
#include "searchdlg.h"

MainWindow::MainWindow(QWidget *parent) :
    BASECLASSWND(parent),
    ui(new Ui::MainWindow)
{
    m_isReady = false;
    m_pViewActionGroup = NULL;
    m_pRequester = NULL;
    m_pNetAccessMngr = NULL;
    m_pDirectionsDlg = NULL;
    m_pCurrDirectionsDlg = NULL;
    m_pSearchDlg = NULL;

    m_pGoogleDirections = NULL;
    m_pTrafficLayer = NULL;
    m_pCyclingLayer = NULL;

    m_pAutoRotateAction = NULL;
    m_pAccuracyAction = NULL;
#ifdef Q_WS_MAEMO_5
    m_pSourceUpdatesAction = NULL;
#endif

    for ( int c=0; c<4; c++ ) m_pToolButtons[c] = NULL;

    m_isAutoCenter = true;
    m_isAutoRotate = false;
    m_angle =0;

    ui->setupUi(this);

    m_startAddress.setCoordinates( 40.7527700 ,  -73.9765000); // grand central station ny

    init();
}

MainWindow::~MainWindow() {
    delete ui;

    if ( m_pRequester ) { delete m_pRequester; m_pRequester=NULL; }
    if ( m_pGoogleDirections ) { delete m_pGoogleDirections; m_pGoogleDirections=NULL; }
    if ( m_pDirectionsDlg ) { delete m_pDirectionsDlg; m_pDirectionsDlg=NULL; }
    if ( m_pCurrDirectionsDlg ) { delete m_pCurrDirectionsDlg; m_pCurrDirectionsDlg=NULL; }
    if ( m_pSearchDlg ) { delete m_pSearchDlg; m_pSearchDlg=NULL; }
}

void MainWindow::init() {
#ifdef Q_WS_MAEMO_5
    QString cachePath = "/home/user/.geeps/cache";

    setAttribute(Qt::WA_Maemo5StackedWindow);
    setWindowFlags(windowFlags() | Qt::Window);

#else
    QString cachePath = QApplication::applicationDirPath() + "/cache";

#endif

    QDir dir(cachePath);
    if ( !dir.exists() ) {
        if ( !dir.mkpath(dir.path()) ) {
            qCritical() << "error creating app folder: " << dir.path();
        }
    }

    qDebug() << "cache folder:" << cachePath;
    QNetworkDiskCache *pCache = new QNetworkDiskCache(this);
    pCache->setCacheDirectory(cachePath);

    dir = QDir(cachePath + "/persistent");
    if ( !dir.exists() ) {
        if ( !dir.mkpath(dir.path()) ) {
            qCritical() << "error creating app folder: " << dir.path();
        }
    }
    QWebSettings::enablePersistentStorage(dir.path());

    m_pNetAccessMngr = new QNetworkAccessManager(this);
    m_pNetAccessMngr->setCache(pCache);

    // connect(m_pNetAccessMngr, SIGNAL(finished(QNetworkReply*)), SLOT(onRequestfinished(QNetworkReply*)) );

    QWebPage *pPage = ui->webView->page();
    QWebFrame *pFrame = pPage->mainFrame();

    pFrame->setScrollBarPolicy(Qt::Vertical, Qt::ScrollBarAlwaysOff);
    pFrame->setScrollBarPolicy(Qt::Horizontal, Qt::ScrollBarAlwaysOff);

    pPage->setNetworkAccessManager(m_pNetAccessMngr);
    pPage->settings()->setAttribute(QWebSettings::DeveloperExtrasEnabled, true);

    ui->webView->settings()->setAttribute(QWebSettings::DeveloperExtrasEnabled, true);

    dir = QDir(cachePath + "/local");
    if ( !dir.exists() ) {
        if ( !dir.mkpath(dir.path()) ) {
            qCritical() << "error creating app folder: " << dir.path();
        }
    }
    pPage->settings()->setLocalStoragePath(dir.path());

    // offline?
    dir = QDir(cachePath + "/offline");
    if ( !dir.exists() ) {
        if ( !dir.mkpath(dir.path()) ) {
            qCritical() << "error creating app folder: " << dir.path();
        }
    }
    QWebSettings::setOfflineStoragePath(dir.path());

    pPage->settings()->setAttribute(QWebSettings::OfflineStorageDatabaseEnabled, true);
    pPage->settings()->setAttribute(QWebSettings::OfflineWebApplicationCacheEnabled, true);
    connect(pFrame, SIGNAL(javaScriptWindowObjectCleared()), SLOT(javaScriptWindowObjectCleared()));

    connect(ui->webView, SIGNAL(loadFinished(bool)), SLOT(onLoadFinished(bool)));

    // "requesters"
    m_pRequester = new GeocodeRequester(m_pNetAccessMngr, this);
    connect(m_pRequester, SIGNAL(finish(bool,QString)), SLOT(onFindDestinationAddress(bool,QString)));

    m_pGoogleDirections = new GoogleDirections(m_pNetAccessMngr, this);
    connect(m_pGoogleDirections, SIGNAL(finish(bool,QString)), SLOT(onDirectionsFinished(bool,QString)));

#ifdef Q_WS_MAEMO_5
    connect(&m_geoLocation, SIGNAL(updated()), SLOT(onPositionUpdated()));
    grabZoomKeys(true);

    // tool buttons
    m_pToolButtons[0] = new ToolButton(QIcon(":/images/west.png"), 0, this);
    m_pToolButtons[1] = new ToolButton(QIcon(":/images/north.png"), 0, this);
    m_pToolButtons[2] = new ToolButton(QIcon(":/images/east.png"), 0, this);
    m_pToolButtons[3] = new ToolButton(QIcon(":/images/south.png"), 0, this);
    connect(m_pToolButtons[0], SIGNAL(clicked()), SLOT(onWest()));
    connect(m_pToolButtons[1], SIGNAL(clicked()), SLOT(onNorth()));
    connect(m_pToolButtons[2], SIGNAL(clicked()), SLOT(onEast()));
    connect(m_pToolButtons[3], SIGNAL(clicked()), SLOT(onSouth()));
    for ( int c=0; c<4; c++ ) m_pToolButtons[c]->setVisible(false);
#endif

    /*
    MouseSuppressor *pSupperssor = new MouseSuppressor(ui->webView);
    connect(pSupperssor, SIGNAL(click(QMouseEvent*)), SLOT(onClick(QMouseEvent*)));
    */

    FingerScrollEventFilter *pFilter = new FingerScrollEventFilter(ui->webView);
    connect(pFilter, SIGNAL(scroll(QPoint)), SLOT(onScroll(QPoint)));

    // directions dlg
    Q_ASSERT(!m_pDirectionsDlg);
    m_pDirectionsDlg = new DirectionsDlg(this);
#ifdef Q_WS_MAEMO_5
    m_pDirectionsDlg->setAttribute(Qt::WA_Maemo5StackedWindow);
    m_pDirectionsDlg->setWindowFlags(Qt::Window);
    // set default settings
    // ...

    // start gps
    QSettings settings;
    onSourceUpdates(settings.value("UpdateMode", 0).toInt());
#endif

    load();
}

void MainWindow::load() {
    ui->webView->setUrl( QUrl::fromUserInput("http://www.crochik.com/geeps/geeps-102510.html") );
}

void MainWindow::onLoadFinished(bool isSuccess) {
    if ( !isSuccess ) {
        int btt = QMessageBox::question(
            this, "Failed to Load Map", "Would you like to try again?",
            QMessageBox::Yes, QMessageBox::No
        );
        if ( btt==QMessageBox::Yes ) {
            // load();
            QTimer::singleShot(100,this, SLOT(load()));
            return;
        }

        // abort
        close();
        return;
    }

    // ready!
    qDebug() << "loaded html page";
    ui->webView->disconnect(this, SLOT(onLoadFinished(bool)));

    m_isReady = true; // ready to start javascript

    // init map
    QString script =
        QString("var latlng = new google.maps.LatLng(%1, %2);").arg(
            m_startAddress.latitudeStr(),
            m_startAddress.longitudeStr()
        ) %
        "var myOptions = {"
        "zoom: 18,"
        "center: latlng,"
        "mapTypeId: google.maps.MapTypeId.ROADMAP, "
#ifndef Q_WS_WIN
        "disableDefaultUI: true, "
        "disableDoubleClickZoom: true,"
        "draggable: false,"
        "scrollwheel: false,"
        "mapTypeControl: false,"
        "navigationControl: false,"
        "scaleControl: false,"
#else
        "disableDefaultUI: false, "
        "disableDoubleClickZoom: false,"
        "draggable: true,"
        "scrollwheel: true,"
        "mapTypeControl: false,"
        "navigationControl: false,"
        "scaleControl: false,"
        "keyboardShortcuts: true,"
#endif
        "streetViewControl: false"
        "};"
        "map = new google.maps.Map(document.getElementById(\"map_canvas\"), myOptions);"
        "directionsDisplay.setMap(map);";

    runJavascript(script);

    createMenu();
}

void MainWindow::createMenu() {
    QSettings settings;

    // views
    int viewMode = settings.value("Mode", 1).toInt();
    m_pViewActionGroup = new QActionGroup(this);
    m_pViewActionGroup->setExclusive(true);

    QAction *pAction;
    pAction = new QAction("Hybrid", m_pViewActionGroup); pAction->setCheckable(true); pAction->setChecked(viewMode==0);
    pAction = new QAction("Road Map", m_pViewActionGroup); pAction ->setCheckable(true); pAction->setChecked(viewMode==1);
    pAction = new QAction("Satellite", m_pViewActionGroup); pAction->setCheckable(true); pAction->setChecked(viewMode==2);
    pAction = new QAction("Terrain", m_pViewActionGroup); pAction->setCheckable(true); pAction->setChecked(viewMode==3);
    menuBar()->addActions(m_pViewActionGroup->actions());
    connect(m_pViewActionGroup, SIGNAL(triggered(QAction*)), this, SLOT(onViewChanged(QAction*)));
    setView(viewMode);

    // menuBar()->addAction("Reload", this, SLOT(reload()));
    // menuBar()->addAction("Find Address", this, SLOT(find()));

    menuBar()->addAction("Directions", this, SLOT(directions()));

    menuBar()->addAction("Local Search", this, SLOT(search()));

    // menuBar()->addAction("Reset Tracking", this, SLOT(resetRoute()));

    /*
    m_pAutoRotateAction = menuBar()->addAction("Auto Rotate", this, SLOT(onAutoRotate()));
    m_pAutoRotateAction->setCheckable(true);
    m_pAutoRotateAction->setChecked(false);
    */

    bool traffic = settings.value("TrafficLayer", false).toBool();
    bool bicycle = settings.value("BicyclingLayer", false).toBool();
    m_pTrafficLayer = menuBar()->addAction("Traffic Layer", this, SLOT(onTrafficLayer()));
    m_pTrafficLayer->setCheckable(true);
    m_pTrafficLayer->setChecked(traffic);
    onTrafficLayer();

    m_pCyclingLayer = menuBar()->addAction("Bicycling Layer", this, SLOT(onBicyclingLayer()));
    m_pCyclingLayer->setCheckable(true);
    m_pCyclingLayer->setChecked(bicycle);
    onBicyclingLayer();

    m_pAccuracyAction = menuBar()->addAction("Show Accuracy", this, SLOT(onShowAccuracy()));
    m_pAccuracyAction->setCheckable(true);
    m_pAccuracyAction->setChecked(settings.value("ShowAccuracy", false).toBool());
    onShowAccuracy();

#ifdef Q_WS_MAEMO_5
    addFullscreenToMenu();
    addOrientationToMenu();
    // new QShortcut(QKeySequence("F"), this, SLOT(onFullscreen()));

    /*
    new QShortcut(QKeySequence("F7"), this, SLOT(zoomIn()));
    new QShortcut(QKeySequence("F8"), this, SLOT(zoomOut()));

    new QShortcut(QKeySequence("8"), this, SLOT(test()));
    new QShortcut(QKeySequence("9"), this, SLOT(test()));
    */

    int updates = settings.value("UpdateMode", 0).toInt();

    m_pSourceUpdatesAction = new PickSelectorAction(this);
    m_pSourceUpdatesAction->setText("Position Updates");
    m_pSourceUpdatesAction->items() << "Off" << "every second" << "every 2 seconds" << "every 5 seconds"
            << "every 10 seconds" << "every 15 seconds" << "every 30 seconds";

            // << "10 per second" << "5 per second"
            // << "every minute" << "every 5 minutes" << "every 10 minutes" << "every 15 minutes" << "every half hour" << "every hour";

    m_pSourceUpdatesAction->setSelectedIndex(updates);

    connect(m_pSourceUpdatesAction, SIGNAL(selectionChanged(int)), this, SLOT(onSourceUpdates(int)));
    menuBar()->addAction(m_pSourceUpdatesAction);
    // onSourceUpdates(updates);
#endif
}

void MainWindow::onShowAccuracy() {
    if ( !m_pAccuracyAction->isChecked() ) runJavascript("if ( radius ) {radius.setMap(null); }");

    QSettings settings;
    settings.setValue("ShowAccuracy", m_pAccuracyAction->isChecked());
}

void MainWindow::onTrafficLayer() {
    QSettings settings;
    settings.setValue("TrafficLayer", m_pTrafficLayer->isChecked());

    if ( m_pTrafficLayer->isChecked() ) {
        // runJavascript("trafficLayer = new google.maps.TrafficLayer(); trafficLayer.setMap(map);");
        runJavascript("if (trafficLayer) { trafficLayer.setMap(map); }");
        return;
    }

    // runJavascript("if (trafficLayer) { trafficLayer.setMap(null); trafficLayer=null; }");
    runJavascript("if (trafficLayer) { trafficLayer.setMap(null); }");
}

void MainWindow::onBicyclingLayer() {
    QSettings settings;
    settings.setValue("BicyclingLayer", m_pCyclingLayer->isChecked());

    if ( m_pCyclingLayer->isChecked() ) {
        // runJavascript("bikeLayer = new google.maps.BicyclingLayer(); bikeLayer.setMap(map);");
        runJavascript("if (bikeLayer) { bikeLayer.setMap(map); }");
        return;
    }

    // runJavascript("if (bikeLayer) { bikeLayer.setMap(null); bikeLayer=null; }");
    runJavascript("if (bikeLayer) { bikeLayer.setMap(null); }");
}

void MainWindow::onAutoRotate() {
    m_isAutoRotate = m_pAutoRotateAction->isChecked();
    if ( !m_isAutoRotate ) {
        // north
        QString script = QString("document.getElementById(\"map_canvas\").style[\"WebkitTransform\"] = \"rotate(%1deg)\";").arg(0);
        runJavascript(script);

        runJavascript("document.getElementById(\"map_canvas\").style[\"width\"] = \"100%\";");
        runJavascript("document.getElementById(\"map_canvas\").style[\"height\"] = \"100%\";");
    } else {
        m_angle = 0;
        // resize to square
        QString sz = QString::number(ui->webView->width()>ui->webView->height() ? ui->webView->height() : ui->webView->width());
        runJavascript(QString("document.getElementById(\"map_canvas\").style[\"width\"] = \"%1px\";").arg(sz));
        runJavascript(QString("document.getElementById(\"map_canvas\").style[\"height\"] = \"%1px\";").arg(sz));
    }
}

void MainWindow::onSourceUpdates(int index) {
    int updates = 0;
    switch (index) {
        // case 1: updates = 10*60*60; break;
        // case 2: updates = 5*60*60; break;
        case 1: updates = 60*60; break;
        case 2: updates = 30*60; break;
        case 3: updates = 12*60; break;
        case 4: updates = 6*60; break;
        case 5: updates = 4*60; break;
        case 6: updates = 2*60; break;
        case 7: updates = 60; break;
        case 8: updates = 12; break;
        case 9: updates = 6; break;
        case 10: updates = 4; break;
        case 11: updates = 2; break;
        case 12: updates = 1; break;
    }

    QSettings settings;
    settings.setValue("UpdateMode", index);

#ifdef Q_WS_MAEMO_5
    m_geoLocation.setUpdatesPerHour(updates);
#endif
}

void MainWindow::onRequestfinished(QNetworkReply*) {
    // qDebug() << "request finished";
}

void MainWindow::onClick(QMouseEvent *) {
    if ( m_pToolButtons[0] && !m_pToolButtons[0]->isVisible() ) m_timerCount = 0;

    // show buttons
    for ( int c=0; c<4; c++ ) if ( m_pToolButtons[c] ) m_pToolButtons[c]->setVisible(true);
    incrementTimer();
}

void MainWindow::incrementTimer() {
    // start timer
    m_timerCount++;
    QTimer::singleShot(5000, this, SLOT(onHideButtons()));
}

void MainWindow::onHideButtons() {
    if ( --m_timerCount ) return;

    for ( int c=0; c<4; c++ ) if ( m_pToolButtons[c] ) m_pToolButtons[c]->setVisible(false);
    showFullScreen();
}

void MainWindow::onViewChanged(QAction* pAction) {
    setView(m_pViewActionGroup->actions().indexOf(pAction));
}

void MainWindow::setView(int index) {
    QString value;
    switch ( index ) {
        case 0: value = "HYBRID"; break;
        case 1: value = "ROADMAP"; break;
        case 2: value = "SATELLITE"; break;
        case 3: value = "TERRAIN"; break;
        default:
            return;
    }

    QSettings settings;
    settings.setValue("Mode", index);
    runJavascript("setMapTypeId(google.maps.MapTypeId." + value + ");");
}

void MainWindow::runJavascript(QString script) {
    if ( !m_isReady ) {
        qDebug() << "page not loaded yet: ignore javascript: " << script;
        return;
    }

    QWebFrame *pFrame = ui->webView->page()->mainFrame();
    qDebug() << script;
    pFrame->evaluateJavaScript(script);
}

void MainWindow::resetRoute() {
    runJavascript("if ( polyLine ) polyLine.setMap(null); polyLine=null;");
}

void MainWindow::search() {
    if ( !m_pSearchDlg ) {
        m_pSearchDlg = new SearchDlg(m_pNetAccessMngr, this);

#ifdef Q_WS_MAEMO_5
        m_pSearchDlg->setAttribute(Qt::WA_Maemo5StackedWindow);
        m_pSearchDlg->setWindowFlags(Qt::Window);
#endif
    }

    m_pSearchDlg->m_latLongStr = QString("%1,%2").arg(
        m_startAddress.latitudeStr(),
        m_startAddress.longitudeStr()
    );

    int res = m_pSearchDlg->exec();

    // add business to map
    // QString script;
    if ( m_pSearchDlg->m_hasChanged ) {
        double lat0,lat1,lng0,lng1;
        for ( int c=0; c<10; c++ ) {
            // create marker
            QString letter = QString('a' + c);

            // remove previous (if any)
            if ( c>=m_pSearchDlg->m_model.rowCount() ) {
                runJavascript( QString("if (mark%1) { mark%1.setMap(null); }").arg(letter) );
                continue;
            }

            // add marker
            Business *pBus = m_pSearchDlg->m_model[c];
            runJavascript(
                QString("if ( mark%1 ) {mark%1.setPosition(new google.maps.LatLng(%2,%3)); mark%1.setMap(map); }").arg(
                    letter,
                    pBus->latitudeStr(),
                    pBus->longitudeStr()
                )
            );

            if ( !c ) {
                lat1 = lat0 = pBus->latitude();
                lng1 = lng0 = pBus->longitude();
                continue;
            }

            if ( lat0>pBus->latitude() ) lat0=pBus->latitude();
            if ( lat1<pBus->latitude() ) lat1=pBus->latitude();
            if ( lng0>pBus->longitude() ) lng0=pBus->longitude();
            if ( lng1<pBus->longitude() ) lng1=pBus->longitude();
        }

        if ( res!= QDialog::Accepted ) {
            // zoom
            m_isAutoCenter = false;
            runJavascript(
                QString("fitBounds(%1,%2,%3,%4);").arg(
                    QString::number(lat0, 'g', 10),
                    QString::number(lng0, 'g', 10),
                    QString::number(lat1, 'g', 10),
                    QString::number(lng1, 'g', 10)
                )
            );
        }
        m_pSearchDlg->m_hasChanged = false;
    }

    if ( res!= QDialog::Accepted )  {
        return;
    }

    // one business selected
    Business bus = m_pSearchDlg->selectedBusiness();
    qDebug() << bus.description() << bus.m_address.address();

    resetDirections();
    m_destination.setCoordinates(bus.latitude(), bus.longitude());
    m_destination.setDescription(
        // assemble address
        QString("%1, %2, %3, %4").arg( //"%1, %2, %3, %4 (%5)"
                bus.m_address.address(),
                bus.m_address.city(),
                bus.m_address.region(),
                bus.m_address.country() // ,
                // bus.description()
        )
        // use lat/long (it only seems to matter for v3 of the api)
        // QString("%1 %2").arg(bus.latitudeStr(), bus.longitudeStr())
    );

#ifdef Q_WS_MAEMO_5
    if ( !m_geoLocation.isUpdating() ) {
        // no current position: show dialog
        directions();
        return;
    }

    // get directions using last settings
    Q_ASSERT(m_pDirectionsDlg);
    showProgressIndicator(true);
    m_pGoogleDirections->query(
        m_startAddress.description(),
        m_destination.description(),
        m_pDirectionsDlg->mode(),
        m_pDirectionsDlg->showAlternatives(),
        m_pDirectionsDlg->units(),
        m_pDirectionsDlg->avoid()
    );
#else
    // no current position: show dialog
    directions();

#endif
}

void MainWindow::directions() {
    if ( m_pCurrDirectionsDlg ) {
        showCurrentDirections();
        return;
    }

    Q_ASSERT(m_pDirectionsDlg);
    m_pDirectionsDlg->setStartAddress(m_startAddress.description()); // !!!!
    m_pDirectionsDlg->setDestinationAddress(m_destination.description());

    bool isUpdating = false;
#ifdef Q_WS_MAEMO_5
    isUpdating = m_geoLocation.isUpdating();
    m_pDirectionsDlg->setStartAddressToCurrentPos(isUpdating);
#endif

    if ( m_pDirectionsDlg->exec() != QDialog::Accepted ) return;

    if ( !isUpdating ) {
        m_startAddress.setDescription(m_pDirectionsDlg->startAddress());
        m_startAddress.setCoordinates(0,0); // ???
    }

    /*
    // absolute coordinates?
    QRegExp exp("^-?\\d+\\.\\d+\\s-?\\d+\\.\\d+$");
    if ( !exp.indexIn(m_pDirectionsDlg->destinationAddress()) ) {
        qDebug() << "assume it is using latitude and longitude";
        // do not change description?
        // ...
        m_destination.setDescription(m_pDirectionsDlg->destinationAddress());
        // set lat/lng
        // m_destination.setCoordinates();

        getDirections();
        return;
    }
    */

    // double check destination
    showProgressIndicator(true);
    m_pRequester->query(m_pDirectionsDlg->destinationAddress());
}

void MainWindow::onFindDestinationAddress(bool isSuccess, QString errorMessage) {
    showProgressIndicator(false);

    if ( !isSuccess ) {
        QMessageBox::warning(this, "Directions - Destination", errorMessage);
        return;
    }

    if ( m_pRequester->m_address.length()!=1 ) {
        Q_ASSERT(m_pDirectionsDlg);
        FindDlg dlg(*m_pRequester, m_pDirectionsDlg->destinationAddress(), this);
        dlg.setWindowTitle("Destination Address");
#ifdef Q_WS_MAEMO_5
        dlg.setAttribute(Qt::WA_Maemo5StackedWindow);
        dlg.setWindowFlags(Qt::Window);
#endif
        if ( dlg.exec()!=QDialog::Accepted ) return;
        m_destination = dlg.m_selectedAddress;

    } else {
        m_destination = m_pRequester->m_address[0];

        // keep original description?
        // m_destination.setDescription(m_pDirectionsDlg->destinationAddress());
    }

    if ( m_pDirectionsDlg ) m_pDirectionsDlg->setDestinationAddress(m_destination.description());

    // save poi ?
    // ...

    getDirections();
}

void MainWindow::getDirections() {
    resetRoute();

    Q_ASSERT(m_pGoogleDirections);
    Q_ASSERT(m_pDirectionsDlg);

    showProgressIndicator(true);
    m_pGoogleDirections->query(
        m_startAddress.description(),
        m_destination.description(),
        m_pDirectionsDlg->mode(),
        m_pDirectionsDlg->showAlternatives(),
        m_pDirectionsDlg->units(),
        m_pDirectionsDlg->avoid()
    );
}

void MainWindow::onDirectionsFinished(bool isSuccess, QString errorMessage) {
    showProgressIndicator(false);

    if ( !isSuccess ) {
        QMessageBox::warning(this, "Directions", errorMessage);
        return;
    }

    if ( m_pCurrDirectionsDlg ) {
        delete m_pCurrDirectionsDlg;
        m_pCurrDirectionsDlg = NULL;
    }

    m_currRouteIndex = 0;
    m_currStepIndex = -1;
    Directions& dir(m_pGoogleDirections->m_directions[m_currRouteIndex]);

    // markers
    runJavascript(
        QString(
            "if ( startMarker ) {startMarker.setPosition(new google.maps.LatLng(%1,%2)); startMarker.setMap(map);} "
        ).arg(
            dir.m_start.latitudeStr(),
            dir.m_start.longitudeStr()
        )
    );
    runJavascript(
        QString(
            "if ( endMarker ) {endMarker.setPosition(new google.maps.LatLng(%1,%2)); endMarker.setMap(map);} "
        ).arg(
            dir.m_end.latitudeStr(),
            dir.m_end.longitudeStr()
        )
    );

    runJavascript(QString("showPolyline(\"%1\");").arg(dir.m_encodedPoints));

    // use google maps (v3)
    // getDirectionsOnMap();

    // create current directions dialog
    m_pCurrDirectionsDlg = new CurrDirectionsDlg(m_pGoogleDirections->m_directions, this);
    m_pCurrDirectionsDlg->setWindowTitle(m_destination.description());
#ifdef Q_WS_MAEMO_5
    m_pCurrDirectionsDlg->setAttribute(Qt::WA_Maemo5StackedWindow);
    m_pCurrDirectionsDlg->setWindowFlags(Qt::Window);
#endif
}

void MainWindow::getDirectionsOnMap() {
    Q_ASSERT(m_pDirectionsDlg);

    QString script = "var request = {";

    // m_pDirectionsDlg->showAlternatives();
    // m_pDirectionsDlg->units();

    script += "travelMode: google.maps.DirectionsTravelMode." % m_pDirectionsDlg->mode().toUpper() % ",";
    foreach (QString str, m_pDirectionsDlg->avoid() ) {
        if ( str=="highways" ) script += "avoidHighways: true,";
        if ( str=="tolls" ) script += "avoidTolls: true,";
    }

    // just show default
    // if ( m_pDirectionsDlg->showAlternatives() ) script += "provideRouteAlternatives: true,";

    script +=
            "origin: \"" % m_startAddress.description()% "\","
            "destination: \"" % m_destination.description()% "\"};"

        "directionsService.route("
            "request,"
            "function(result, status) {"
                "if (status == google.maps.DirectionsStatus.OK) {"
                      "directionsDisplay.setMap(map);"
                      "directionsDisplay.setDirections(result);"
                      "directionsDisplay.setRouteIndex(0);"
                "}"
            "}"
        ");";

    // m_currRouteIndex = 0;

    runJavascript(script);
}

void MainWindow::find() {
    FindDlg dlg(m_pNetAccessMngr, this);
#ifdef Q_WS_MAEMO_5
    dlg.setAttribute(Qt::WA_Maemo5StackedWindow);
    dlg.setWindowFlags(Qt::Window);
#endif

    if ( dlg.exec()!=QDialog::Accepted ) return;

    QWebFrame *pFrame = ui->webView->page()->mainFrame();
    pFrame->evaluateJavaScript(QString("addMark(%1, %2, \"%3\");").arg(
            dlg.m_selectedAddress.latitudeStr(),
            dlg.m_selectedAddress.longitudeStr(),
            dlg.m_selectedAddress.description()
        )
    );
}

void MainWindow::resetDirections() {
    resetRoute();

    // reset v3
    runJavascript("directionsDisplay.setMap(null);");

    //
    runJavascript("if (dirPolyline) {dirPolyline.setMap(null); }");
    runJavascript("if (stepMarker) { stepMarker.setMap(null); }");
    runJavascript("if (startMarker) { startMarker.setMap(null); }");
    runJavascript("if (endMarker) { endMarker.setMap(null); }");

    if ( m_pCurrDirectionsDlg ) {delete m_pCurrDirectionsDlg; m_pCurrDirectionsDlg = NULL;}
}

void MainWindow::showCurrentDirections() {
    Q_ASSERT(m_pCurrDirectionsDlg);

    int res = m_pCurrDirectionsDlg->exec();
    if ( m_pCurrDirectionsDlg->selectedRoutIndex()!=m_currRouteIndex ) {
        // change route
        m_currRouteIndex = m_pCurrDirectionsDlg->selectedRoutIndex();
        showDirections();
    }

#ifdef GOOGLEMAPS_JSON
    QString script = "var json = " % m_pGoogleDirections->m_directions[m_pCurrDirectionsDlg->selectedIndex()].m_json % ";"
        "directionsDisplay.setDirections(json); alert(\"ok!\");";

    runJavascript(script);
#endif

    if ( res!= QMessageBox::Accepted ) {
        return;
    }

    switch ( m_pCurrDirectionsDlg->m_button ) {
        case CurrDirectionsDlg::ButtonClear:
            resetDirections();
            break;

        case CurrDirectionsDlg::ButtonNew:
            resetDirections();
            directions();
            break;

        case CurrDirectionsDlg::ButtonRecalculate:
            resetDirections();

            Q_ASSERT(m_pDirectionsDlg);
            showProgressIndicator(true);
            m_pRequester->query(m_pDirectionsDlg->destinationAddress());
            break;

        case CurrDirectionsDlg::ButtonStep:
            Q_ASSERT(m_pGoogleDirections);
            m_currStepIndex = m_pCurrDirectionsDlg->selectedStepIndex();
            showStep();
            break;

        default:
            break;
    }
}

void MainWindow::showDirections() {
    Directions& dir(m_pGoogleDirections->m_directions[m_currRouteIndex]);
    runJavascript(QString("showPolyline(\"%1\");").arg(dir.m_encodedPoints));

    m_currStepIndex = -1;
    showStep();
}

void MainWindow::showStep() {
    Q_ASSERT(m_pGoogleDirections);

    Directions& dir(m_pGoogleDirections->m_directions[m_currRouteIndex]);
    if ( m_currStepIndex<0 ) {
        // remove maker?
        runJavascript("if (stepMarker) {stepMarker.setMap(null);}");
        return;
    }

    Q_ASSERT(m_currStepIndex>=0 && m_currStepIndex<dir.m_steps.length());
    Step step = dir.m_steps[m_currStepIndex];

    m_isAutoCenter = false;
    runJavascript(
        QString("setCenter(%1, %2);").arg(
            step.latitudeStr(),
            step.longitudeStr()
        )
    );

    QString stepImage = m_currStepIndex<9 ?
        QString("step%1.png").arg(QString::number(1+m_currStepIndex)) :
        QString("step.png");

    runJavascript(
        QString(
            "if ( stepMarker ) {stepMarker.setPosition(new google.maps.LatLng(%1,%2)); "
            "stepMarker.setIcon(\"http://www.crochik.com/geeps/images/%3\");"
            "stepMarker.setMap(map);}"
        ).arg(
            step.latitudeStr(),
            step.longitudeStr(),
            stepImage
        )
    );
}

void MainWindow::reload() {
    ui->webView->reload();
}

void MainWindow::javaScriptWindowObjectCleared() {
    /*
#ifdef Q_WS_MAEMO_5
    QWebFrame *pFrame = ui->webView->page()->mainFrame();
    pFrame->addToJavaScriptWindowObject("crochik", &m_geoLocation);
#endif
    */
}

void MainWindow::changeEvent(QEvent *e) {
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void MainWindow::onPositionUpdated() {
    qDebug() << "MainWindow::onPositionUpdated";

#ifdef Q_WS_MAEMO_5
    if ( m_startAddress.latitude()==m_geoLocation.latitude() && m_startAddress.longitude()==m_geoLocation.longitude() ) {
        // position didn't change... skip update?
        return;
    }

    Location last = m_startAddress;

    // reset start
    m_startAddress.setCoordinates( m_geoLocation.latitude(), m_geoLocation.longitude());
    m_startAddress.setDescription(QString("%1 %2").arg(m_startAddress.latitudeStr(), m_startAddress.longitudeStr()));

    // test of autorotation
    if ( m_isAutoCenter && m_isAutoRotate ) {
        double dx = m_geoLocation.longitude() - last.longitude();
        double dy = m_geoLocation.latitude() - last.latitude();

        int angle = 0;
        if ( (dx>0 && dy>0 && dx>dy) || (dx<0 && dy<0 && dx<dy) || (dx>0 && dy<0 && dx>-dy) || (dx<0 && dy>0 && dx<-dy) ) {
            // horizontal
            angle = dx>0 ? 90 : 270;
        } else {
            // vertical
            angle = dy<0 ? 0 : 180;
        }

        if ( dx || dy ) {
            if ( angle==m_angle ) {
                QString script = QString("document.getElementById(\"map_canvas\").style[\"WebkitTransform\"] = \"rotate(%1deg)\";").arg(angle);
                runJavascript(script);
            } else {

                // resetRoute();
            }
            m_angle = angle;
        }
    }

    runJavascript(
        QString("updateCenter(%1, %2, %3, %4);").arg(
            m_isAutoCenter?"true":"false",
            m_geoLocation.latitudeStr(),
            m_geoLocation.longitudeStr(),
            m_pAccuracyAction && m_pAccuracyAction->isChecked() ? QString::number(m_geoLocation.horizontalAccuracy()) : "0"
        )
    );

    if ( !m_pCurrDirectionsDlg ) return;

    runJavascript(
        QString("track(%1, %2);").arg(
            m_startAddress.latitudeStr(),
            m_startAddress.longitudeStr()
        )
    );
#endif
}

void MainWindow::resizeEvent(QResizeEvent *pEvent) {
    BASECLASSWND::resizeEvent(pEvent);

    ui->webView->setGeometry(0, 0, this->width(), this->height());

    if ( m_pToolButtons[0] ) m_pToolButtons[0]->move(0, (this->height()-m_pToolButtons[0]->height())/2);
    if ( m_pToolButtons[1] ) m_pToolButtons[1]->move((this->width()-m_pToolButtons[1]->width())/2,0);
    if ( m_pToolButtons[2] ) m_pToolButtons[2]->move(this->width()-m_pToolButtons[2]->width(), (this->height()-m_pToolButtons[2]->height())/2);
    if ( m_pToolButtons[3] ) m_pToolButtons[3]->move((this->width()-m_pToolButtons[3]->width())/2, this->height()-m_pToolButtons[3]->height());
}

void MainWindow::zoomIn() {
    runJavascript("zoomIn();");
}

void MainWindow::zoomOut() {
    runJavascript("zoomOut();");
}

void MainWindow::test() {
    // runJavascript("test(10);");
}

void MainWindow::keyPressEvent(QKeyEvent *event) {
    BASECLASSWND::keyPressEvent(event);
}

void MainWindow::keyReleaseEvent(QKeyEvent *event) {
    switch ( event->key() ) {
        case Qt::Key_F7:
            zoomIn();
            break;

        case Qt::Key_F8:
            zoomOut();
            break;

        case Qt::Key_Left:
            if ( (event->modifiers()&Qt::ShiftModifier) == Qt::ShiftModifier ) {
                prevStep();
            } else {
                onWest();
            }
            break;

        case Qt::Key_Right:
            if ( (event->modifiers()&Qt::ShiftModifier) == Qt::ShiftModifier ) {
                nextStep();
            } else {
                onEast();
            }
            break;

        case Qt::Key_Up:
            if ( (event->modifiers()&Qt::ShiftModifier) == Qt::ShiftModifier ) {
                nextRoute();
            } else {
                onNorth();
            }
            break;

        case Qt::Key_Down:
            if ( (event->modifiers()&Qt::ShiftModifier) == Qt::ShiftModifier ) {
                prevRoute();
            } else {
                onSouth();
            }
            break;

        case Qt::Key_Enter:
            m_isAutoCenter = true;
            runJavascript(
                QString("setCenter(%1, %2);").arg(
                    m_startAddress.latitudeStr(),
                    m_startAddress.longitudeStr()
                )
            );
            break;

        case Qt::Key_F:
            if ( isFullScreen() ) {
                showNormal();
            } else {
                onFullscreen();
            }
            break;

        case Qt::Key_0: // = 0x30,
            // ?????
            break;

        case Qt::Key_1: // = 0x31,
        case Qt::Key_2: // = 0x32,
        case Qt::Key_3: // = 0x33,
        case Qt::Key_4: // = 0x34,
        case Qt::Key_5: // = 0x35,
        case Qt::Key_6: // = 0x36,
        case Qt::Key_7: // = 0x37,
        case Qt::Key_8: // = 0x38,
        case Qt::Key_9: // = 0x39,
            runJavascript(QString("setZoom(%1);").arg(
                QString::number((event->key()-Qt::Key_0)*2))
            );
            break;

        default: //ignore
            BASECLASSWND::keyPressEvent(event);
            return;
    }

    event->accept();
}

void MainWindow::prevRoute() {
    // route
    if ( !m_pGoogleDirections || m_pGoogleDirections->m_directions.length()<=1 ) return;
    m_currRouteIndex--;
    if ( m_currRouteIndex<0) m_currRouteIndex=m_pGoogleDirections->m_directions.length()-1;
    showDirections();
}

void MainWindow::nextRoute() {
    // route
    if ( !m_pGoogleDirections || m_pGoogleDirections->m_directions.length()<=1 ) return;
    m_currRouteIndex++;
    if ( m_currRouteIndex>=m_pGoogleDirections->m_directions.length()) m_currRouteIndex=0;
    showDirections();
}

void MainWindow::nextStep() {
    // step
    if ( !m_pGoogleDirections || m_currRouteIndex<0 ) return;
    if ( m_currRouteIndex>=m_pGoogleDirections->m_directions.length() ) return;
    Directions &dir(m_pGoogleDirections->m_directions[m_currRouteIndex]);
    m_currStepIndex++;
    if ( m_currStepIndex>=dir.m_steps.length() ) m_currStepIndex = dir.m_steps.length()-1; // 0;
    showStep();
}

void MainWindow::prevStep() {
    // step
    if ( !m_pGoogleDirections || m_currRouteIndex<0 ) return;
    if ( m_currRouteIndex>=m_pGoogleDirections->m_directions.length() ) return;

    // Directions &dir(m_pGoogleDirections->m_directions[m_currRouteIndex]);
    m_currStepIndex--;
    if ( m_currStepIndex<0 ) m_currStepIndex = 0; //dir.m_steps.length()-1;
    showStep();
}

void MainWindow::onScroll(QPoint off) {
    m_isAutoCenter = false;
    runJavascript(
        QString("panBy(%1,%2);").arg(
            QString::number(-off.x()),
            QString::number(-off.y())
        )
    );
}

void MainWindow::onWest() {
    incrementTimer();
    m_isAutoCenter = false;
    runJavascript(QString("panBy(%1,0);").arg(-ui->webView->width()/2));
}

void MainWindow::onEast() {
    incrementTimer();
    m_isAutoCenter = false;
    runJavascript(QString("panBy(%1,0);").arg(ui->webView->width()/2));
}

void MainWindow::onNorth() {
    incrementTimer();
    m_isAutoCenter = false;
    runJavascript(QString("panBy(0,%1);").arg(-ui->webView->height()/2));
}

void MainWindow::onSouth() {
    incrementTimer();
    m_isAutoCenter = false;
    runJavascript(QString("panBy(0,%1);").arg(ui->webView->height()/2));
}

void MainWindow::showProgressIndicator(bool show) {
#ifdef Q_WS_MAEMO_5
    ShowProgressIndicator(this, show);
#endif
}

// ---------------------------------------------------------------------------------------------------------------------
MouseSuppressor::MouseSuppressor(QWebView *pView)
    : QObject(pView)
{
    pView->installEventFilter(this);
}

bool MouseSuppressor::eventFilter(QObject *, QEvent *e) {
    switch (e->type()) {
        case QEvent::MouseButtonPress:
            if (static_cast<QMouseEvent *>(e)->button() == Qt::LeftButton) m_isMousePressed = true;
            return true;

        case QEvent::MouseButtonRelease:
            if ( m_isMousePressed ) {
                QMouseEvent *pEvent = static_cast<QMouseEvent *>(e);
                click(pEvent);
                m_isMousePressed = false;
            }
            return true;

        case QEvent::MouseMove:
            return true;

        default:
            break;
    }
    return false;
}


