/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007, 2008 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "ui.h"
#include "uireport.h"

/* Report window */
#define REPORTYEARS 4
#define HEADERLINE 0.1
#define LEFTMARGIN 0.15
#define INDENT 0.05
#define FIRSTCOLUMN 0.45
#define COLUMNSEP 0.1
#define HEADING1SIZE 16
#define HEADING2SIZE 14
#define NORMALSIZE 12

/* These tags are used to identify texts that need to be changed
   according to the selected units */
#define CO2TAG "CO2TAG"
#define TOTALCO2TAG "TOTALCO2TAG"
#define CONSUMETAG "CONSUMETAG"
#define COSTPERKMTAG "COSTPERKMTAG"
#define COSTTAG "COSTTAG"
#define PRICEPERUNITTAG "PRICEPERUNITTAG"

extern unit_t curunit;
extern gboolean carchanged;
extern enum papertype reportpaper;

static gboolean delete_reportwin( GtkWidget *widget,
				  GdkEvent  *event,
				  gpointer   data )
{
  extern AppUIData ui;

  gtk_widget_destroy(GTK_WIDGET(ui.app->reportwindow));
  return FALSE;
}

void destroy_reportwin( GtkWidget *widget,
		      gpointer   data )
{
  AppUIData *a;

  a=(AppUIData *)data;

  gtk_widget_destroy(GTK_WIDGET(a->app->reportwindow));
  a->app->reportwindow=NULL;
}

void report_window_fullscreen(gpointer data)
{
  AppUIData *a;

  a=(AppUIData *)data;
  a->reportfullscreen = !a->reportfullscreen;
  if (a->reportfullscreen) {
    gtk_window_fullscreen(GTK_WINDOW(a->app->reportwindow));
  } else {
    gtk_window_unfullscreen(GTK_WINDOW(a->app->reportwindow));
  }
}

void cb_report_fullscreen( GtkAction * action, gpointer data )
{
  report_window_fullscreen(data);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(((AppUIData *)(data))->report_menu_item_fullscreen),
				 ((AppUIData *)(data))->reportfullscreen);
}

#if PLAINGTK == 0
void report_scroll(AppUIData *pui, gboolean dir)
{
  GtkAdjustment *vertical;

  vertical = gtk_scrolled_window_get_vadjustment((GtkScrolledWindow *)pui->reportscrollwin);
  if (dir) {
    vertical->value -= vertical->step_increment;
    if (vertical->value<vertical->lower)
      vertical->value = vertical->lower;
  }
  else {
    vertical->value += vertical->step_increment;
    if (vertical->value>vertical->upper)
      vertical->value = vertical->upper;
  }
  gtk_scrolled_window_set_vadjustment((GtkScrolledWindow *)pui->reportscrollwin,vertical);
  gtk_adjustment_value_changed(vertical);
}

gboolean report_key_press_cb(GtkWidget * widget, GdkEventKey * event,
			     AppUIData *pui)
{
  switch (event->keyval) {

    case GDK_F6:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->report_menu_item_fullscreen),
				     !pui->reportfullscreen);
      return TRUE;
    case GDK_Down:
      report_scroll(pui,FALSE);
      return TRUE;
      break;
    case GDK_Up:
      report_scroll(pui,TRUE);
      return TRUE;
    case GDK_F7:
      report_zoom(pui->reportcanvas, ZOOMIN);
      return TRUE;
    case GDK_F8:
      report_zoom(pui->reportcanvas, ZOOMOUT);
      return TRUE;
      break;
    }

    return FALSE;
}
#endif

#if PLAINGTK == 1
void create_reportwin_menu(GtkWidget *window, GtkWidget *root, AppUIData *pui)
#else
void create_reportwin_menu(HildonWindow *window, AppUIData *pui)
#endif
{
  GtkWidget *main_menu;
  GtkWidget *menu_view;
  GtkWidget *item_view;
  GtkWidget *menu_zoom;
  GtkWidget *item_save;
  GtkWidget *item_zoom;
  GtkWidget *item_zoom_in;
  GtkWidget *item_zoom_out;
  GtkWidget *item_help;
  GtkWidget *item_exit;
#if PLAINGTK == 1
  GtkWidget *menubar;
  GtkWidget *item_file;
#endif

#if MAEMO_VERSION_MAJOR == 5
  main_menu = HILDON_APP_MENU (hildon_app_menu_new ());

  item_save = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					  HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					  _("Save report"),
					  NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_save));

  item_exit = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					  HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					  _("Close"),
					  NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_exit));

  item_zoom_in = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					     HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					     _("Zoom in"),
					     NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_zoom_in));

  item_zoom_out = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					      _("Zoom out"),
					      NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_zoom_out));

  g_signal_connect(G_OBJECT(item_save), "clicked",
		   G_CALLBACK(callback_save_report), pui->reportcanvas);
  g_signal_connect_swapped(G_OBJECT(item_exit), "clicked",
			   G_CALLBACK(delete_reportwin), G_OBJECT(window));
  g_signal_connect(G_OBJECT(item_zoom_in), "clicked",
		   G_CALLBACK(callback_report_zoom_in), pui->reportcanvas);
  g_signal_connect(G_OBJECT(item_zoom_out), "clicked",
		   G_CALLBACK(callback_report_zoom_out), pui->reportcanvas);

  hildon_window_set_app_menu(HILDON_WINDOW(window), main_menu);

#else 
  main_menu = gtk_menu_new();
  menu_view = gtk_menu_new();
  menu_zoom = gtk_menu_new();

  /* Create main menu items */
  item_save = gtk_menu_item_new_with_label(_("Save report..."));
  item_view = gtk_menu_item_new_with_label(_("View"));
  item_zoom = gtk_menu_item_new_with_label(_("Zoom"));
#if PLAINGTK == 0
  item_help = gtk_menu_item_new_with_label(_("Help"));
#endif
  item_exit = gtk_menu_item_new_with_label(_("Close window"));

  /* Create view menu items */
  pui->report_menu_item_fullscreen = gtk_check_menu_item_new_with_label(_("Full screen"));
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->report_menu_item_fullscreen),
				 pui->reportfullscreen);

  /* Create zoom menu items */
  item_zoom_in = gtk_menu_item_new_with_label(_("Zoom in"));
  item_zoom_out = gtk_menu_item_new_with_label(_("Zoom out"));

  /* Add menu items to right menus */
  /* Main menu */
  gtk_menu_append(main_menu, item_save);
  gtk_menu_append(main_menu, gtk_separator_menu_item_new());
  gtk_menu_append(main_menu, item_view);
  gtk_menu_append(main_menu, gtk_separator_menu_item_new());
  gtk_menu_append(main_menu, item_zoom);
  gtk_menu_append(main_menu, gtk_separator_menu_item_new());
#if PLAINGTK == 0
  gtk_menu_append(main_menu, item_help);
#endif
  gtk_menu_append(main_menu, item_exit);

  /* View menu */
  gtk_menu_append(menu_view, pui->report_menu_item_fullscreen);

  /* Zoom menu */
  gtk_menu_append(menu_zoom, item_zoom_in);
  gtk_menu_append(menu_zoom, item_zoom_out);

#if PLAINGTK == 1
  menubar = gtk_menu_bar_new();
  gtk_box_pack_start (GTK_BOX (root), menubar, FALSE, FALSE, 2);
  gtk_widget_show (menubar);
  item_file = gtk_menu_item_new_with_label ("fuelpad");
  gtk_widget_show (item_file);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (item_file), main_menu);
  gtk_menu_bar_append (GTK_MENU_BAR (menubar), item_file);
#else
  hildon_window_set_menu(HILDON_WINDOW(window), GTK_MENU(main_menu));
#endif
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item_view), menu_view);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item_zoom), menu_zoom);

  /* Attach the callback functions to the activate signal */
  g_signal_connect(G_OBJECT(item_save), "activate",
		   G_CALLBACK(callback_save_report), pui->reportcanvas);
  g_signal_connect(G_OBJECT(item_zoom_in), "activate",
		   G_CALLBACK(callback_report_zoom_in), pui->reportcanvas);
  g_signal_connect(G_OBJECT(item_zoom_out), "activate",
		   G_CALLBACK(callback_report_zoom_out), pui->reportcanvas);
#if LIBHELP == 1
  g_signal_connect(G_OBJECT(item_help), "activate",
		   G_CALLBACK(help_activated), HELP_ID_REPORT);
#endif
  g_signal_connect_swapped(G_OBJECT(item_exit), "activate",
			   G_CALLBACK(delete_reportwin), G_OBJECT(window));

  g_signal_connect(G_OBJECT(pui->report_menu_item_fullscreen), "toggled",
		   G_CALLBACK(cb_report_fullscreen), pui);
#endif
  /* Make all menu widgets visible */
  gtk_widget_show_all(GTK_WIDGET(main_menu));
}

gdouble report_init_line(gdouble y)
{
  return y;
}

/**
 * \fn gdouble report_line_down(gdouble y, gint height, 
			 gdouble numlines, gint page_height)
 * \brief Computes the y coordinate for the next line
 * \param y The current y coordinate
 * \param height The height of current font in points
 * \param numlines How many lines down
 * \param page_height Page height in the same unit as given to gtk_plot_canvas_new()
 * \return The new y coordinate
 *
 * This function calculates the y coordinate for the next line
 * numlines down.
 *
 */
gdouble report_line_down(gdouble y, gint height, 
			 gdouble numlines, gint page_height)
{
  return y+numlines*height/YSCALE(page_height);
}

/**
 * \fn GtkPlotCanvasChild *report_put_text(GtkWidget *canvas,
				    gdouble x1, gdouble y1,
				    const gchar *font, gint height,
				    GtkJustification justification,
				    const gchar *real_text)
 * \brief Display a text on a gtk_plot_canvas
 * \return A handle to the displayed text
 *
 * This is a convenience function for creating and adding a text to a gtk_plot_canvas.
 *
 */
GtkPlotCanvasChild *report_put_text(GtkWidget *canvas,
				    gdouble x1, gdouble y1,
				    const gchar *font, gint height,
				    GtkJustification justification,
				    const gchar *real_text)
{
  GtkPlotCanvasChild *child;

  child = gtk_plot_canvas_text_new(font, height, 0, NULL, NULL, TRUE,
				   justification,
				   real_text);
  gtk_plot_canvas_put_child(GTK_PLOT_CANVAS(canvas), child, x1, YSCALE(y1), 0, 0);

  return child;
}

/**
 * \fn void report_convert_to_iso8859_15(GtkWidget *canvas)
 * \brief Converts all the text to iso8859_15
 *
 * This function changes all text in canvas from UTF-8 to ISO-8859-15
 * since gtkextra does not handle UTF-8. This is needed in
 * printing/saving the canvas as postscript.
 *
 */
void report_convert_to_iso8859_15(GtkWidget *canvas)
{
  gchar *str;
  GList* pchild;
  GtkPlotCanvasText *apu;

  pchild=g_list_first(GTK_PLOT_CANVAS(canvas)->childs);

  for (;pchild != NULL;pchild=g_list_next(pchild)) {
    if (GTK_IS_PLOT_CANVAS_TEXT(pchild->data)) {
      str=g_convert(GTK_PLOT_CANVAS_TEXT(pchild->data)->text.text, -1, 
		    "ISO-8859-15", "UTF-8", NULL, NULL, NULL);
      apu=GTK_PLOT_CANVAS_TEXT(pchild->data);
      gtk_plot_canvas_text_set_attributes(apu, NULL, apu->text.height, 
					  apu->text.angle, 
					  NULL, NULL, 
					  apu->text.transparent,
					  apu->text.justification,
					  str);
      g_free(str);
    }
  }
}

void callback_save_report( GtkAction * action, gpointer data )
{
  gint pagesize;
  gchar *filename = NULL;
  filename = interface_file_chooser ("Save postscript file",
				     GTK_FILE_CHOOSER_ACTION_SAVE,
				     "ps",
				     HELP_ID_REPORT);

  if (filename == NULL) {
    filename = "NULL";
  }
  else {
    report_convert_to_iso8859_15(GTK_WIDGET(data));
    if (GTK_PLOT_CANVAS(data)->width == GTK_PLOT_A4_W)
      pagesize=GTK_PLOT_A4;
    else
      pagesize=GTK_PLOT_LETTER;
    gtk_plot_canvas_export_ps(GTK_PLOT_CANVAS(data), filename, GTK_PLOT_PORTRAIT, FALSE, pagesize);
    ui_show_banner(_("Report saved."));
  }
}

gdouble get_magnification(GtkWidget *canvas)
{
  return GTK_PLOT_CANVAS(canvas)->magnification;
}

void report_zoom(GtkWidget *canvas, zoom_t dir)
{
  static gdouble zoom=1.0;
  gdouble curmag;

  curmag = get_magnification(GTK_WIDGET(canvas));

  if (dir==ZOOMIN) {
    if (zoom<1.0) {
      zoom += 0.1;
      gtk_plot_canvas_set_magnification(GTK_PLOT_CANVAS(canvas), 1.1*curmag);
    }
    else
      ui_show_banner(_("Can't zoom in more"));
  }
  else {
    if (zoom>0.1) {
      zoom -= 0.1;
      gtk_plot_canvas_set_magnification(GTK_PLOT_CANVAS(canvas), curmag/1.1);
    }
    else
      ui_show_banner(_("Can't zoom out more"));
  }
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(canvas));
}

void callback_report_zoom_in( GtkAction * action, gpointer data )
{
  report_zoom(GTK_WIDGET(data), ZOOMIN);
}

void callback_report_zoom_out( GtkAction * action, gpointer data )
{
  report_zoom(GTK_WIDGET(data), ZOOMOUT);
}

/**
 * \fn void fill_year_combo(GtkWidget *combo)
 * \brief Fills the combo box with years contained in the database
 * \param *combo Pointer to a created combo box
 *
 * This callback reads the database, finds out which years contain
 * data for the current car and fills the combo box given as a
 * parameter with these years.
 *
 * \todo This should be changed so that fuelpad.c has a function which
 * is called to obtain the data (i.e. sqlite would not be used here)
 *
 */
void fill_year_combo(GtkWidget *combo)
{
  char listitemtext[ENTRYDATEMAX]="";
  gint i;

  i=0;

  if (SQLITE_OK == sqlite3_bind_int(ppStmtGetYears, 1, currentcar)) {
    while(SQLITE_ROW == sqlite3_step(ppStmtGetYears)) {
      g_snprintf(listitemtext, ENTRYDATEMAX, "%s", 
		 sqlite3_column_text(ppStmtGetYears,0));
#if MAEMO_VERSION_MAJOR == 5
      hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (combo),listitemtext);
#else
      gtk_combo_box_append_text(GTK_COMBO_BOX (combo),listitemtext);
#endif
      i++;
    }
    sqlite3_reset(ppStmtGetYears);
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR (combo), 0, i-1);
#else
    gtk_combo_box_set_active(GTK_COMBO_BOX (combo),i-1);
#endif
  }
}

GtkWidget *create_year_combo(AppUIData *pui)
{
  GtkWidget *combo;

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
#else
  combo = gtk_combo_box_new_text();
#endif
  fill_year_combo(combo);
#if MAEMO_VERSION_MAJOR == 5
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (pui->lastyearbutton),
                                     HILDON_TOUCH_SELECTOR (combo));
#endif
  return combo;
}

/**
 * \fn void update_yearcombo_cb(GtkWidget *widget, gpointer combo)
 * \brief Updates the already created combo box with years contained in the database
 * \param *combo Pointer to a created combo box
 *
 * This callback deletes the data in the combobox, reads the database,
 * finds out which years contain data for the current car and refills
 * the combo box given as a parameter with these years.
 *
 */
void update_yearcombo_cb(GtkWidget *widget, gpointer combo)
{
#if MAEMO_VERSION_MAJOR == 5
  HildonTouchSelectorColumn *column;
#endif
  GtkTreeModel *model;
  gint i, count;

#if MAEMO_VERSION_MAJOR == 5
  model = hildon_touch_selector_get_model(HILDON_TOUCH_SELECTOR(combo), 0);
  gtk_list_store_clear(GTK_LIST_STORE(model));
/*   hildon_touch_selector_remove_column (HILDON_TOUCH_SELECTOR(combo), 0);   */
/* /\*   combo = hildon_touch_selector_new_text (); *\/ */
/*   column = hildon_touch_selector_append_text_column (HILDON_TOUCH_SELECTOR(combo), */
/* 						     model, */
/* 						     TRUE); */
/*   hildon_touch_selector_column_set_text_column(column, 0); */
#else
  model = gtk_combo_box_get_model(GTK_COMBO_BOX(combo));
  count = gtk_tree_model_iter_n_children(model, NULL);
  for (i=0; i < count; i++)
    gtk_combo_box_remove_text(GTK_COMBO_BOX(combo), 0);
#endif
  fill_year_combo(GTK_WIDGET(combo));
}

#if PLAINGTK == 1
static void create_report_toolbar(GtkWidget *window, GtkWidget *vbox, GtkWidget *canvas)
#else
static void create_report_toolbar(HildonWindow *window, GtkWidget *canvas)
#endif
{
  GtkWidget *toolbar;
  GtkToolItem *tb_save;
  GtkToolItem *tb_zoom_in;
  GtkToolItem *tb_zoom_out;
  GtkToolItem *tb_close;

  /* Create toolbar */
  toolbar = gtk_toolbar_new();

  /* Create toolbar button items */
  tb_save = gtk_tool_button_new_from_stock(GTK_STOCK_SAVE);
  tb_zoom_in = gtk_tool_button_new_from_stock(GTK_STOCK_ZOOM_IN);
  tb_zoom_out = gtk_tool_button_new_from_stock(GTK_STOCK_ZOOM_OUT);
  tb_close = gtk_tool_button_new_from_stock(GTK_STOCK_CLOSE);

  gtk_tool_item_set_expand(GTK_TOOL_ITEM(tb_save),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(tb_zoom_in),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(tb_zoom_out),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(tb_close),TRUE);
   
  /* Add all items to toolbar */
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), tb_save, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), tb_zoom_in, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), tb_zoom_out, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), tb_close, -1);

  g_signal_connect(G_OBJECT(tb_save), "clicked",
		   G_CALLBACK(callback_save_report), canvas);
  g_signal_connect(G_OBJECT(tb_zoom_in), "clicked",
		   G_CALLBACK(callback_report_zoom_in), canvas);
  g_signal_connect(G_OBJECT(tb_zoom_out), "clicked",
		   G_CALLBACK(callback_report_zoom_out), canvas);
  g_signal_connect_swapped(G_OBJECT(tb_close), "clicked",
			   G_CALLBACK(gtk_widget_destroy), G_OBJECT(window));

  /* Add toolbar HildonWindow */
#if PLAINGTK == 1
  gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(toolbar), FALSE, FALSE, 5);
#else
  hildon_window_add_toolbar(window, GTK_TOOLBAR(toolbar));
#endif
}

/**
 * \fn gchar *report_conv_cost_label(const char *s)
 * \brief Change a text 
 * \param *s The text to be changed
 * \return Text changed to take the unit account
 *
 * The text to be changed must be one of TOTALCO2TAG, CO2TAG,
 * CONSUMETAG, COSTPERKMTAG, COSTTAG OR PRICEPERUNITTAG. These are
 * changed so that the selected unit is taken into account.
 *
 */
gchar *report_conv_cost_label(const char *s)
{
  if (strncmp(PRICEPERUNITTAG,s,strlen(PRICEPERUNITTAG))==0) {
    if (curunit.volumeunit==SI)
      return g_strdup(_("Avg. price/litre"));
  else
      return g_strdup(_("Avg. price/gallon"));
  }
  if (strncmp(CONSUMETAG,s,strlen(CONSUMETAG))==0) {
    if (curunit.consumeunit==SI)
      return g_strdup(_("Avg. l/100 km"));
  else
      return g_strdup(_("Avg. miles/gallon"));
  }
  if (strncmp(CO2TAG,s,strlen(CO2TAG))==0) {
    if (curunit.massunit==SI) {
      if (curunit.lengthunit==SI)
	return g_strconcat(_("Avg. CO2"), " ", (curunit.lengthunit==SI)? 
			   _("g/km") : _("g/mile"), NULL);
    }
    else
      return g_strconcat(_("Avg. CO2"), " ", (curunit.lengthunit==SI)? 
			 _("lb/100 km") : _("lb/100 miles"), NULL);
  }
  if (strncmp(TOTALCO2TAG,s,strlen(TOTALCO2TAG))==0) {
    if (curunit.massunit==SI) {
      if (curunit.lengthunit==SI)
	return g_strconcat(_("Tot. CO2"), " ",  _("kg"), NULL);
    }
    else
      return g_strconcat(_("Tot. CO2"), " ",_("lb"), NULL);
  }
  if (strncmp(COSTTAG,s,strlen(COSTTAG))==0)
    return g_strjoin(" ",_("Cost"), curunit.currency->str, NULL);
  if (strncmp(COSTPERKMTAG,s,strlen(COSTPERKMTAG))==0)
    return g_strconcat(_("Cost"), " ", curunit.currency->str, "/",
		       (curunit.lengthunit==SI)? _("km") : _("mile"), 
		       NULL);
  else
    return g_strdup(s);
}

void report_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GtkWidget *vbox1;
  gint page_width, page_height;
  gfloat scale;
  gdouble texty;
  gdouble textx;

  gchar *cartexts[] = {gettext_noop("Car"),
			  gettext_noop("Car"),
			  gettext_noop("Model year"),
			  gettext_noop("Register"});
  gchar *costtexts[] = {gettext_noop("Overall expenses"),
			  gettext_noop("Fuel"), 
			  gettext_noop("Oil"),
			  gettext_noop("Service"), 
			  gettext_noop("Tires"),
			  gettext_noop("Total cost")};
  gchar *kmtexts[] = {gettext_noop("Distance"), 
			gettext_noop("Beginning of year"), 
			gettext_noop("End of year"), 
			gettext_noop("Overall distance")};
  gchar *fueltexts[] = {gettext_noop("Fuel and emissions"), 
			  gettext_noop("Total consumption"), 
			  COSTTAG, 
			  CONSUMETAG,
			  CO2TAG,
			  TOTALCO2TAG,
			  PRICEPERUNITTAG,
			  COSTPERKMTAG};
  gchar *oiltexts[] = {gettext_noop("Oil"),
			 COSTTAG, 
			 COSTPERKMTAG};
  gchar *servicetexts[] = {gettext_noop("Service"), 
			     COSTTAG, 
			     COSTPERKMTAG};
  gchar *tirestexts[] = {gettext_noop("Tires"), 
			     COSTTAG, 
			     COSTPERKMTAG};
  gint i;

  gint lastyear;
  gchar *lastyearbuf;
  gchar car[30];
  gchar modelyear[30];
  gchar reg[30];
  gchar year[ENTRYDATEMAX];
  gchar startkm[ENTRYKMMAX];
  gchar endkm[ENTRYKMMAX];
  gchar yearkm[ENTRYKMMAX];
  gchar sumfill[ENTRYKMMAX]; 
  gchar sumprice[ENTRYKMMAX];
  gchar avgconsum[ENTRYKMMAX];
  gchar avgco2[ENTRYKMMAX];
  gchar avgprice[ENTRYKMMAX];
  gchar totalco2[ENTRYKMMAX];
  gchar pricekm[ENTRYKMMAX];
  gchar priceoil[ENTRYKMMAX];
  gchar priceoilkm[ENTRYKMMAX];
  gchar priceservice[ENTRYKMMAX];
  gchar priceservicekm[ENTRYKMMAX];
  gchar pricetires[ENTRYKMMAX];
  gchar pricetireskm[ENTRYKMMAX];
  gchar totalprice[ENTRYDATEMAX];

 if (arg1 == GTK_RESPONSE_ACCEPT) {

#if MAEMO_VERSION_MAJOR == 5
/*    lastyearbuf=hildon_button_get_value(HILDON_BUTTON(pui->lastyearbutton)); */
   lastyearbuf=hildon_touch_selector_get_current_text(HILDON_TOUCH_SELECTOR(pui->yearcombo));
#else
   lastyearbuf = gtk_combo_box_get_active_text(GTK_COMBO_BOX(pui->yearcombo));
#endif
   lastyear=atoi(lastyearbuf);
#if MAEMO_VERSION_MAJOR == 5
#else
   g_free(lastyearbuf);
#endif

/*    if (gtk_combo_box_get_active(GTK_COMBO_BOX(pui->papercombo))==0) { */
   if (reportpaper == A4PAPER) {
     page_width = GTK_PLOT_A4_W;
     page_height = GTK_PLOT_A4_H;
#if MAEMO_VERSION_MAJOR == 5
     scale = 800.0/GTK_PLOT_A4_W;
#else
     scale = 670.0/GTK_PLOT_A4_W;
#endif
   }
   else {
     page_width = GTK_PLOT_LETTER_W;
     page_height = GTK_PLOT_LETTER_H;
#if MAEMO_VERSION_MAJOR == 5
     scale = 800.0/GTK_PLOT_LETTER_W;
#else
     scale = 670.0/GTK_PLOT_LETTER_W;
#endif
   }

   /* Destroy the report settings dialog */
   gtk_widget_destroy(GTK_WIDGET(dialog));

   if (carchanged)
     update_car_changed(pui);

#if PLAINGTK == 1
   pui->app->reportwindow=gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title(GTK_WINDOW(pui->app->reportwindow), "Report");
   gtk_widget_set_usize(pui->app->reportwindow,800,600);
   gtk_container_border_width(GTK_CONTAINER(pui->app->reportwindow),0);
#else
#if MAEMO_VERSION_MAJOR == 5
  pui->app->reportwindow = hildon_stackable_window_new();
#else
   pui->app->reportwindow = HILDON_WINDOW(hildon_window_new());
#endif
   hildon_program_add_window(pui->app->program, pui->app->reportwindow);

  /* Callback for hardware keys */
  g_signal_connect(G_OBJECT(pui->app->reportwindow), "key_press_event",
		   G_CALLBACK(report_key_press_cb), pui);
#endif

  g_signal_connect (G_OBJECT(pui->app->reportwindow), "destroy",
		    G_CALLBACK (destroy_reportwin), pui);

   vbox1=gtk_vbox_new(FALSE,0);
   gtk_container_add(GTK_CONTAINER(pui->app->reportwindow),vbox1);
   gtk_widget_show(vbox1);

   pui->reportcanvas = gtk_plot_canvas_new(page_width, page_height, scale);

#if PLAINGTK == 1
   create_reportwin_menu(pui->app->reportwindow, vbox1, pui);
#else
   create_reportwin_menu(pui->app->reportwindow, pui);
#endif


#if MAEMO_VERSION_MAJOR == 5
#else
#if PLAINGTK == 1
   create_report_toolbar(pui->app->reportwindow, vbox1, pui->reportcanvas);
#else
   create_report_toolbar(pui->app->reportwindow, pui->reportcanvas);
#endif
#endif

#if MAEMO_VERSION_MAJOR == 5
  pui->reportscrollwin = hildon_pannable_area_new();
  g_object_set(G_OBJECT(pui->reportscrollwin), "mov-mode", HILDON_MOVEMENT_MODE_BOTH, NULL);
#else
   pui->reportscrollwin=gtk_scrolled_window_new(NULL, NULL);
   gtk_container_border_width(GTK_CONTAINER(pui->reportscrollwin),0);
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(pui->reportscrollwin),
				  GTK_POLICY_ALWAYS,GTK_POLICY_ALWAYS);
#endif
   gtk_box_pack_start(GTK_BOX(vbox1),pui->reportscrollwin, TRUE, TRUE,0);
   gtk_widget_show(pui->reportscrollwin);

#if MAEMO_VERSION_MAJOR == 5
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(pui->reportscrollwin), pui->reportcanvas);
#else
   gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pui->reportscrollwin), pui->reportcanvas);
#endif

   gtk_widget_show(pui->reportcanvas);

   texty=report_init_line(HEADERLINE);
   report_put_text(pui->reportcanvas, .5, texty, "Helvetica-Bold", HEADING1SIZE, 
		   GTK_JUSTIFY_CENTER,
		   _("Car expenses"));
   texty=report_line_down(texty, HEADING1SIZE, 3.0, page_height);

   report_put_text(pui->reportcanvas, LEFTMARGIN, texty, "Helvetica-Bold", 
		   HEADING2SIZE, GTK_JUSTIFY_LEFT, gettext(cartexts[0]));
   texty=report_line_down(texty, HEADING2SIZE, 2.0, page_height);

   if (SQLITE_OK == sqlite3_bind_int(ppStmtCurCar, 1, currentcar)) {
     if (SQLITE_ROW == sqlite3_step(ppStmtCurCar)) {
       g_snprintf(car, 30, "%s %s", 
		  sqlite3_column_text(ppStmtCurCar, 0),
		  sqlite3_column_text(ppStmtCurCar, 1));
       g_snprintf(reg, 30, "%s", sqlite3_column_text(ppStmtCurCar, 2));
       g_snprintf(modelyear, 30, "%s", sqlite3_column_text(ppStmtCurCar, 3));
     }
     sqlite3_reset(ppStmtCurCar);
   }
   for (i=1;i<sizeof(cartexts)/sizeof(char *);i++) {
     report_put_text(pui->reportcanvas, LEFTMARGIN+INDENT, texty, "Helvetica", NORMALSIZE, 
		     GTK_JUSTIFY_LEFT,
		     gettext(cartexts[i]));
     texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
   }

   texty=report_line_down(texty, HEADING2SIZE, 1.0, page_height);
   report_put_text(pui->reportcanvas,
		   LEFTMARGIN, texty, "Helvetica-Bold", HEADING2SIZE, GTK_JUSTIFY_LEFT,
		   gettext(kmtexts[0]));
   texty=report_line_down(texty, HEADING2SIZE, 2.0, page_height);

   for (i=1;i<sizeof(kmtexts)/sizeof(char *);i++) {
     char *str;
     str=report_conv_cost_label(gettext(kmtexts[i]));
     report_put_text(pui->reportcanvas, LEFTMARGIN+INDENT, texty, "Helvetica", NORMALSIZE, GTK_JUSTIFY_LEFT,
		     str);
     g_free(str);
     texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
   }

   texty=report_line_down(texty, HEADING2SIZE, 1.0, page_height);
   report_put_text(pui->reportcanvas,
		   LEFTMARGIN, texty, "Helvetica-Bold", HEADING2SIZE, GTK_JUSTIFY_LEFT,
		   gettext(fueltexts[0]));
   texty=report_line_down(texty, HEADING2SIZE, 2.0, page_height);

   for (i=1;i<sizeof(fueltexts)/sizeof(char *);i++) {
     char *str;
     str=report_conv_cost_label(gettext(fueltexts[i]));
     report_put_text(pui->reportcanvas,
		     LEFTMARGIN+INDENT, texty, "Helvetica", NORMALSIZE, GTK_JUSTIFY_LEFT,
		     str);
     g_free(str);
     texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
   }

   texty=report_line_down(texty, HEADING2SIZE, 1.0, page_height);
   report_put_text(pui->reportcanvas,
		   LEFTMARGIN, texty, "Helvetica-Bold", HEADING2SIZE, GTK_JUSTIFY_LEFT,
		   gettext(oiltexts[0]));
   texty=report_line_down(texty, HEADING2SIZE, 2.0, page_height);

   for (i=1;i<sizeof(oiltexts)/sizeof(char *);i++) {
     char *str;
     str=report_conv_cost_label(gettext(oiltexts[i]));
     g_print("%s\n",str);
     report_put_text(pui->reportcanvas,
		     LEFTMARGIN+INDENT, texty, "Helvetica", NORMALSIZE, GTK_JUSTIFY_LEFT,
		     str);
     g_free(str);
     texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
   }

   texty=report_line_down(texty, HEADING2SIZE, 1.0, page_height);
   report_put_text(pui->reportcanvas,
		   LEFTMARGIN, texty, "Helvetica-Bold", HEADING2SIZE, GTK_JUSTIFY_LEFT,
		   gettext(servicetexts[0]));
   texty=report_line_down(texty, HEADING2SIZE, 2.0, page_height);

   for (i=1;i<sizeof(servicetexts)/sizeof(char *);i++) {
     char *str;
     str=report_conv_cost_label(gettext(servicetexts[i]));
     report_put_text(pui->reportcanvas,
		     LEFTMARGIN+INDENT, texty, "Helvetica", NORMALSIZE, GTK_JUSTIFY_LEFT,
		     str);
     g_free(str);
     texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
   }

   texty=report_line_down(texty, HEADING2SIZE, 1.0, page_height);
   report_put_text(pui->reportcanvas,
		   LEFTMARGIN, texty, "Helvetica-Bold", HEADING2SIZE, GTK_JUSTIFY_LEFT,
		   gettext(tirestexts[0]));
   texty=report_line_down(texty, HEADING2SIZE, 2.0, page_height);

   for (i=1;i<sizeof(tirestexts)/sizeof(char *);i++) {
     char *str;
     str=report_conv_cost_label(gettext(tirestexts[i]));
     report_put_text(pui->reportcanvas,
		     LEFTMARGIN+INDENT, texty, "Helvetica", NORMALSIZE, GTK_JUSTIFY_LEFT,
		     str);
     g_free(str);
     texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
   }

   texty=report_line_down(texty, HEADING2SIZE, 1.0, page_height);
   report_put_text(pui->reportcanvas,
		   LEFTMARGIN, texty, "Helvetica-Bold", HEADING2SIZE, GTK_JUSTIFY_LEFT,
		   gettext(costtexts[0]));
   texty=report_line_down(texty, HEADING2SIZE, 2.0, page_height);

   for (i=1;i<sizeof(costtexts)/sizeof(char *);i++) {
     char *str;
     str=report_conv_cost_label(gettext(costtexts[i]));
     report_put_text(pui->reportcanvas,
		     LEFTMARGIN+INDENT, texty, "Helvetica", NORMALSIZE, GTK_JUSTIFY_LEFT,
		     str);
     g_free(str);
     texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
   }

   texty=report_line_down(texty, HEADING2SIZE, 1.0, page_height);

   texty=report_init_line(HEADERLINE);
   texty=report_line_down(texty, HEADING1SIZE, 3.0, page_height);
   texty=report_line_down(texty, HEADING2SIZE, 2.0, page_height);
   report_put_text(pui->reportcanvas, FIRSTCOLUMN, texty, "Helvetica", NORMALSIZE,
		   GTK_JUSTIFY_LEFT,
		   car);
   texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
   report_put_text(pui->reportcanvas, FIRSTCOLUMN, texty, "Helvetica", NORMALSIZE,
		   GTK_JUSTIFY_LEFT,
		   modelyear);
   texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
   report_put_text(pui->reportcanvas, FIRSTCOLUMN, texty, "Helvetica", NORMALSIZE,
		   GTK_JUSTIFY_LEFT,
		   reg);
   texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);

   /* Fill the column with data */
   if (SQLITE_OK == sqlite3_bind_int(ppStmtGetReport, 1, currentcar)) {
     textx = FIRSTCOLUMN;
     while(SQLITE_ROW == sqlite3_step(ppStmtGetReport)) {
       g_snprintf(year, ENTRYDATEMAX, "%s",
		  sqlite3_column_text(ppStmtGetReport,0));
       if (atoi(year) > lastyear) break;
       if (atoi(year) > lastyear-REPORTYEARS) {
	 g_snprintf(startkm, ENTRYKMMAX, "%.0f",
		    SIlength2user(sqlite3_column_double(ppStmtGetReport,1)));
	 g_snprintf(endkm, ENTRYKMMAX, "%.0f",
		    SIlength2user(sqlite3_column_double(ppStmtGetReport,2)));
	 g_snprintf(yearkm, ENTRYKMMAX, "%.0f",
		    SIlength2user(sqlite3_column_double(ppStmtGetReport,3)));
	 g_snprintf(sumfill, ENTRYKMMAX, "%.1f",
		    SIvolume2user(sqlite3_column_double(ppStmtGetReport,4)));
	 g_snprintf(sumprice, ENTRYKMMAX, "%.1f",
		    sqlite3_column_double(ppStmtGetReport,5));
	 g_snprintf(avgconsum, ENTRYKMMAX, "%.1f",
		    SIconsumption2user(sqlite3_column_double(ppStmtGetReport,6)));
	 g_snprintf(avgco2, ENTRYKMMAX, "%.0f",
		    SIemission2user(sqlite3_column_double(ppStmtGetReport,6)*
				    get_emission_per_litre(db_get_fueltype(currentcar))/100));
	 g_snprintf(totalco2, ENTRYKMMAX, "%.0f",
		    SIemission2user(sqlite3_column_double(ppStmtGetReport,4)*
				    get_emission_per_litre(db_get_fueltype(currentcar))/1000));
	 g_snprintf(avgprice, ENTRYKMMAX, "%.2f",
		    SIppl2user(sqlite3_column_double(ppStmtGetReport,7)));
	 g_snprintf(pricekm, ENTRYKMMAX, "%.2f",
		    user2SIlength(sqlite3_column_double(ppStmtGetReport,8)));
	 g_snprintf(priceoil, ENTRYKMMAX, "%.2f",
		    sqlite3_column_double(ppStmtGetReport,9));
	 g_snprintf(priceoilkm, ENTRYKMMAX, "%.2f",
		    user2SIlength(sqlite3_column_double(ppStmtGetReport,10)));
	 g_snprintf(priceservice, ENTRYKMMAX, "%.2f",
		    sqlite3_column_double(ppStmtGetReport,11));
	 g_snprintf(priceservicekm, ENTRYKMMAX, "%.2f",
		    user2SIlength(sqlite3_column_double(ppStmtGetReport,12)));
	 g_snprintf(pricetires, ENTRYKMMAX, "%.2f",
		    sqlite3_column_double(ppStmtGetReport,13));
	 g_snprintf(pricetireskm, ENTRYKMMAX, "%.2f",
		    user2SIlength(sqlite3_column_double(ppStmtGetReport,14)));

	 /* Year header*/
	 texty=report_init_line(HEADERLINE);
	 texty=report_line_down(texty, HEADING1SIZE, 3.0, page_height);
	 texty=report_line_down(texty, HEADING2SIZE, 2.0, page_height);
	 texty=report_line_down(texty, NORMALSIZE, ((sizeof(cartexts)/sizeof(char *))-1)*1.2, page_height);
	 texty=report_line_down(texty, HEADING2SIZE, 1.0, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, year);
	 texty=report_line_down(texty, HEADING2SIZE, 2.0, page_height);

	 /* Km */
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, startkm);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, endkm);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, yearkm);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);

	 texty=report_line_down(texty, HEADING2SIZE, 1.0, page_height);
	 texty=report_line_down(texty, HEADING2SIZE, 2.0, page_height);

	 /* Fuel */
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, sumfill);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, sumprice);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, avgconsum);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, avgco2);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, totalco2);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, avgprice);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, pricekm);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);

	 texty=report_line_down(texty, HEADING2SIZE, 1.0, page_height);
	 texty=report_line_down(texty, HEADING2SIZE, 2.0, page_height);

	 /* Oil */
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, priceoil);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, priceoilkm);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);

	 texty=report_line_down(texty, HEADING2SIZE, 1.0, page_height);
	 texty=report_line_down(texty, HEADING2SIZE, 2.0, page_height);

	 /* Service */
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, priceservice);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, priceservicekm);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);

	 texty=report_line_down(texty, HEADING2SIZE, 1.0, page_height);
	 texty=report_line_down(texty, HEADING2SIZE, 2.0, page_height);

	 /* Tires */
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, pricetires);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, pricetireskm);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);

	 texty=report_line_down(texty, HEADING2SIZE, 1.0, page_height);
	 texty=report_line_down(texty, HEADING2SIZE, 2.0, page_height);

	 /* Overall expenses */
	 g_snprintf(totalprice, ENTRYDATEMAX, "%.2f", 
		    atof(sumprice)+atof(priceoil)+atof(priceservice)
		    +atof(pricetires));

	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, sumprice);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, priceoil);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, priceservice);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, pricetires);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);
	 report_put_text(pui->reportcanvas, textx, texty, "Helvetica", NORMALSIZE, 
			 GTK_JUSTIFY_LEFT, totalprice);
	 texty=report_line_down(texty, NORMALSIZE, 1.2, page_height);


	 /* Next column*/
	 textx += COLUMNSEP;
       }
     }
     sqlite3_reset(ppStmtGetReport);
   }
   else
     g_print("Something wrong in obtaining the report data\n");


   gtk_widget_show_all(GTK_WIDGET(pui->app->reportwindow));
   if (pui->mainfullscreen)
     gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->report_menu_item_fullscreen),
				    !pui->reportfullscreen);
 }
 else {
   /* Destroy the report settings dialog */
   gtk_widget_destroy(GTK_WIDGET(dialog));
 }
}

void callback_report( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *button;

  if (pui->app->reportwindow != NULL) {
    gtk_window_present(GTK_WINDOW(pui->app->reportwindow));
    return;
  }

  if (!db_open()) {
    dialog = gtk_dialog_new_with_buttons(_("Report"),
					 GTK_WINDOW(pui->app->mainwindow),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK,
					 GTK_RESPONSE_REJECT,
					 NULL);
    vbox = GTK_WIDGET(GTK_DIALOG(dialog)->vbox);

    label=gtk_label_new(_("Can't access database - reporting not possible"));
    gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 5);
    gtk_widget_show(label);
  }
  else {
    dialog = gtk_dialog_new_with_buttons(_("Report"),
					 GTK_WINDOW(pui->app->mainwindow),
					 GTK_DIALOG_MODAL,
					 _("Generate"), GTK_RESPONSE_ACCEPT,
					 GTK_STOCK_CANCEL,
					 GTK_RESPONSE_REJECT,
					 NULL);
    vbox = GTK_WIDGET(GTK_DIALOG(dialog)->vbox);

    table = gtk_table_new(2, 3, FALSE);
    gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 5);

#if MAEMO_VERSION_MAJOR == 5
  pui->carbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->carbutton), _("Car"));
  gtk_table_attach(GTK_TABLE(table), pui->carbutton, 0, 3, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->carbutton);
#else
    label=gtk_label_new(_("Car:"));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);
#endif

#if MAEMO_VERSION_MAJOR == 5
    pui->carcombo = create_car_combo(pui, pui->carbutton);
#else
    pui->carcombo = create_car_combo(pui);
    gtk_table_attach(GTK_TABLE(table), pui->carcombo, 1, 2, 0, 1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->carcombo);
#endif

#if MAEMO_VERSION_MAJOR == 5
    pui->lastyearbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    hildon_button_set_title (HILDON_BUTTON (pui->lastyearbutton), _("Last year"));
    gtk_table_attach(GTK_TABLE(table), pui->lastyearbutton, 0, 2, 1, 2, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->lastyearbutton);
#else
    label=gtk_label_new(_("Last year:"));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);
#endif

    pui->yearcombo = create_year_combo(pui);
#if MAEMO_VERSION_MAJOR == 5
#else
    gtk_table_attach(GTK_TABLE(table), pui->yearcombo, 1, 2, 1, 2,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->yearcombo);
#endif

#if MAEMO_VERSION_MAJOR == 5
    button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
    gtk_button_set_label(GTK_BUTTON(button), _("Update"));
#else
    button = gtk_button_new_with_label(_("Update"));
#endif
    gtk_table_attach(GTK_TABLE(table), button, 2, 3, 1, 2,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(button);
    g_signal_connect (GTK_OBJECT (button), "clicked",
			G_CALLBACK(update_yearcombo_cb), pui->yearcombo);

    gtk_widget_show(table);

  }

#if LIBHELP == 1
  help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_REPORT,
				 pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (report_response),
		    pui);
  gtk_widget_show(dialog);

}
