/**
 * Copyright (C) 2008-09 Tan Miaoqing
 * Contact: Tan Miaoqing <rabbitrun84@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <glib.h>

#include "url-parser.h"

static gboolean
setup_regex (GRegex **combo_regex)
{
  GRegex *new_combo_regex;
  GError *error = NULL;

#define URICHAR "([-._~[:alnum:]!$&*+;=:@/?#]|%[[:xdigit:]]{2})"

  /* URL regexp */
  static const gchar url_regex_str[] = "(https?|rtsp|mms|webcal)://" URICHAR "+";

#define DOMAIN_LABEL "[[:alnum:]]([-[:alnum:]]*[[:alnum:]])?"
#define DOMAIN_NAME DOMAIN_LABEL "(\\." DOMAIN_LABEL ")*"

  /* www.* host regexp */
  static const gchar wwwhost_regex_str[] = "\\bwww\\." DOMAIN_NAME;

  gchar *combo_regex_str = g_strdup_printf (
      "(?<BR>\\R)|"
      "(?<URL>%s)|"
      "(?<WWWHOST>%s)",
      url_regex_str,
      wwwhost_regex_str);

  new_combo_regex = g_regex_new(combo_regex_str,
                                G_REGEX_OPTIMIZE | G_REGEX_CASELESS,
                                0,
                                &error);
  g_free (combo_regex_str);

  if (error) {
    g_clear_error (&error);
    return FALSE;
  }

  if (*combo_regex)
    g_regex_unref (*combo_regex);
  *combo_regex = new_combo_regex;

  return TRUE;

#undef DOMAIN_NAME
#undef DOMAIN_LABEL
#undef URICHAR
}

static void
markup_string_append_link (GString *str,
                           const gchar *url,
                           const gchar *text)
{
    gchar *escaped;

    escaped = g_markup_printf_escaped (
        "<a href=\"%s\">%s</a>", url, text);
    g_string_append (str, escaped);
    g_free (escaped);
}

void
url_parser_process_message (const char *message,
                            char       **html_out)
{
  static GRegex *regex;
  GMatchInfo *match_info;
  GString *result_string;
  gint previous_start = 0;
  gint previous_end = 0;
  GError *error = NULL;

  g_return_if_fail (message != NULL);
  g_return_if_fail (html_out != NULL);

  if (!regex) {
    if (!setup_regex (&regex))
      return ;
  }

  result_string = g_string_new ("");

  g_regex_match_full (regex, message, -1, 0, 0, &match_info, &error);

  if (error)
    g_warning ("%s: g_regex_match_full: %s", G_STRFUNC, error->message);

  while (!error && g_match_info_matches (match_info)) {
    gint start = 0;
    gint end = 0;

    g_match_info_fetch_pos (match_info, 0, &start, &end);

    if (start > previous_end) {
      gchar *text;
      gchar *escaped;

      text = g_strndup (message+previous_end, start-previous_end);
      escaped = g_markup_escape_text (text, -1);
      g_string_append (result_string, escaped);

      g_free (escaped);
      g_free (text);
    }

    previous_start = start;
    previous_end = end;

    if (g_match_info_fetch_named_pos (
          match_info, "BR", &start, &end) &&
        start >= 0) {
          g_string_append (result_string, "<br/>\n");
    }
    else if (g_match_info_fetch_named_pos (
               match_info, "URL", &start, &end) &&
             start >= 0) {
          gchar *url;

          url = g_match_info_fetch_named (match_info, "URL");
          markup_string_append_link (result_string, url, url);
          g_free (url);
    }
    else if (g_match_info_fetch_named_pos (
               match_info, "WWWHOST", &start, &end) &&
             start >= 0) {
          gchar *url;
          gchar *link;

          link = g_match_info_fetch_named (match_info, "WWWHOST");
          url = g_strconcat ("http://", link, NULL);

          markup_string_append_link (result_string, url, link);

          g_free (url);
          g_free (link);
    }

    g_match_info_next (match_info, &error);
    if (error)
      g_warning ("%s: g_match_info_next: %s", G_STRFUNC, error->message);
  }

  if (error) {
    g_clear_error (&error);
  }

  g_match_info_free (match_info);

  if (message[previous_end]) {
    const gchar *text = message + previous_end;
    gchar *escaped = g_markup_escape_text (text, -1);

    g_string_append (result_string, escaped);
    g_free (escaped);
  }

  *html_out = g_string_free (result_string, FALSE);
}
