/**
 * Copyright (C) 2008-09 Tan Miaoqing
 * Contact: Tan Miaoqing <rabbitrun84@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <glib-object.h>
#include <hildon/hildon-program.h>
#include <hildon/hildon-file-chooser-dialog.h>
#include <hildon-mime.h>
#include <libosso.h>
#include <osso-log.h> /* OSSO Debug macros */
#include <rtcom-eventlogger/eventlogger.h>
#include <rtcom-eventlogger-ui/rtcom-log-columns.h>

#include <libosso-abook/osso-abook.h>
#include <libosso-abook/osso-abook-contact-chooser.h>
#include <libosso-abook/osso-abook-contact.h>
#include <libosso-abook/osso-abook-contact-model.h>

#include "tp-status-feed.h"
#include "tp-status-feed-private.h"
#include "timeline-view.h"
#include "contact-status-view.h"
#include "contact-history-view.h"
#include "delete-status-window.h"
#include "eventlogger-util.h"

G_DEFINE_TYPE (TpStatusFeed, tp_status_feed, G_TYPE_OBJECT);

static GtkWidget *create_main_status_window (TpStatusFeed *statusfeed);
static GtkWidget *create_contact_status_window (TpStatusFeed *statusfeed);

static void show_friend_status (GtkWidget *widget, gpointer data);
static void show_my_status (GtkWidget *widget, gpointer data);

/**************************************/
/* Private functions                  */
/**************************************/

/******** Menu ********/

/************* main status menu **************/

static void
_friends_view_deleted_cb (DeleteStatusWindow *del_window,
                          gpointer data)
{
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (data);

  /* refresh friends view */
  timeline_view_refresh (
      TIMELINE_VIEW (priv->main_status.friend_view));

  /* disconnect the "deleted" signal */
  g_signal_handlers_disconnect_by_func (del_window,
      _friends_view_deleted_cb, data);
}

static void
menu_delete_cb (GtkWidget *widget,
                gpointer data)
{
  TpStatusFeed *statusfeed = TP_STATUS_FEED (data);
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);

  g_debug ("%s: %s", G_STRFUNC, priv->friends_flag ? "Friend":"Me");

  GtkWidget *del_window = g_object_ref_sink (
      delete_status_window_new (priv->aggregator));

  if (priv->friends_flag) {
    delete_status_window_populate_friends_status (
        DELETE_STATUS_WINDOW (del_window));

    /* Only need to refresh friends view after deleting statuses
     * when it's grouped by contact */
    if (priv->main_status.group_by == RTCOM_EL_QUERY_GROUP_BY_CONTACT) {
      g_signal_connect (DELETE_STATUS_WINDOW (del_window), "deleted",
          G_CALLBACK (_friends_view_deleted_cb), statusfeed);
    }
  }
  else
    delete_status_window_populate_my_status (
            DELETE_STATUS_WINDOW (del_window));

  hildon_program_add_window (priv->program,
      HILDON_WINDOW (del_window));
}

#if 0
static void
menu_settings_cb (GtkWidget *widget,
                  gpointer data)
{
}
#endif

static void
menu_accounts_cb (GtkWidget *widget,
                  gpointer data)
{
  TpStatusFeed *statusfeed = TP_STATUS_FEED (data);
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);

  osso_cp_plugin_execute (priv->osso_context, ACCOUNTS_CP_PLUGIN_NAME,
                          priv->main_status.window, TRUE);
}

static void
backup_dialog_response_cb (GtkDialog *dialog,
                           gint response,
                           gpointer data)
{
  if (response == GTK_RESPONSE_OK) {
    DBusConnection *connection;
    gchar *filename;
    gchar *dir;

    filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
    dir = g_build_filename (g_get_home_dir (), STORAGE_DIR, NULL);

    if (g_str_has_prefix (filename, dir)) {

      /* Get system bus */
      connection = dbus_bus_get (DBUS_BUS_SESSION, NULL);

      if (connection) {
        hildon_mime_open_file (connection, filename);
        dbus_connection_unref (connection);
      }
      else {
        hildon_banner_show_information (
            GTK_WIDGET (dialog),
            NULL, "File open failed");
      }
    }
    else {
      hildon_banner_show_information (
          GTK_WIDGET (dialog),
          NULL, "Wrong directory");
    }

    g_free (filename);
    g_free (dir);
  }
}

static void
menu_backup_cb (GtkWidget *widget,
                gpointer data)
{
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (data);
  GtkWidget *dialog;
  gchar *dir;

  dir = g_build_filename (g_get_home_dir (),
      STORAGE_DIR, NULL);

  if (g_file_test (dir, G_FILE_TEST_EXISTS) == FALSE) {
    hildon_banner_show_information (
        GTK_WIDGET (priv->main_status.window),
        NULL, "Statuses will be backed up after you close FriendStatus program");
    goto END;
  }

  dialog = hildon_file_chooser_dialog_new (
      GTK_WINDOW (priv->main_status.window),
      GTK_FILE_CHOOSER_ACTION_OPEN);

  gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), dir);

  g_signal_connect (dialog, "response",
      G_CALLBACK (backup_dialog_response_cb), data);

  g_signal_connect (dialog, "destroy",
      G_CALLBACK (gtk_widget_destroy), data);

  gtk_widget_show_all (GTK_WIDGET (dialog));

END:
  g_free (dir);
}

static void
menu_about_cb (GtkWidget *widget,
               gpointer data)
{
  gchar *authors [] = {AUTHOR, NULL};

  GtkWidget *about_dialog = g_object_new (GTK_TYPE_ABOUT_DIALOG,
      "program-name", PROGRAM_NAME,
      "logo-icon-name", LOGO_ICON_NAME,
      "authors", authors,
      "comments", COMMENTS,
      "website", WEBSITE,
      NULL);

  gtk_window_set_destroy_with_parent (GTK_WINDOW (about_dialog), TRUE);

  g_signal_connect (about_dialog, "response",
      G_CALLBACK (gtk_widget_destroy), about_dialog);

  g_signal_connect (about_dialog, "destroy",
      G_CALLBACK (gtk_widget_destroy), about_dialog);

  gtk_window_set_modal (GTK_WINDOW (about_dialog), TRUE);
  gtk_widget_show (about_dialog);
}

static inline GtkWidget *
menu_append_item (GtkWidget  *menu,
                  const char *label,
                  gboolean    visible,
                  GCallback   callback,
                  gpointer    data)
{
    GtkWidget *item;

    item = gtk_button_new_with_label (label);

    hildon_gtk_widget_set_theme_size (item, HILDON_SIZE_FINGER_HEIGHT);
    hildon_app_menu_append (HILDON_APP_MENU (menu), GTK_BUTTON (item));
    if (!visible)
      gtk_widget_hide (item);
    g_signal_connect (item, "clicked", callback, data);

    return item;
}

static inline GtkWidget *
menu_append_filter (GtkWidget *menu,
                    GtkWidget *button,
                    const char *label,
                    GCallback   callback,
                    gpointer    data)
{
  GtkWidget *filter;
  static HildonSizeType BUTTON_SIZE =
    HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH;

  if (!button)
    filter = hildon_gtk_radio_button_new (BUTTON_SIZE, NULL);
  else
    filter = hildon_gtk_radio_button_new_from_widget (BUTTON_SIZE,
                                                      GTK_RADIO_BUTTON (button));

  gtk_button_set_label (GTK_BUTTON (filter), label);
  g_signal_connect (filter, "clicked", callback, data);
  hildon_app_menu_add_filter (HILDON_APP_MENU (menu), GTK_BUTTON (filter));
  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (filter), FALSE);

  return filter;
}

static HildonAppMenu *
main_status_window_menu (TpStatusFeed *statusfeed)
{
  GtkWidget *menu = NULL;
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);

  if (G_LIKELY (!priv->main_menu.menu)) {
    priv->main_menu.menu = menu = hildon_app_menu_new ();

    /* Add filters */
    /* TODO If I tap highlighted filter, the callback will be triggered!!!! */
    priv->main_menu.friend_filter = menu_append_filter (
        menu, NULL, "Friends", G_CALLBACK (show_friend_status), statusfeed);
    priv->main_menu.me_filter = menu_append_filter (
            menu, priv->main_menu.friend_filter, "Me",
            G_CALLBACK (show_my_status), statusfeed);

    /* Add items */
    priv->main_menu.delete = menu_append_item (
        menu, "Delete statuses", TRUE, G_CALLBACK (menu_delete_cb), statusfeed);
#if 0
    priv->main_menu.settings = menu_append_item (
        menu, "Settings", TRUE, G_CALLBACK (menu_settings_cb), statusfeed);
#endif
    priv->main_menu.accounts = menu_append_item (
        menu, "Accounts", TRUE, G_CALLBACK (menu_accounts_cb), statusfeed);

    priv->main_menu.backup = menu_append_item (
        menu, "View statuses backup", TRUE, G_CALLBACK (menu_backup_cb), statusfeed);

    priv->main_menu.about = menu_append_item (
        menu, "About", TRUE, G_CALLBACK (menu_about_cb), statusfeed);
  }

  g_object_ref_sink (menu);
  gtk_widget_show_all (GTK_WIDGET (menu));

  return HILDON_APP_MENU (priv->main_menu.menu);
}

/************* contact status menu **************/

static void
_contact_status_deleted_cb (DeleteStatusWindow *del_window,
                            gpointer data)
{
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (data);

  /* refresh contact_status_view */
  contact_status_view_refresh (
      CONTACT_STATUS_VIEW (priv->contact_status.view));

  /* refresh main view */
  timeline_view_refresh (
      TIMELINE_VIEW (priv->main_status.friend_view));

  /* disconnect the "deleted" signal */
  g_signal_handlers_disconnect_by_func (del_window,
      _contact_status_deleted_cb, data);
}

static void
contact_menu_delete_cb (GtkWidget *widget,
                        gpointer data)
{
  TpStatusFeed *statusfeed = TP_STATUS_FEED (data);
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);

  GtkWidget *del_window = g_object_ref_sink (
      delete_status_window_new (priv->aggregator));

  g_signal_connect (DELETE_STATUS_WINDOW (del_window), "deleted",
      G_CALLBACK (_contact_status_deleted_cb), statusfeed);

  const gchar *account = contact_status_view_get_contact_account (
      CONTACT_STATUS_VIEW (priv->contact_status.view));
  const gchar *im_field = contact_status_view_get_contact_im_field (
      CONTACT_STATUS_VIEW (priv->contact_status.view));

  g_debug ("%s: account: %s, im_field: %s",
           G_STRFUNC, account, im_field);

  delete_status_window_populate_contact_status (
      DELETE_STATUS_WINDOW (del_window), account, im_field);

  hildon_program_add_window (priv->program,
      HILDON_WINDOW (del_window));
}

static HildonAppMenu *
contact_status_window_menu (TpStatusFeed *statusfeed)
{
  GtkWidget *menu = NULL;
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);

  if (G_LIKELY (!priv->contact_menu.menu)) {
    priv->contact_menu.menu = menu = hildon_app_menu_new ();

    /* Add items */
    priv->contact_menu.delete = menu_append_item (menu, "Delete statuses",
        TRUE, G_CALLBACK (contact_menu_delete_cb), statusfeed);
  }

  g_object_ref_sink (menu);
  gtk_widget_show_all (GTK_WIDGET (menu));

  return HILDON_APP_MENU (priv->contact_menu.menu);
}

/******** Sub view: contact status window ********/

/**
 * show_contact_status_window:
 *
 * contact_status_window is opened when a status message row is tapped
 * (activated). It shows the selected status message with avatar, presence
 * and service icon of the corresponding contact.
 *
 * Tapping the avatar will open the contact starter for initiating
 * communication via call, sms, email, etc.
 *
 * This window also shows all status history of this contact.
 */
static void
show_contact_status_window (GtkTreeView        *treeview,
                            GtkTreePath        *path,
                            GtkTreeViewColumn  *col,
                            gpointer           data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  /* username of the contact who updates this status (e.g. someone@gmail.com) */
  gchar *remote_account, *remote_name;
  /* local MissonControl account (e.g. gabble/jabber/account0) */
  gchar *local_account;
  gint event_id = 0;
  GdkPixbuf *service_icon = NULL;
  ContactStatus *status;

  TpStatusFeed *statusfeed = TP_STATUS_FEED (data);
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);

  model = gtk_tree_view_get_model (treeview);
  if (!gtk_tree_model_get_iter (model, &iter, path)) /*TODO can't simply return */
    return ;

  gtk_tree_model_get (model, &iter,
                      RTCOM_LOG_VIEW_COL_REMOTE_NAME, &remote_name,
                      RTCOM_LOG_VIEW_COL_REMOTE_ACCOUNT, &remote_account,
                      RTCOM_LOG_VIEW_COL_LOCAL_ACCOUNT, &local_account,
                      RTCOM_LOG_VIEW_COL_EVENT_ID, &event_id,
                      RTCOM_LOG_VIEW_COL_SERVICE_ICON, &service_icon,
                      -1);

  /* full text of this status message */
  status = eventlogger_util_get_contact_status_by_id (priv->eventlogger,
                                                      event_id);
  if (!status) {
    /* This may happen when in collapsed view, the last status got deleted
     * but view is not updated timely
     * Then get the new last status of this contact */
    status = eventlogger_util_get_last_contact_status (
        priv->eventlogger, local_account, remote_account);
  }
  if (!status) {
    /* This may happen when in collapsed view, all statuses of this contact
     * got deletd but view is not updated timely. Simply return */
    return ;
  }

  /* remote_account and remote_name will be freed by contact_status_view */
  contact_status_view_update_contact (CONTACT_STATUS_VIEW (priv->contact_status.view),
                                      local_account,
                                      remote_account,
                                      remote_name,
                                      g_strdup (status->text),
                                      g_strdup (status->start_time),
                                      service_icon);

  gtk_window_set_title (GTK_WINDOW (priv->contact_status.window), remote_name);
  gtk_widget_show_all (GTK_WIDGET (priv->contact_status.view));
  gtk_widget_show (GTK_WIDGET (priv->contact_status.window));
  gtk_widget_grab_focus (GTK_WIDGET (priv->contact_status.view));

  eventlogger_util_contact_status_free (status);
  g_free (local_account);
}

static gboolean
contact_status_window_delete_event_cb (GtkWidget *widget,
                                       GdkEvent  *event,
                                       gpointer   data)
{
  TpStatusFeed *statusfeed = TP_STATUS_FEED (data);
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);

  /* TODO i need a cleanup function? */
  gtk_widget_hide (widget);
  contact_status_view_clean_history (
      CONTACT_STATUS_VIEW (priv->contact_status.view));

  return TRUE;
}

static GtkWidget *
create_contact_status_window (TpStatusFeed *statusfeed)
{
  GtkWidget *window;
  HildonAppMenu *menu;
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);

  window = hildon_stackable_window_new ();
  hildon_program_add_window (priv->program, HILDON_WINDOW (window));

  priv->contact_status.view = g_object_ref_sink (contact_status_view_new ());
  gtk_container_add (GTK_CONTAINER (window),
                     GTK_WIDGET (priv->contact_status.view));

  contact_status_view_set_abook_aggregator (CONTACT_STATUS_VIEW (
                                            priv->contact_status.view),
                                            priv->aggregator);
  /* Menu */
  menu = contact_status_window_menu (statusfeed);
  hildon_window_set_app_menu (HILDON_WINDOW (window), menu);

  g_signal_connect (GTK_WIDGET(window), "delete-event",
                    G_CALLBACK (contact_status_window_delete_event_cb),
                    statusfeed);
  return window;
}

/******** Main view: main status window ********/

static gboolean
tp_status_feed_cleanup (gpointer data)
{
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (data);

  eventlogger_util_save_statuses_to_storage (priv->eventlogger,
      priv->last_friend_status_id, priv->last_my_status_id);

  /* TODO delete too old statuses from el.db */

  gtk_main_quit ();

  return FALSE;
}

static gboolean
tp_status_feed_quit (GtkWidget *window,
                     GdkEvent *event,
                     gpointer data)
{
  gtk_widget_hide_all (window);

  /* TODO first save statuses to file storage
   * then clean up el.db?
   * The process might be long, so we should hide UI for now
   * and do other tasks in an idle function. Once done, quit the main */
  g_idle_add ((GSourceFunc) tp_status_feed_cleanup, data);

  return TRUE;
}

static void
update_status_cb (GtkWidget *widget,
                  gpointer data)
{
  TpStatusFeed *statusfeed = TP_STATUS_FEED (data);
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);
  DBusGConnection *connection;
  DBusGProxy *proxy;
  GError *error = NULL;

  /* Get system bus */
  connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
  if (!connection) {
          osso_abook_handle_gerror (GTK_WINDOW (priv->main_status.window), error);
          return;
  }

  /* Create proxy & request send file dialog */
  proxy = dbus_g_proxy_new_for_name (connection,
                                     "com.nokia.PresenceUI",
                                     "/com/nokia/PresenceUI",
                                     "com.nokia.PresenceUI");

  dbus_g_proxy_call_no_reply (proxy, "StartUp", G_TYPE_INVALID);

  g_object_unref (proxy);
  dbus_g_connection_unref (connection);
}

static void
contact_chooser_response_cb (GtkWidget *chooser,
                             GtkResponseType response,
                             gpointer   data)
{
  TpStatusFeed *statusfeed = TP_STATUS_FEED (data);
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);
  GList *contacts;
  OssoABookContact *master_contact;

  g_debug ("%s", G_STRFUNC);

  if (response != GTK_RESPONSE_OK)
    goto destroy;

  contacts = osso_abook_contact_chooser_get_selection
    (OSSO_ABOOK_CONTACT_CHOOSER (chooser));

  /* Only one master master contact should be selected */
  if (contacts) {
    GList *roster_contacts, *rc;
    const gchar *name;

    master_contact = contacts->data;
    name = osso_abook_contact_get_display_name (master_contact);
    roster_contacts = osso_abook_contact_get_roster_contacts (master_contact);

    /* TODO not support multiple roster contacts yet!!!!!!!!
     * A master contact may be linked to multiple roster contacts
     * e.g. as a result of merging */
    if (roster_contacts) {
/*      for (*/rc = roster_contacts; /*rc; rc = rc->next) {*/
        OssoABookContact *contact;
        McAccount *account;
        const char *vcard_field;
        gchar *im_field;
        ContactStatus *last_status;

        contact = rc->data;
        vcard_field = osso_abook_contact_get_vcard_field (contact);
        account = osso_abook_contact_get_account (contact);
        im_field = osso_abook_contact_get_value (E_CONTACT (contact), vcard_field);

        last_status = eventlogger_util_get_last_contact_status (
            priv->eventlogger, account->name, im_field);

        if (!last_status) {
          gchar *banner = g_strdup_printf ("%s has no status history", name);
          hildon_banner_show_information (
              GTK_WIDGET (priv->main_status.window),
              NULL, banner);

          g_free (banner);
          g_free (im_field);
          g_list_free (roster_contacts);
          g_list_free (contacts);
          goto destroy;
        }

        /* TODO Should update once for all roster contacts!!!! */
        /* im_field and g_strdup (name) will be freed by one-statusp-view */
        contact_status_view_update_contact (CONTACT_STATUS_VIEW (
                                            priv->contact_status.view),
                                            account->name,
                                            im_field,
                                            g_strdup (name),
                                            g_strdup (last_status->text),
                                            g_strdup (last_status->start_time),
                                            NULL); /* TODO proper service icon */

        gtk_window_set_title (GTK_WINDOW (priv->contact_status.window), name);
        gtk_widget_show_all (GTK_WIDGET (priv->contact_status.view));
        gtk_widget_show (GTK_WIDGET (priv->contact_status.window));
        gtk_widget_grab_focus (GTK_WIDGET (priv->contact_status.view));

        eventlogger_util_contact_status_free (last_status);
/*      }*/
      g_list_free (roster_contacts);
    }

    g_list_free (contacts);
  }
destroy:
  gtk_widget_destroy (GTK_WIDGET (chooser));
}

static void
select_abook_cb (GtkWidget *widget,
                 gpointer data)
{
  TpStatusFeed *statusfeed = TP_STATUS_FEED (data);
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);
  GtkWidget *chooser;
  GtkWidget *parent;
  OssoABookContactModel *model;

  g_debug ("%s", G_STRFUNC);

  parent = hildon_window_stack_peek (priv->stack);

  chooser = osso_abook_contact_chooser_new_with_capabilities (
      GTK_WINDOW (parent),
      "View status history of a contact",
      OSSO_ABOOK_CAPS_ALL,
      OSSO_ABOOK_CONTACT_ORDER_PRESENCE); /* TODO caps ADDRESSBOOK is a bit broken */

  model = osso_abook_contact_chooser_get_model (
      OSSO_ABOOK_CONTACT_CHOOSER (chooser));
  osso_abook_list_store_set_roster (
      OSSO_ABOOK_LIST_STORE (model),
      OSSO_ABOOK_ROSTER (priv->aggregator));
  osso_abook_contact_chooser_set_maximum_selection (
      OSSO_ABOOK_CONTACT_CHOOSER (chooser), 1);

  g_signal_connect (chooser, "response",
                    G_CALLBACK (contact_chooser_response_cb),
                    data);

  gtk_window_set_transient_for (GTK_WINDOW (chooser),
                                GTK_WINDOW (parent));
  gtk_window_set_modal (GTK_WINDOW (chooser), TRUE);

  gtk_widget_show_all (GTK_WIDGET (chooser));
}

static void
select_group_cb (GtkWidget *widget,
                 gpointer data)
{
  TpStatusFeed *statusfeed = TP_STATUS_FEED (data);
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);
  GtkWidget *icon;

  if (priv->main_status.group_by == RTCOM_EL_QUERY_GROUP_BY_NONE) {
    priv->main_status.group_by = RTCOM_EL_QUERY_GROUP_BY_CONTACT;
    gtk_button_set_label (GTK_BUTTON (widget), "Show by time");
    timeline_view_set_group_by (TIMELINE_VIEW (priv->main_status.friend_view),
                                RTCOM_EL_QUERY_GROUP_BY_CONTACT);
    icon = gtk_image_new_from_icon_name (
        "general_clock", HILDON_ICON_SIZE_FINGER);

    /* Workaround for the new optimization in eventlogger UI:
     * EL UI caches markup texts and only removed them when received
     * row-changed/row-received/etc signal. Therefore, in order to show
     * event-count in the view when grouped by contact,
     * I need to emit row-changed signal before populating the status */
    timeline_view_row_changed (
        TIMELINE_VIEW (priv->main_status.friend_view));
  }
  else {
    priv->main_status.group_by = RTCOM_EL_QUERY_GROUP_BY_NONE;
    gtk_button_set_label (GTK_BUTTON (widget), "Show by contact");
    timeline_view_set_group_by (TIMELINE_VIEW (priv->main_status.friend_view),
                                RTCOM_EL_QUERY_GROUP_BY_NONE);
    icon = gtk_image_new_from_icon_name (
        "general_default_avatar", HILDON_ICON_SIZE_FINGER);
  }

  timeline_view_populate_friends_status (
      TIMELINE_VIEW (priv->main_status.friend_view));

  gtk_button_set_image (GTK_BUTTON (widget), icon);
  gtk_button_set_image_position (GTK_BUTTON (widget), GTK_POS_LEFT);
}

static void
me_view_refresh_history (TpStatusFeedPrivate *priv)
{
  const gchar *local_uid; /* osso-abook-self */

  if (priv->main_status.me_history_view) {
    /* TODO Disconnect the "row-activated" signal */
    gtk_container_remove (GTK_CONTAINER (priv->main_status.me_view_vbox),
        priv->main_status.me_history_view);
  }

  priv->main_status.me_history_view = contact_history_view_new ();
  gtk_container_add (GTK_CONTAINER (priv->main_status.me_view_vbox),
      priv->main_status.me_history_view);

  /* remote_uid = local_uid, both must not be NULL */
  local_uid = osso_abook_contact_get_uid (
      OSSO_ABOOK_CONTACT (priv->self_contact));

  contact_history_view_set_eventlogger (
      CONTACT_HISTORY_VIEW (priv->main_status.me_history_view),
      priv->eventlogger);
  contact_history_view_update_contact (
      CONTACT_HISTORY_VIEW (priv->main_status.me_history_view),
      local_uid, local_uid);

  /* TODO g_signal_connect */
}

static void
show_my_status (GtkWidget *widget,
                gpointer data)
{
  TpStatusFeed *statusfeed = TP_STATUS_FEED (data);
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);

  if (FALSE == priv->friends_flag)
    return ;

  me_view_refresh_history (priv);

  gtk_window_set_title (GTK_WINDOW (priv->main_status.window),
                        "My Status");
  gtk_widget_show_all (priv->main_status.me_view);
  gtk_widget_hide (priv->main_status.friend_view);
  priv->friends_flag = FALSE;
}

static void
show_friend_status (GtkWidget *widget,
                    gpointer data)
{
  TpStatusFeed *statusfeed = TP_STATUS_FEED (data);
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);

  if (priv->friends_flag)
    return ;

  gtk_window_set_title (GTK_WINDOW (priv->main_status.window),
                        "Friends Status");
  gtk_widget_show (priv->main_status.friend_view);
  gtk_widget_hide (priv->main_status.me_view);
  priv->friends_flag = TRUE;
}

static GtkWidget *
main_status_button_new (const char *icon_name,
                        const char *text,
                        HildonSizeType size_flags)
{
  GtkWidget *button;

  button = hildon_gtk_button_new (size_flags);
  GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS | GTK_CAN_DEFAULT);
  gtk_button_set_focus_on_click (GTK_BUTTON (button), FALSE);

  gtk_button_set_label (GTK_BUTTON (button), text);

  if (icon_name) {
    GtkWidget *icon;

    icon = gtk_image_new_from_icon_name (
        icon_name, HILDON_ICON_SIZE_FINGER);
    gtk_button_set_image (GTK_BUTTON (button), icon);
    gtk_button_set_image_position (GTK_BUTTON (button), GTK_POS_LEFT);
  }

  return button;
}

static gboolean
startup_populate_friend_view (gpointer data)
{
  timeline_view_populate_all_status (data);

  return FALSE;
}

static GtkWidget *
create_me_view (TpStatusFeed *statusfeed)
{
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);
  OssoABookSelfContact *self_contact;
  GtkWidget *me_view;
  GtkWidget *panarea;
  GtkWidget *hbox;
  GtkWidget *image;
  GtkWidget *bubble;

  me_view = gtk_alignment_new (0, 0, 1, 1);
  g_object_set (me_view,
                "top-padding", HILDON_MARGIN_HALF,
                "left-padding", HILDON_MARGIN_DOUBLE,
                "right-padding", HILDON_MARGIN_DOUBLE,
                NULL);

  panarea = hildon_pannable_area_new ();
  g_object_set (panarea,
                "mov-mode", HILDON_MOVEMENT_MODE_VERT,
                "hscrollbar-policy", GTK_POLICY_NEVER,
                NULL);

  priv->main_status.me_view_vbox = gtk_vbox_new (FALSE, HILDON_MARGIN_DEFAULT);

  hbox = gtk_hbox_new (FALSE, 0);
  self_contact = osso_abook_self_contact_get_default ();
  image = osso_abook_avatar_image_new_with_avatar (
      OSSO_ABOOK_AVATAR (self_contact), OSSO_ABOOK_PIXEL_SIZE_AVATAR_CHOOSER);
  priv->main_status.update_button = hildon_gtk_button_new (HILDON_SIZE_AUTO);
  gtk_button_set_relief (
      GTK_BUTTON (priv->main_status.update_button), GTK_RELIEF_NONE);
  bubble = gtk_image_new_from_icon_name (
      "status_logger_bubble", HILDON_ICON_SIZE_FINGER);
  gtk_container_add (
      GTK_CONTAINER (priv->main_status.update_button), bubble);

  gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), priv->main_status.update_button, FALSE, FALSE, 0);

  gtk_box_pack_start (GTK_BOX (priv->main_status.me_view_vbox), hbox, FALSE, FALSE, 0);
  hildon_pannable_area_add_with_viewport (HILDON_PANNABLE_AREA (panarea),
      priv->main_status.me_view_vbox);
  gtk_container_add (GTK_CONTAINER (me_view), panarea);

  g_signal_connect (G_OBJECT (priv->main_status.update_button),
                    "clicked",
                    G_CALLBACK (update_status_cb),
                    statusfeed);

  return me_view;
}

/**
 * create_main_status_window:
 *
 * The main window shows status messages in a timeline view
 * If a status message row is activated, the contact_status_window subview
 * will be open
 */
static GtkWidget *
create_main_status_window (TpStatusFeed *statusfeed)
{
  GtkWidget *window;
  GtkWidget *top_box, *hbox;
  HildonAppMenu *menu;
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);

  window = hildon_stackable_window_new ();
  hildon_program_add_window (priv->program, HILDON_WINDOW (window));

  /* Friend View */
  priv->main_status.friend_view = g_object_ref_sink (timeline_view_new (FALSE));
  timeline_view_add_search_bar (TIMELINE_VIEW (priv->main_status.friend_view),
                                window);
  timeline_view_set_abook_aggregator (TIMELINE_VIEW (priv->main_status.friend_view),
                                      priv->aggregator);
  timeline_view_set_select_status_cb (TIMELINE_VIEW (priv->main_status.friend_view),
                                      show_contact_status_window,
                                      statusfeed);

  priv->main_status.group_button = main_status_button_new (
        "general_clock", "Show by time",
        HILDON_SIZE_FINGER_HEIGHT);
  priv->main_status.abook_button = main_status_button_new (
      "general_contacts", "Select contact",
      HILDON_SIZE_FINGER_HEIGHT);

  /* Me View */
  priv->main_status.me_view = create_me_view (statusfeed);

  /* Packing */
  top_box = timeline_view_get_action_area_box (
      TIMELINE_VIEW (priv->main_status.friend_view));
  gtk_container_add (GTK_CONTAINER (top_box), priv->main_status.group_button);
  gtk_container_add (GTK_CONTAINER (top_box), priv->main_status.abook_button);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), priv->main_status.friend_view, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), priv->main_status.me_view, TRUE, TRUE, 0);
  gtk_container_add (GTK_CONTAINER (window), hbox);

  /* Menu */
  menu = main_status_window_menu (statusfeed);
  hildon_window_set_app_menu (HILDON_WINDOW (window), menu);

  /* Avoid a resizing effect */
  gtk_window_set_default_size (GTK_WINDOW (window),
                               gdk_screen_width (), -1);

  /* Signals */
  g_signal_connect (G_OBJECT (priv->main_status.abook_button),
                    "clicked",
                    G_CALLBACK (select_abook_cb),
                    statusfeed);

  g_signal_connect (G_OBJECT (priv->main_status.group_button),
                    "clicked",
                    G_CALLBACK (select_group_cb),
                    statusfeed);

  g_signal_connect (G_OBJECT(window),
                    "delete_event",
                    G_CALLBACK (tp_status_feed_quit),
                    statusfeed);

  /* By default collapsing all statuses by contact */
  priv->main_status.group_by = RTCOM_EL_QUERY_GROUP_BY_CONTACT;
  timeline_view_set_group_by (
      TIMELINE_VIEW (priv->main_status.friend_view),
      priv->main_status.group_by);

  /* By default to populate all events of STATUS service
   * This is because log_model can be only filtered by service
   * (Call, SMS, Status etc) by setting from rtcom_log_model_populate().
   * In this way, TimelineView won't show any Call, SMS or IM events
   * TODO however, due to limitation of log_model, Friends status view can't
   * avoid to show my status update, and vice versa... unless log_model
   * will also support filtering by EventType */
  g_idle_add ((GSourceFunc)startup_populate_friend_view,
      priv->main_status.friend_view);

  gtk_widget_show_all (window);
  timeline_view_hide_search_bar (TIMELINE_VIEW (
                                 priv->main_status.friend_view));

  return window;
}

/**************************************/
/* GObject functions                  */
/**************************************/

static void
tp_status_feed_init (TpStatusFeed *statusfeed)
{
  TpStatusFeedPrivate *priv;
  GError *err = NULL;

  ULOG_DEBUG_L ("%s", G_STRFUNC);

  priv = statusfeed->priv = G_TYPE_INSTANCE_GET_PRIVATE (statusfeed,
                                                         TP_TYPE_STATUS_FEED,
                                                         TpStatusFeedPrivate);
  priv->program = hildon_program_get_instance ();

  /* Create the default aggregator */
  priv->aggregator = osso_abook_aggregator_get_default (&err);

  if (err) {
      ULOG_ERR_L ("%s: Couldn't create aggregator: %s", G_STRFUNC, err->message);
      g_clear_error (&err);
      if (priv->aggregator) {
          priv->aggregator = NULL; /* Can I just set it to NULL? */
      }
  } else
      g_object_ref (priv->aggregator); /* TODO when to unref? */

  /* Create UI components, and set priv->aggregator to log_model objects
   * of main_status view and contact_status view */
  priv->stack = hildon_window_stack_get_default ();
  priv->main_status.window = create_main_status_window (statusfeed);
  priv->contact_status.window = create_contact_status_window (statusfeed);

  /* Get eventlogger object from log_model of main_status view.
   * It will be used by backend */
  priv->eventlogger = timeline_view_get_eventlogger (
      TIMELINE_VIEW (priv->main_status.friend_view));

  /* Get the last status id before logging any new statuses
   * Saving statuses since this id to storage file
   * when we later close the app */
  priv->last_friend_status_id = eventlogger_util_get_last_status_id (
      priv->eventlogger, "RTCOM_EL_EVENTTYPE_STATUS_FRIEND");
  priv->last_my_status_id = eventlogger_util_get_last_status_id (
      priv->eventlogger, "RTCOM_EL_EVENTTYPE_STATUS_MY");

  contact_status_view_set_eventlogger (CONTACT_STATUS_VIEW (
                                       priv->contact_status.view),
                                       priv->eventlogger);

  /* Connect to contacts-added signal of default aggregator
   * Thus backend will handle all status updates */
  g_signal_connect (
      priv->aggregator, "contacts-added",
      G_CALLBACK (tp_status_feed_backend_contacts_added_cb), statusfeed);

  /* Also start to log self status update */
  tp_status_feed_backend_log_self_status_update (statusfeed);

  /*
    g_signal_connect (
        priv->aggregator, "contacts-removed",
        G_CALLBACK (tp_status_feed_backend_contacts_removed_cb), statusfeed); */

  priv->friends_flag = FALSE;
  /* show friend status view by default */
  show_friend_status (NULL, statusfeed);
}

static void
tp_status_feed_dispose (GObject *obj)
{
  TpStatusFeed *statusfeed = TP_STATUS_FEED (obj);
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);

  osso_abook_roster_stop (OSSO_ABOOK_ROSTER (priv->aggregator));
}

static void
tp_status_feed_finalize (GObject *obj)
{
/*  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (obj); */

  G_OBJECT_CLASS (tp_status_feed_parent_class)->finalize (obj);
}

static void
tp_status_feed_class_init (TpStatusFeedClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->dispose = tp_status_feed_dispose;
  object_class->finalize = tp_status_feed_finalize;

  g_type_class_add_private (klass, sizeof (TpStatusFeedPrivate));
}

/**************************************/
/* Public functions                  */
/**************************************/

TpStatusFeed *
tp_status_feed_get_instance (void)
{
  static TpStatusFeed *instance = NULL;

  if (G_UNLIKELY (!instance)) {
    instance = g_object_new (TP_TYPE_STATUS_FEED, NULL);
    g_object_add_weak_pointer (G_OBJECT (instance), (gpointer)&instance);
  }

  return instance;
}

void
tp_status_set_osso (TpStatusFeed *statusfeed,
                    osso_context_t *osso)
{
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);

  priv->osso_context = osso;
}

