/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include <Qt>
#include <QDBusConnection>
#include <QDBusInterface>

#include "packageselector.h"
#include "ui_packageselector.h"
#include "package.h"
#include "packageview.h"
#include "repository.h"
#include "blacklistselect.h"
#include "aaptinterface.h"
#include "settings.h"
#include "mainwindow.h"

PackageSelector::PackageSelector(Package* pkg, AAptInterface* apt, Settings* set, MainWindow* mw, QWidget *parent) :
    QDialog(parent),
	ui(new Ui::PackageSelector), iNetworkAccessManager(0), iChangelogFetched(false), iFetchInProgress(false)
{
	ui->setupUi(this);
	iPkg = pkg;
	iAptInterface = apt;
	iSettings = set;
	iMainWindow = mw;
	iChangelog = "";

	ui->pushButton_website->setIcon(QPixmap("/usr/share/icons/hicolor/48x48/hildon/general_web.png"));
	if( getMaemoOrgUrl(pkg).isEmpty() && getMaemoOrgUrl(pkg->availablePackage()).isEmpty() ) {
		ui->pushButton_website->setEnabled(false);
	}
	if( pkg->isMarkedForOperation() )
		ui->pushButton_blacklist->setEnabled(false);

	if( pkg->icon() && !pkg->icon()->isNull() ) {
		QPixmap aicon( *pkg->icon() );
		ui->label_appicon->setPixmap( aicon.scaled(48,48) );
	} else {
		ui->label_appicon->setPixmap( QPixmap(":/icons/icons/appdefault.png") );
	}
	this->setWindowTitle( pkg->name() );

	if( pkg->maemoDisplayName() != "" )
		ui->label_header->setText("<b>" + pkg->maemoDisplayName() + "</b>");
	else
		ui->label_header->setText("<b>" + pkg->name() + "</b>");

	ui->radioTabBasic->setChecked(true);
	updateInfo();

	// switch to shorter button labels for portait mode
	if( QApplication::desktop()->width() < QApplication::desktop()->height() )
	{
		ui->radioTabBasic->setText("Info");
		ui->radioTabChanges->setText("Chgs");
		ui->radioTabDepends->setText("Deps");
	}

	Package* upg_pkg = iPkg->availablePackage();
	if( !upg_pkg )
		upg_pkg = iPkg;
	if( (!pkg->isUpgradeable() || (upg_pkg && upg_pkg->upgradeDescription().isEmpty())) && (getMaemoOrgUrl(upg_pkg).isEmpty()) )
		ui->radioTabChanges->hide();

	if( pkg->isUpgradeable() )
		ui->radioInstall->setText("Upgrade");
	else
		ui->radioInstall->setText("Install");

	if( pkg->markedOperation() == Package::PkgOpNone )
	{
		if( pkg->isInstalled() && pkg->isUpgradeable() )
			ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_nop_instupgr.png"));
		else if( pkg->isInstalled() )
			ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_nop_installed.png"));
		else if( !pkg->isInstalled() )
			ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_nop_notinstalled.png"));

		ui->radioNothing->setChecked(true);
	} else if( pkg->markedOperation() == Package::PkgOpInstallUpgrade ) {
		if( pkg->isUpgradeable() )
			ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_upgrade.png"));
		else
			ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_install.png"));
		ui->radioInstall->setChecked(true);
	} else if( pkg->markedOperation() == Package::PkgOpRemove ) {
		ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_remove.png"));
		ui->radioRemove->setChecked(true);
	}

	if( !getMaemoOrgUrl(upg_pkg).isEmpty() && !iChangelogFetched && !iFetchInProgress &&
		iSettings->qsettings()->value("always_fetch_changes",false).toBool() )
	{
		queryChangelog();
	}
}

PackageSelector::~PackageSelector()
{
	if( iNetworkAccessManager ) {
		delete iNetworkAccessManager;
		iNetworkAccessManager = 0;
	}
    delete ui;
}

void PackageSelector::changeEvent(QEvent *e)
{
    QDialog::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

Package::operation PackageSelector::selectedOperation()
{
	if( ui->radioInstall->isChecked() )
		return Package::PkgOpInstallUpgrade;
	if( ui->radioRemove->isChecked() )
		return Package::PkgOpRemove;

	return Package::PkgOpNone;
}

void PackageSelector::on_pushButton_blacklist_clicked()
{
	Package* pkg = iPkg;
	/*if( iPkg->isUpgradeable() && iPkg->availablePackage() )
		pkg = iPkg->availablePackage();*/

	if( pkg->isInstalled() && pkg->isUpgradeable() )
	{
		Package* upg = pkg->availablePackage();
		if( upg )
			pkg = upg;
	}

	BlacklistSelect s(pkg, this);
	BlacklistSelect::blackList old = pkg->blacklisted();
	if( s.exec() ) {		
		if( old != pkg->blacklisted() ) {
			Package* p1 = iAptInterface->packagesInstalled()->value(pkg->name(),0);
			Package* p2 = iAptInterface->packagesAvailable()->value(pkg->name(),0);
			if( p1 ) {
				iAptInterface->removeFromBlacklist(p1,old);
				if( pkg->blacklisted()==BlacklistSelect::BlacklistAll || pkg->blacklisted()==BlacklistSelect::BlacklistNone )
					p1->setBlacklisted( pkg->blacklisted() );
				else if( pkg->blacklisted()==BlacklistSelect::BlacklistThis && pkg->version()==p1->version() )
					p1->setBlacklisted( pkg->blacklisted() );
				else
					p1->setBlacklisted( BlacklistSelect::BlacklistNone );
			}
			if( p2 ) {
				iAptInterface->removeFromBlacklist(p2,old);
				if( pkg->blacklisted()==BlacklistSelect::BlacklistAll || pkg->blacklisted()==BlacklistSelect::BlacklistNone )
					p2->setBlacklisted( pkg->blacklisted() );
				else if( pkg->blacklisted()==BlacklistSelect::BlacklistThis && pkg->version()==p2->version() )
					p2->setBlacklisted( pkg->blacklisted() );
				else
					p2->setBlacklisted( BlacklistSelect::BlacklistNone );
			}
			iAptInterface->writeBlacklist();
		}
		updateInfo();
	}
}

QString PackageSelector::getMaemoOrgUrl(Package* pkg)
{
	QString url = "";

	if( !pkg )
		return url;

	if( pkg->repositories().count() == 0 )
		return url;

	for( int i=0; i<pkg->repositories().count(); i++ )
	{
		if( pkg->repositories().at(i) && pkg->repositories().at(i)->url().startsWith("http://repository.maemo.org") )
			url = "http://maemo.org/packages/view/" + pkg->name() + "/";
	}
	return url;
}

void PackageSelector::updateInfo()
{
	if( ui->radioTabBasic->isChecked() )
		on_radioTabBasic_clicked();
	else if( ui->radioTabChanges->isChecked() )
		on_radioTabChanges_clicked();
	else if( ui->radioTabDepends->isChecked() )
		on_radioTabDepends_clicked();

	ui->radioInstall->setEnabled(true);
	ui->radioRemove->setEnabled(true);
	ui->radioInstall->show();
	ui->radioRemove->show();

	Package* upg_pkg = iPkg->availablePackage();

	if( iPkg->isInstalled() && !iPkg->isUpgradeable() )
	{
		ui->radioInstall->setEnabled(false);
		ui->radioInstall->hide();
	} else if( !iPkg->isInstalled() ) {
		ui->radioRemove->setEnabled(false);
		ui->radioRemove->hide();
	}

	if( iPkg->isBlacklisted() ) {
		ui->radioInstall->setEnabled(false);
		//ui->radioRemove->setEnabled(false);
	}
	if( iPkg->isInstalled() && upg_pkg && upg_pkg->isBlacklisted() ) {
		ui->radioInstall->setEnabled(false);
	}
}

void PackageSelector::on_pushButton_website_clicked()
{
	QString url = getMaemoOrgUrl(iPkg);
	if( url == "" ) {
		url = getMaemoOrgUrl(iPkg->availablePackage());
		if( url == "")
			return;
	}

	QDBusConnection conn = QDBusConnection::connectToBus(QDBusConnection::SessionBus, "faster_application_manager");

	QString service = "com.nokia.osso_browser";
	QString path = "/com/nokia/osso_browser/request";
	QString method = "open_new_window";

	QDBusInterface browser(service, path, service, conn, this);
	browser.call(method, url);
}

void PackageSelector::on_radioTabBasic_clicked()
{
	ui->label_header2->show();
	QString header2;

	Package* upg_pkg = iPkg->availablePackage();

	if( iPkg->isInstalled() ) {
		header2 += "Installed, version <b>" + iPkg->version() + "</b>";
	} else {
		header2 += "Not installed";
	}
	header2 += "<br>";

	if( iPkg->markedOperation() == Package::PkgOpInstallUpgrade )
	{
		if( iPkg->isUpgradeable() )
			header2 += "Marked for <b>upgrade</b><br>";
		else
			header2 += "Marked for <b>installation</b><br>";
	} else if(iPkg->markedOperation() == Package::PkgOpRemove)
	{
		header2 += "Marked for <b>removal</b><br>";
	}

	if( iPkg->isBlacklisted() && !iPkg->isUpgradeable() ) {
		header2 += "BLACKLISTED";
		if( iPkg->blacklisted() == BlacklistSelect::BlacklistAll )
			header2 += " (all)";
		else if( iPkg->blacklisted() == BlacklistSelect::BlacklistThis )
			header2 += " (this)";
		iPkg->setMarkedForOperation(Package::PkgOpNone);
		header2 += "<br>";
	} else if( upg_pkg && upg_pkg->isBlacklisted() ) {
		header2 += "BLACKLISTED";
		if( upg_pkg->blacklisted() == BlacklistSelect::BlacklistAll )
			header2 += " (all)";
		else if( upg_pkg->blacklisted() == BlacklistSelect::BlacklistThis )
			header2 += " (" + upg_pkg->version() + ")";
		iPkg->setMarkedForOperation(Package::PkgOpNone);
		upg_pkg->setMarkedForOperation(Package::PkgOpNone);
		header2 += "<br>";
	}

	ui->label_header2->setText(header2);


	QString longtext;
	longtext += "Category: " + iPkg->section() + "<br>";

	if( iPkg->isInstalled() ) {
		longtext += "Installation date: ";
		if( iPkg->date().isValid() )
			longtext += iPkg->date().toString("yyyy-MM-dd hh:mm");
		else
			longtext += "Unknown";
		longtext += "<br>";
	}

	if( iPkg->isInstalled() && iPkg->installedSize() > 0 )
		longtext += QString("Size: %L1 kB<br>").arg(iPkg->installedSize() );


	if( !iPkg->isInstalled() ){
		longtext += "Available version <b>" + iPkg->version() + "</b>";
		if( iPkg->size() > 0 )
			longtext += QString(" (%L1 kB)").arg(iPkg->size()/1024 );
		longtext += "<br>";
	}
	else if( iPkg->isUpgradeable() ){
		longtext += "Available version <b>" + iPkg->upgradeableVersion() + "</b>";
		if( iPkg->size() > 0 )
			longtext += QString(" (%L1 kB)").arg(iPkg->size()/1024 );
		longtext += "<br>";
	}

	if( !iPkg->isInstalled() ) {
		longtext += "Available package date: ";
		if( iPkg->date().isValid() )
			longtext += iPkg->date().toString("yyyy-MM-dd hh:mm");
		else
			longtext += "Unknown";
		longtext += "<br>";
	} else if( iPkg->isUpgradeable() ) {
		longtext += "Available package date: ";
		if( iPkg->availablePackage() && iPkg->availablePackage()->date().isValid() )
			longtext += iPkg->availablePackage()->date().toString("yyyy-MM-dd hh:mm");
		else
			longtext += "Unknown";
		longtext += "<br>";
	}

	if( !iPkg->isInstalled() )
	{
		longtext += "Repositories: ";
		if( iPkg->repositories().count()>0 ) {
			for(int i=0; i<iPkg->repositories().count(); i++ ) {
				if( iPkg->repositories().at(i) )
					longtext += iPkg->repositories().at(i)->name();
				else
					longtext += "unknown";
				if( i<iPkg->repositories().count()-1 )
					longtext += ", ";
			}
			longtext += "<br>";
		} else {
			longtext += "unknown<br>";
		}
	}

	if( iPkg->isUpgradeable() )
	{
		longtext += "Repositories: ";
		if( upg_pkg && upg_pkg->repositories().count()>0 ) {
			for(int i=0; i<upg_pkg->repositories().count(); i++ ) {
				if( upg_pkg->repositories().at(i) )
					longtext += upg_pkg->repositories().at(i)->name();
				else
					longtext += "unknown";
				if( i<upg_pkg->repositories().count()-1 )
					longtext += ", ";
			}
			longtext += "<br>";
		} else {
			longtext += "unknown<br>";
		}
	}

	if( !iPkg->maintainerRichText().isEmpty() ) {
		longtext +="Maintainer: " + iPkg->maintainerRichText() + "<br>";
	}

	longtext += "<br>" + iPkg->descShort();
	QString descLong = iPkg->descLong();
	if( descLong.length()>0 ) {
		descLong.replace('\n',"<br>");
		longtext += "<font size=\"-1\"><br><br>" + descLong + "</font>";
	}

	ui->label_text->setText( longtext );
}

void PackageSelector::on_radioTabChanges_clicked()
{
	ui->label_header2->hide();
	ui->label_header2->setText("");

	QString text;

	Package* upg_pkg = iPkg->availablePackage();
	if( !upg_pkg )
		upg_pkg = iPkg;

	if( iPkg->isUpgradeable() && upg_pkg && !upg_pkg->upgradeDescription().isEmpty() )
	{
		text += "<u><b>Upgrade description:</b></u>";
		text += "<font size=\"-1\"><br>";
		text += upg_pkg->upgradeDescription();
		text += "</font><br>";
		text.replace('\n',"<br>");
	}

	QString changelog;
	if( !getMaemoOrgUrl(upg_pkg).isEmpty() && !iChangelogFetched && !iFetchInProgress ) {
		queryChangelog();
		changelog = "Fetching changelog...";
	} else {
		changelog = iChangelog;
		if( iFetchInProgress )
			changelog = "Fetching changelog...";
		else if( changelog == "" )
			changelog = "Not available";
	}

	text += "<u><b>Debian changelog:</b></u><font size=\"-1\"><br>";
	text += changelog;
	text += "</font>";

	ui->label_text->setText(text);
}

void PackageSelector::on_radioTabDepends_clicked()
{
	ui->label_header2->hide();
	ui->label_header2->setText("");
	ui->label_text->setText("");

	QStringList deps = Package::toTrimmedRichTextList( iPkg->depends() );
	QStringList confl = Package::toTrimmedRichTextList( iPkg->conflicts() );
	QStringList predeps = Package::toTrimmedRichTextList( iPkg->preDepends() );
	QStringList repl = Package::toTrimmedRichTextList( iPkg->replaces() );
	QStringList prov = Package::toTrimmedRichTextList( iPkg->provides() );
	QStringList brks = Package::toTrimmedRichTextList( iPkg->breaks() );
	QStringList recs = Package::toTrimmedRichTextList( iPkg->recommends() );
	QStringList sugs = Package::toTrimmedRichTextList( iPkg->suggests() );

	QString text;

	if( deps.count()>0 )
	{
		text += "<b><u>Depends:</u></b><br><font size=\"-1\">";
		text += deps.join(", ");
		text += "</font><br>";
	}

	if( predeps.count()>0 )
	{
		text += "<b><u>Pre-Depends:</u></b><br><font size=\"-1\">";
		text += predeps.join(", ");
		text += "</font><br>";
	}

	if( confl.count()>0 )
	{
		text += "<b><u>Conflicts:</u></b><br><font size=\"-1\">";
		text += confl.join(", ");
		text += "</font><br>";
	}

	if( prov.count()>0 )
	{
		text += "<b><u>Provides:</u></b><br><font size=\"-1\">";
		text += prov.join(", ");
		text += "</font><br>";
	}

	if( repl.count()>0 )
	{
		text += "<b><u>Replaces:</u></b><br><font size=\"-1\">";
		text += repl.join(", ");
		text += "</font><br>";
	}

	if( brks.count()>0 )
	{
		text += "<b><u>Breaks:</u></b><br><font size=\"-1\">";
		text += brks.join(", ");
		text += "</font><br>";
	}

	if( recs.count()>0 )
	{
		text += "<b><u>Recommends:</u></b><br><font size=\"-1\">";
		text += recs.join(", ");
		text += "</font><br>";
	}

	if( sugs.count()>0 )
	{
		text += "<b><u>Suggests:</u></b><br><font size=\"-1\">";
		text += sugs.join(", ");
		text += "</font><br>";
	}

	if( deps.count()==0 && predeps.count()==0 && confl.count()==0 && prov.count()==0 &&
		repl.count()==0 && brks.count()==0 && recs.count()==0 && sugs.count()==0 )
	{
		text = "Package has no dependencies listed";
	}

	ui->label_text->setText(text);
}

void PackageSelector::queryChangelog()
{
	if( iChangelogFetched || iFetchInProgress )
		return;

	if( iMainWindow )
		iMainWindow->openNetworkConnection();

	iChangelogFetched = false;
	iFetchInProgress = true;
	iChangelog = "";

	Package* pkg = iPkg->availablePackage();
	if( !pkg )
		pkg = iPkg;

	if( getMaemoOrgUrl(pkg) == "" ) {
		iChangelogFetched = true;
		return;
	}

	if( !iNetworkAccessManager ) {
		iNetworkAccessManager = new QNetworkAccessManager(this);
        connect(iNetworkAccessManager,SIGNAL(finished(QNetworkReply*)),this,SLOT(changelogFetchNetworkReply(QNetworkReply*)));

		if( iSettings->qsettings()->value("use_proxies").toBool() && !iSettings->qsettings()->value("http_proxy").toString().isEmpty() )
		{
			 QNetworkProxy proxy = Settings::createProxyFromString( iSettings->qsettings()->value("http_proxy").toString() );
			 iNetworkAccessManager->setProxy(proxy);
		}
	}

	QUrl url( getMaemoOrgUrl(pkg) );
	QNetworkRequest req(url);

	qDebug() << "fetching changelog";

	iNetworkAccessManager->get(req);
}

void PackageSelector::changelogFetchNetworkReply(QNetworkReply* reply)
{
	if( reply->error() == QNetworkReply::NoError ) {
		QByteArray data = reply->readAll();

		int pos = data.indexOf("<div class=\"changelog\">");
		int pos2 = data.indexOf("</div>", pos);

		if( pos!=-1 && pos2!=-1 ) {
			iChangelog = parseChangeTable( QString::fromUtf8( data.mid(pos,pos2-pos+6) ) );
		} else {
			iChangelog = "";
		}

		qDebug() << "changelog fetched";
	} else {
		iChangelog = "Not available ( error)";
		qDebug() << "changelog fetch error:" << reply->error() << reply->errorString();
	}

	reply->deleteLater();
	iFetchInProgress = false;
	iChangelogFetched = true;

	if( this->isVisible() && ui->radioTabChanges->isChecked() )
		on_radioTabChanges_clicked();
}

QString PackageSelector::parseChangeTable(QString tabledata)
{
	tabledata.remove("<strong>");
	tabledata.remove("</strong>");
	tabledata.remove('\n');

	//qDebug() << tabledata;

	QString newdata = "<table><tbody>";

	int pos = tabledata.indexOf("<tr>", tabledata.indexOf("<tbody>") );
	while( pos != -1 ) {
		int pos2 = tabledata.indexOf("</td>",pos);
		pos2 = tabledata.indexOf("</td>",pos2+5);

		if( pos2 != -1 )
			newdata += tabledata.mid(pos, pos2-pos+5) + "</tr>";

		pos = tabledata.indexOf("<tr>",pos+4);
	}

	newdata += "</tbody></table>";

	//qDebug() << newdata;

	return newdata;
}
