/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include <QtGui>
#include "dimmer.h"

dimmer::dimmer(QWidget *parent) :
    QWidget(parent)
{
	iParent = parent;

	iLayout = new QVBoxLayout(this);
	iLabel = new QLabel(this);
	iLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
	iLabel->setAlignment(Qt::AlignCenter);
	iLayout->addWidget(iLabel);
	iLayout->setSizeConstraint(QLayout::SetMaximumSize);

	iAlpha = 205;
	iBusy = false;
	iAnim = 0;
	iAnimDir = 1;
	iAnimY = this->rect().height()-this->rect().height()/4;
	iProgress = -1;
	iUpdateAnim = false;
	iDownloadSpeed = -1;


	iTimer = new QTimer(this);
	connect(iTimer,SIGNAL(timeout()),this,SLOT(timerEvent()));

	connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));

	hide();
}

dimmer::~dimmer()
{
	// iTimer and other widgets are automatically deleted by their parent
}

void dimmer::paintEvent(QPaintEvent *)
{
	QPainter painter(this);
	QColor dim("black");
	dim.setAlpha(iAlpha);
	painter.setPen(dim);
	painter.setBrush(dim);
	painter.drawRect(rect());

	painter.setBrush( QApplication::palette().color(QPalette::Highlight) );

	int step = 15;
	int animMax = 10;

	if( iProgress == -1 ) {
		painter.drawEllipse(QPoint(rect().left()+(rect().width()/2-(animMax*step)/2)+iAnim*step,
							iAnimY), 10, 10);
	}

	if( iUpdateAnim ) {
		iAnim += iAnimDir;
		if( iAnim>animMax )
			iAnimDir=-1;
		if( iAnim==-1 )
			iAnimDir=1;
	}

	if( iProgress >= 0 ) {
		if( iProgress>100 )
			iProgress = 100;

		painter.setBrush( QApplication::palette().color(QPalette::Window) );
		painter.setPen( dim );
		painter.drawRect(rect().left()+30, rect().bottom()-30, rect().right()-rect().left()-60, 10 );
		painter.setBrush( QApplication::palette().color(QPalette::Highlight) );
		painter.setPen( QApplication::palette().color(QPalette::Highlight) );
		int pw = ( rect().right()-rect().left()-60 ) * iProgress / 100;
		painter.drawRect(rect().left()+30, rect().bottom()-30, pw, 10 );

		if( iDownloadSpeed >= 0 ) {
			painter.setBrush( QApplication::palette().color(QPalette::BrightText) );
			painter.setPen( QApplication::palette().color(QPalette::BrightText) );
			QRect textrect(rect().left(),rect().bottom()-70,rect().width(),30);
			painter.drawText(textrect,QString("%1 kB/s").arg(iDownloadSpeed),Qt::AlignHCenter|Qt::AlignVCenter);
		}
	}
}

void dimmer::timerEvent()
{
	iUpdateAnim = true;
	repaint(0,iAnimY-10,rect().width(),22);
	iUpdateAnim = false;
}

void dimmer::resizeEvent(QResizeEvent* event)
{
	iAnimY = this->rect().height() - this->rect().height() / 4;
	QWidget::resizeEvent(event);
}

void dimmer::orientationChanged()
{
	//resize( iParent->size() );
}

void dimmer::dim(QString title, QString message)
{
	QString colorname = QApplication::palette().color(QPalette::BrightText).name();
	iTitle = title;
	iBusy = true;
	iLabel->setText("<font color=\"" + colorname + "\"><b><u>" + iTitle + "</u></b><br><br>" + message + "</font>");
	iProgress = -1;
	iDownloadSpeed = -1;
	iAnim = 0;
	iAnimDir = 1;

	show();

	resize(iParent->size());

	iTimer->start(250);
}

void dimmer::updateText(QString message)
{
	QString colorname = QApplication::palette().color(QPalette::BrightText).name();
	iLabel->setText("<font color=\"" + colorname + "\"><b><u>" + iTitle + "</u></b><br><br>" + message + "</font>");
}

void dimmer::setProgress(int p_)
{
	iProgress = p_;
	if( p_==-1 )
		iDownloadSpeed = -1;
	repaint(0,rect().bottom()-30,rect().width(),20);
}

void dimmer::setDownloadSpeed(int kbps_)
{
	iDownloadSpeed = kbps_;
	repaint(QRect(rect().left(),rect().bottom()-70,rect().width(),30));
}

void dimmer::undim()
{
	iBusy = false;
	iTimer->stop();
	hide();
}
