import inspect
import logging
import operator

from dwimd.plugins import get_plugins, get_plugin
from dwimd.values import make_descriptor
from dwimd.util import extract_globals


# constants
read_sentinel = object()

# global state
gestalts = {}
current_plan = None


class PlanningException(Exception):
    pass

class PluginException(Exception):
    pass


class Environment(object):
    def __init__(self, globals, plan):
        self.globals = globals
        self.plan = plan
        self.written_action_values = {} # action -> value
        self.disabled_actions = set() # action


class Plan(object):
    def __init__(self):
        self.forced_actions = set()
        self.required_actions = set()
        self.values = {}
        self.votes = {} # action -> [value, votes]
        self.globals = None

    def compute(self):
        for action, value_votes_dict in self.votes.items():
            max_item = max((value, key) for (key, value) in value_votes_dict.items())
            votes, value = max_item
            if votes > 0:
                self.values[action] = value
        self.votes.clear()

    def __repr__(self):
        return "<Plan values=%r>" % (self.values, )


def require(action, value=None):
    logging.debug("Requirement of %r=%r" % (action, value))
    def require_action(key, value):
        if key in current_plan.forced_actions and current_plan.values[key] != value:
            raise PlanningException("Gestalt or action cannot be required, "
                    "because the action " + key.name + " is forced.")
        if key in current_plan.required_actions and current_plan.values[key] != value:
            raise PlanningException("Gestalt or action cannot be required, "
                    "because the action " + key.name + " is already required.")
        current_plan.required_actions.add(key)
        current_plan.values[key] = value
        try:
            del current_plan.votes[key]
        except KeyError:
            pass

    if type(action) == str:
        # require a gestalt
        assert value is None
        for key, value in gestalts[action].items():
            require_action(make_descriptor(key), value)
    else:
        assert hasattr(action, "name")
        require_action(action, value)

def suggest(action, value=None):
    logging.debug("Suggestion for %r=%r" % (action, value))
    def suggest_action(key, value):
        if key not in current_plan.values:
            current_plan.values[key] = value
    if type(action) == str:
        # require a gestalt
        assert value is None
        for key, value in gestalts[action].items():
            suggest_action(make_descriptor(key), value)
    else:
        assert hasattr(action, "name")
        suggest_action(action, value)

def vote(action, value, votes=1):
    logging.debug("Vote for %r=%r: %r" % (action, value, votes))
    assert hasattr(action, "name")
    key = action
    if key in current_plan.forced_actions and current_plan.values[key] != value:
        raise PlanningException("Action " + key.name + " cannot be voted for, "
                "because the action is forced.")
    if key in current_plan.required_actions and current_plan.values[key] != value:
        raise PlanningException("Action " + key.name + " cannot be voted for, "
                "because the action is required.")
    current_plan.votes.setdefault(key, {value: 0})[value] += votes # XXX does this work?

def force(action, value=None):
    logging.debug("Forcing %r=%r" % (action, value))
    def force_action(key, value):
        if key in current_plan.forced_actions and current_plan.values[key] != value:
            raise PlanningException("Gestalt or action cannot be forced, "
                    "because the action " + key.name + " is forced.")
        current_plan.forced_actions.add(key)
        current_plan.values[key] = value
        try:
            del current_plan.votes[key]
        except KeyError:
            pass

    if type(action) == str:
        # require a gestalt
        assert value is None
        for key, value in gestalts[action].items():
            force_action(make_descriptor(key), value)
    else:
        assert hasattr(action, "name")
        force_action(action, value)


def define_gestalt(name, **kwargs):
    logging.debug("Define gestalt %r: %r" % (name, kwargs))
    if name in gestalts:
        raise PlanningException("Duplicate gestalt definition for %r" % (name, ))
    gestalts[name] = kwargs


def init_globals():
    myglobals = {"require": require,
                 "suggest": suggest,
                 "vote": vote,
                 "force": force,
                 "define_gestalt": define_gestalt}

    for pname, plugin in get_plugins("actions"):
        assert pname not in myglobals
        myglobals.update(getattr(plugin, "values", {}))
        if hasattr(plugin, "further_names"):
            for key in plugin.further_names:
                if key in myglobals and not getattr(plugin, "allow_duplicates", False):
                    raise PluginException("Duplicate action name %r" % (key,))
                myglobals[key] = make_descriptor(key)
        myglobals[pname] = make_descriptor(pname)
    return myglobals


def update_globals(myglobals, wanted_globals, check_duplicates=False):
    for pname, plugin in get_plugins("sensors"):
        exporting = getattr(plugin, "exporting", None)
        imported_globals = getattr(plugin, "imported_globals", {})
        if exporting is not None and not ((
            set(exporting) | set(imported_globals.keys())) & wanted_globals):
            continue
        sample = plugin.sample()

        sample.update(imported_globals)
        for key in sample.keys():
            if check_duplicates:
                if key in myglobals and not getattr(plugin, "allow_duplicates", False):
                    raise PlanningException("Duplicate sensor key %r" % (key,))
            myglobals[key] = sample[key]


def run_functions(functions, env=None):
    global current_plan
    gestalts.clear()
    wanted_globals = reduce(operator.or_, (extract_globals(func) for func in functions), set())
    logging.debug("Only using globals %r" % (wanted_globals, ))
    if env is None:
        plan = Plan()
        globals = init_globals()
        env = Environment(globals, plan)
        update_globals(globals, wanted_globals, True)
    else:
        env.plan = Plan()
        update_globals(env.globals, wanted_globals)
    current_plan = env.plan
    myglobals = env.globals
    for function in functions:
        logging.debug("Running function %r" % (function.__name__, ))
        exec function.func_code in myglobals, {}
    current_plan = None
    env.plan.compute()
    return env


def run_config_modules(env=None):
    all_functions = []
    for pname, config in get_plugins("configs"):
        functions = inspect.getmembers(config, inspect.isfunction)
        all_functions.extend([func for _, func in sorted(functions)])
    if not all_functions:
        raise Exception("No config files with functions in them found.")
    return run_functions(all_functions, env)


def act(env):
    logging.debug("Acting these values: %r" % (env.plan.values, ))
    valuenames = set(env.plan.values.keys())
    for pname, plugin in get_plugins("actions"):
        action = make_descriptor(pname)
        if action in env.plan.values:
            value = env.plan.values[action]
            valuenames.remove(action)
            if hasattr(plugin, "further_names"):
                values = {pname: value}
                for name in plugin.further_names:
                    values[name] = env.plan.values.get(make_descriptor(name))
                    if make_descriptor(name) in valuenames:
                        valuenames.remove(make_descriptor(name))
            else:
                values = value
            read = read_sentinel # try hard to read only once
            if action in env.disabled_actions:
                read = plugin.read()
                if read == env.written_action_values[action]:
                    env.disabled_actions.remove(action)
                    logging.debug("Enabling action %r" % (action, ))
                else:
                    continue
            if action in env.written_action_values:
                old_value = env.written_action_values[action]
                read = plugin.read()
                if old_value != read:
                    # XXX add gui notification here
                    logging.debug("Disabling action %r, old %r, new %r" % (action, old_value, read))
                    env.disabled_actions.add(action)
                    continue
            if hasattr(plugin, "read"):
                if read is read_sentinel:
                    read = plugin.read()
                if read is NotImplemented:
                    plugin.act(values) # fallback
                elif value != read:
                    plugin.act(values)
                    env.written_action_values[action] = value
            else:
                plugin.act(values)

    if valuenames:
        logging.error("Unsupported actions found: %r" % (valuenames, ))

