#include "vimeovideolisttab.h"
#include "vimeovideoinfowindow.h"
#include "vimeoalbumdialog.h"
#include "videolistdelegate.h"
#include "vimeovideoselectionwindow.h"
#include "videoplaybackwindow.h"
#include "../base/notifications.h"
#include "../base/settings.h"
#include "windowstack.h"
#include "../base/urlgrabber.h"
#include "../base/vimeo.h"
#include "../base/transfermanager.h"
#include "../base/playbacklistmodel.h"
#include "../base/vimeovideolistmodel.h"
#include "listview.h"
#include <QAction>
#include <QLabel>
#include <QGridLayout>
#include <QMenu>
#include <QTimer>
#include <QMaemo5Style>

VimeoVideoListTab::VimeoVideoListTab(QWidget *parent) :
    AbstractTab(parent),
    m_model(new VimeoVideoListModel(this)),
    m_listView(new ListView(this)),
    m_deleteFromUploadsAction(new QAction(tr("Delete from uploads"), this)),
    m_addToLikesAction(new QAction(tr("Add to likes"), this)),
    m_deleteFromLikesAction(new QAction(tr("Delete from likes"), this)),
    m_addToAlbumAction(new QAction(tr("Add to album"), this)),
    m_watchLaterAction(new QAction(tr("Watch later"), this)),
    m_addToQueueAction(new QAction(tr("Add to playback queue"), this)),
    m_videoDownloadAction(new QAction(tr("Download"), this)),
    m_audioDownloadAction(new QAction(tr("Download (audio)"), this)),
    m_contextMenu(new QMenu(this)),
    m_deleteFromUploadsContextAction(m_contextMenu->addAction(tr("Delete from uploads"), this, SLOT(deleteVideoFromUploads()))),
    m_addToLikesContextAction(m_contextMenu->addAction(tr("Add to likes"), this, SLOT(addVideoToLikes()))),
    m_deleteFromLikesContextAction(m_contextMenu->addAction(tr("Delete from likes"), this, SLOT(deleteVideoFromLikes()))),
    m_addToAlbumContextAction(m_contextMenu->addAction(tr("Add to album"), this, SLOT(showAlbumDialog()))),
    m_watchLaterContextAction(m_contextMenu->addAction(tr("Watch later"), this, SLOT(addVideoToWatchLaterAlbum()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No videos found")), this)),
    m_grid(new QGridLayout(this))
{
    this->addAction(m_deleteFromUploadsAction);
    this->addAction(m_addToLikesAction);
    this->addAction(m_deleteFromLikesAction);
    this->addAction(m_addToAlbumAction);
    this->addAction(m_watchLaterAction);
    this->addAction(m_addToQueueAction);
    this->addAction(m_videoDownloadAction);
    this->addAction(m_audioDownloadAction);

    m_noResultsLabel->hide();
    m_grid->setContentsMargins(0, 0, 0, 0);
    m_grid->addWidget(m_listView, 0, 0);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    this->connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreVideos()));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_deleteFromUploadsAction, SIGNAL(triggered()), this, SLOT(deleteVideosFromUploads()));
    this->connect(m_addToLikesAction, SIGNAL(triggered()), this, SLOT(addVideosToLikes()));
    this->connect(m_deleteFromLikesAction, SIGNAL(triggered()), this, SLOT(deleteVideosFromLikes()));
    this->connect(m_addToAlbumAction, SIGNAL(triggered()), this, SLOT(addVideosToAlbum()));
    this->connect(m_watchLaterAction, SIGNAL(triggered()), this, SLOT(addVideosToWatchLaterAlbum()));
    this->connect(m_addToQueueAction, SIGNAL(triggered()), this, SLOT(queueVideos()));
    this->connect(m_videoDownloadAction, SIGNAL(triggered()), this, SLOT(addVideoDownloads()));
    this->connect(m_audioDownloadAction, SIGNAL(triggered()), this, SLOT(addAudioDownloads()));

    this->onOrientationChanged();
}

void VimeoVideoListTab::showEvent(QShowEvent *event) {
    AbstractTab::showEvent(event);

    if ((m_model) && (m_model->refreshRequired())) {
        QTimer::singleShot(1000, m_model, SLOT(reset()));
    }
}

void VimeoVideoListTab::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void VimeoVideoListTab::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void VimeoVideoListTab::setVideoFeed(const QUrl &feed) {
    m_model->getVideos(feed);
    this->setMenuActions();
}

void VimeoVideoListTab::setMenuActions() {
    bool signedIn = Vimeo::instance()->userSignedIn();

    m_deleteFromUploadsAction->setEnabled((signedIn) && (m_model->feed() == VIMEO_UPLOADS_FEED));
    m_addToLikesAction->setEnabled((signedIn) && (m_model->feed() != VIMEO_LIKES_FEED));
    m_deleteFromLikesAction->setEnabled((signedIn) && (m_model->feed() == VIMEO_LIKES_FEED));
    m_addToAlbumAction->setEnabled(signedIn);
    m_watchLaterAction->setEnabled((signedIn) && (m_model->feed() != VIMEO_WATCH_LATER_FEED));
    m_deleteFromUploadsContextAction->setVisible((signedIn) && (m_model->feed() == VIMEO_UPLOADS_FEED));
    m_addToLikesContextAction->setVisible((signedIn) && (m_model->feed() != VIMEO_LIKES_FEED));
    m_deleteFromLikesContextAction->setVisible((signedIn) && (m_model->feed() == VIMEO_LIKES_FEED));
    m_addToAlbumContextAction->setVisible(signedIn);
    m_watchLaterContextAction->setVisible((signedIn) && (m_model->feed() != VIMEO_WATCH_LATER_FEED));
}

void VimeoVideoListTab::onLoadingChanged(bool loading) {
    Notifications::toggleBusy(loading);
    m_noResultsLabel->setVisible((!loading) && (m_model->rowCount() == 0));
}

void VimeoVideoListTab::onCountChanged(int count) {
    m_noResultsLabel->setVisible((count == 0) && (!m_model->loading()));
}

void VimeoVideoListTab::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        this->showVideoInfo(m_model->get(index.row()));
    }
}

void VimeoVideoListTab::showVideoInfo(QSharedPointer<VideoItem> video) {
    VimeoVideoInfoWindow *window = new VimeoVideoInfoWindow(video, WindowStack::instance()->currentWindow());
    window->show();
}

void VimeoVideoListTab::playVideo(const QModelIndex &index) {
    if (Settings::instance()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(QList< QSharedPointer<VideoItem> >() << m_model->get(index.row()), WindowStack::instance()->currentWindow());
        window->show();
    }
    else {
        UrlGrabber::instance()->getVimeoVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
    }
}

void VimeoVideoListTab::deleteVideosFromUploads() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_model, tr("Delete from uploads"), SLOT(deleteVideosFromUploads()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListTab::addVideosToLikes() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_model, tr("Add to likes"), SLOT(addVideosToLikes()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListTab::deleteVideosFromLikes() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_model, tr("Delete from likes"), SLOT(deleteVideosFromLikes()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListTab::addVideosToAlbum() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_model, tr("Add to album"), SLOT(showAlbumDialog()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListTab::addVideosToWatchLaterAlbum() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_model, tr("Watch later"), SLOT(addVideosToWatchLaterAlbum()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListTab::addVideoDownloads() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_model, tr("Download"), SLOT(addVideoDownloads()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListTab::addAudioDownloads() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_model, tr("Download"), SLOT(addAudioDownloads()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListTab::queueVideos() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_model, tr("Add to queue"), SLOT(queueVideos()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void VimeoVideoListTab::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_deleteFromUploadsContextAction);
}

void VimeoVideoListTab::deleteVideoFromUploads() {
    Vimeo::instance()->deleteFromUploads(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void VimeoVideoListTab::addVideoToLikes() {
    Vimeo::instance()->addToLikes(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void VimeoVideoListTab::deleteVideoFromLikes() {
    Vimeo::instance()->deleteFromLikes(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void VimeoVideoListTab::showAlbumDialog() {
    VimeoAlbumDialog *dialog = new VimeoAlbumDialog(this);
    dialog->open();

    this->connect(dialog, SIGNAL(albumChosen(QString)), this, SLOT(addVideoToAlbum(QString)));
    this->connect(dialog, SIGNAL(addToNewAlbum(QVariantMap)), this, SLOT(addVideoToNewAlbum(QVariantMap)));
}

void VimeoVideoListTab::addVideoToAlbum(const QString &id) {
    Vimeo::instance()->addToAlbum(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()), id);
}

void VimeoVideoListTab::addVideoToNewAlbum(const QVariantMap &album) {
    Vimeo::instance()->createAlbum(album, QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void VimeoVideoListTab::addVideoToWatchLaterAlbum() {
    Vimeo::instance()->addToWatchLaterAlbum(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void VimeoVideoListTab::addVideoDownload() {
    TransferManager::instance()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void VimeoVideoListTab::addAudioDownload() {
    TransferManager::instance()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void VimeoVideoListTab::queueVideo() {
    PlaybackListModel::playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}
