#include "transferlistwindow.h"
#include "transferlistdelegate.h"
#include "transferprioritydialog.h"

TransferListWindow::TransferListWindow(Session *session, QWidget *parent) :
    StackedWindow(session, parent),
    m_listView(new ListView(this)),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No transfers queued")), this)),
    m_contextMenu(new QMenu(this)),
    m_priorityAction(m_contextMenu->addAction(tr("Set priority"), this, SLOT(showPriorityDialog()))),
    m_pauseResumeAction(m_contextMenu->addAction(tr("Pause"), this, SLOT(pauseOrResumeTransfer()))),
    m_cancelAction(m_contextMenu->addAction(tr("Cancel"), this, SLOT(cancelTransfer())))
{
    setWindowTitle(tr("Transfers"));
    setCentralWidget(m_listView);

    m_listView->setModel(m_session->transferManager());
    m_listView->setItemDelegate(new TransferListDelegate(m_listView));

    m_session->transferManager()->setViewVisible(true);

    connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    connect(m_session->transferManager(), SIGNAL(countChanged()), this, SLOT(onCountChanged()));

    onCountChanged();
    onOrientationChanged();
}

TransferListWindow::~TransferListWindow() {
    m_session->transferManager()->setViewVisible(false);
}

void TransferListWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void TransferListWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void TransferListWindow::showContextMenu(const QPoint &pos) {
    TransferItem::Status status = m_session->transferManager()->get(m_listView->currentIndex().row()).data()->status();
    m_pauseResumeAction->setText(status == TransferItem::Active ? tr("Pause") : tr("Resume"));
    m_contextMenu->popup(pos, m_pauseResumeAction);
}

void TransferListWindow::showPriorityDialog() {
    TransferPriorityDialog *dialog = new TransferPriorityDialog(this);
    dialog->setValue(m_session->transferManager()->get(m_listView->currentIndex().row()).data()->priority());
    dialog->show();

    connect(dialog, SIGNAL(valueChanged(TransferItem::Priority)), this, SLOT(setTransferPriority(TransferItem::Priority)));
}

void TransferListWindow::setTransferPriority(TransferItem::Priority priority) {
    m_session->transferManager()->setTransferPriority(m_listView->currentIndex().row(), priority);
}

void TransferListWindow::pauseOrResumeTransfer() {
    if (m_session->transferManager()->get(m_listView->currentIndex().row()).data()->status() == TransferItem::Active) {
        m_session->transferManager()->pauseTransfer(m_listView->currentIndex().row());
    }
    else {
        m_session->transferManager()->resumeTransfer(m_listView->currentIndex().row());
    }
}

void TransferListWindow::cancelTransfer() {
    m_session->transferManager()->cancelTransfer(m_listView->currentIndex().row());
}

void TransferListWindow::onCountChanged() {
    m_noResultsLabel->setVisible(m_session->transferManager()->rowCount() == 0);
}
