#include "cookiejar.h"
#include <QRegExp>

CookieJar::CookieJar(CookieJar::Mode mode, QObject *parent) :
    QNetworkCookieJar(parent),
    m_mode(mode)
{
}

QList<QNetworkCookie> CookieJar::cookiesForUrl(const QUrl &url) const {
    QList<QNetworkCookie> cookies;

    if (url.toString().contains(QRegExp("(dailymotion.com/video|dailymotion.com/cdn)"))) {
        if (m_mode == CookieJar::DownloadMode) {
            cookies = m_dailymotionCookies;
        }

        cookies << QNetworkCookie("family_filter", "false");
    }
    else {
        cookies = QNetworkCookieJar::cookiesForUrl(url);
    }

    return cookies;
}

bool CookieJar::youtubeCookiesSet() const {
    return !m_youtubeCookies.isEmpty();
}

bool CookieJar::dailymotionCookiesSet() const {
    return !m_dailymotionCookies.isEmpty();
}

void CookieJar::setYouTubeCookies(const QByteArray &cookieString) {
    clearYouTubeCookies();
    m_youtubeCookies = QNetworkCookie::parseCookies(cookieString);
    setCookiesFromUrl(m_youtubeCookies, QUrl("http://www.youtube.com"));
}

void CookieJar::setYouTubeCookies() {
    QList<QNetworkCookie> cookies = allCookies();
    QByteArray cookieString;

    for (int i = 0; i < cookies.size(); i++) {
        QNetworkCookie cookie = cookies.at(i);

        if (cookie.domain().contains("google")) {
            cookieString.append(cookie.toRawForm(QNetworkCookie::NameAndValueOnly) + "\n");
            cookies.removeAt(i);
        }
    }

    setAllCookies(cookies);
    setYouTubeCookies(cookieString);
    emit gotYouTubeCookies(cookieString);
}

void CookieJar::setDailymotionCookies(const QByteArray &cookieString) {
    clearDailymotionCookies();
    m_dailymotionCookies = QNetworkCookie::parseCookies(cookieString);
    setCookiesFromUrl(m_dailymotionCookies, QUrl("http://www.dailymotion.com"));
}

void CookieJar::setDailymotionCookies() {
    QList<QNetworkCookie> cookies = allCookies();
    QByteArray cookieString;

    for (int i = 0; i < cookies.size(); i++) {
        QNetworkCookie cookie = cookies.at(i);

        if (cookie.domain().contains("dailymotion")) {
            cookieString.append(cookie.toRawForm(QNetworkCookie::NameAndValueOnly) + "\n");
            cookies.removeAt(i);
        }
    }

    setAllCookies(cookies);
    setDailymotionCookies(cookieString);
    emit gotDailymotionCookies(cookieString);
}

void CookieJar::storeYouTubeCookies() {
    QList<QNetworkCookie> cookies = cookiesForUrl(QUrl("http://www.youtube.com"));
    m_youtubeCookies = cookies;
    QByteArray cookieString;

    while (!cookies.isEmpty()) {
        cookieString.append(cookies.takeFirst().toRawForm(QNetworkCookie::NameAndValueOnly) + "\n");
    }

    emit gotYouTubeCookies(cookieString);
    emit alert(tr("YouTube cookies stored"));
}

void CookieJar::storeDailymotionCookies() {
    QList<QNetworkCookie> cookies = cookiesForUrl(QUrl("http://www.dailymotion.com"));
    m_dailymotionCookies = cookies;
    QByteArray cookieString;

    while (!cookies.isEmpty()) {
        cookieString.append(cookies.takeFirst().toRawForm(QNetworkCookie::NameAndValueOnly) + "\n");
    }

    emit gotDailymotionCookies(cookieString);
    emit alert(tr("Dailymotion cookies stored"));
}

void CookieJar::clearYouTubeCookies() {
    m_youtubeCookies.clear();
    QList<QNetworkCookie> oldCookies = allCookies();
    QList<QNetworkCookie> newCookies;

    while (!oldCookies.isEmpty()) {
        QNetworkCookie cookie = oldCookies.takeFirst();

        if (!cookie.domain().contains(QRegExp("(google|youtube)"))) {
            newCookies.append(cookie);
        }
    }

    setAllCookies(newCookies);
}

void CookieJar::clearYouTubeCookiesFromDB(bool displayAlert) {
    clearYouTubeCookies();
    emit gotYouTubeCookies(QByteArray());

    if (displayAlert) {
        emit alert(tr("YouTube cookies deleted"));
    }
}

void CookieJar::clearDailymotionCookies() {
    m_dailymotionCookies.clear();
    QList<QNetworkCookie> oldCookies = allCookies();
    QList<QNetworkCookie> newCookies;

    while (!oldCookies.isEmpty()) {
        QNetworkCookie cookie = oldCookies.takeFirst();

        if (!cookie.domain().contains("dailymotion")) {
            newCookies.append(cookie);
        }
    }

    setAllCookies(newCookies);
}

void CookieJar::clearDailymotionCookiesFromDB(bool displayAlert) {
    clearDailymotionCookies();
    emit gotDailymotionCookies(QByteArray());

    if (displayAlert) {
        emit alert(tr("Dailymotion cookies deleted"));
    }
}

void CookieJar::clearTwitterCookies() {
    QList<QNetworkCookie> oldCookies = allCookies();
    QList<QNetworkCookie> newCookies;

    while (!oldCookies.isEmpty()) {
        QNetworkCookie cookie = oldCookies.takeFirst();

        if (!cookie.domain().contains("twitter")) {
            newCookies.append(cookie);
        }
    }

    setAllCookies(newCookies);
}
