#include "transferlistwindow.h"
#include "transferlistdelegate.h"
#include "transferprioritydialog.h"
#include "transferdetailsdialog.h"
#include "listview.h"
#include "transferlistmodel.h"
#include "../base/transfermanager.h"
#include <QLabel>
#include <QMenu>
#include <QMenuBar>
#include <QAction>
#include <QMaemo5Style>

TransferListWindow::TransferListWindow(QWidget *parent) :
    StackedWindow(parent),
    m_model(new TransferListModel(this)),
    m_listView(new ListView(this)),
    m_noResultsLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No transfers queued")), this)),
    m_contextMenu(new QMenu(this)),
    m_priorityAction(m_contextMenu->addAction(tr("Set priority"), this, SLOT(showPriorityDialog()))),
    m_pauseResumeAction(m_contextMenu->addAction(tr("Pause"), this, SLOT(pauseOrResumeTransfer()))),
    m_cancelAction(m_contextMenu->addAction(tr("Cancel"), this, SLOT(cancelTransfer()))),
    m_startAction(this->menuBar()->addAction(tr("Start"), TransferManager::instance(), SLOT(start()))),
    m_pauseAction(this->menuBar()->addAction(tr("Pause"), TransferManager::instance(), SLOT(pause())))
{
    this->setWindowTitle(tr("Transfers"));
    this->setCentralWidget(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(new TransferListDelegate(m_listView));

    this->connect(m_listView, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));

    this->onCountChanged(m_model->rowCount());
    this->onOrientationChanged();
}

TransferListWindow::~TransferListWindow() {}

void TransferListWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void TransferListWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void TransferListWindow::onItemClicked(const QModelIndex &index) {
    TransferDetailsDialog *dialog = new TransferDetailsDialog(m_model->get(index.row()), this);
    dialog->open();
}

void TransferListWindow::showContextMenu(const QPoint &pos) {
    QModelIndex index = m_listView->currentIndex();

    if (!index.isValid()) {
        return;
    }

    switch (index.data(TransferItem::StatusRole).toInt()) {
    case Transfers::Paused:
        m_pauseResumeAction->setEnabled(true);
        m_pauseResumeAction->setText(tr("Start"));
        break;
    case Transfers::Failed:
        m_pauseResumeAction->setEnabled(true);
        m_pauseResumeAction->setText(tr("Start"));
        break;
    default:
        m_pauseResumeAction->setText(tr("Pause"));

        switch (index.data(TransferItem::TransferTypeRole).toInt()) {
        case Transfers::Upload:
            m_pauseResumeAction->setEnabled(false);
            break;
        default:
            m_pauseResumeAction->setEnabled(true);
            break;
        }

        break;
    }

    m_contextMenu->popup(pos, m_priorityAction);
}

void TransferListWindow::showPriorityDialog() {
    TransferPriorityDialog *dialog = new TransferPriorityDialog(this);
    dialog->setValue(static_cast<Transfers::Priority>(m_listView->currentIndex().data(TransferItem::PriorityRole).toInt()));
    dialog->show();

    this->connect(dialog, SIGNAL(valueChanged(Transfers::Priority)), this, SLOT(setTransferPriority(Transfers::Priority)));
}

void TransferListWindow::setTransferPriority(Transfers::Priority priority) {
    m_model->setData(m_listView->currentIndex(), priority, TransferItem::PriorityRole);
}

void TransferListWindow::pauseOrResumeTransfer() {
    QModelIndex index = m_listView->currentIndex();

    if (!index.isValid()) {
        return;
    }

    Transfers::Status status = static_cast<Transfers::Status>(index.data(TransferItem::StatusRole).toInt());

    switch (status) {
    case Transfers::Paused:
        m_model->setData(index, Transfers::Queued, TransferItem::StatusRole);
        break;
    case Transfers::Failed:
        m_model->setData(index, Transfers::Queued, TransferItem::StatusRole);
        break;
    default:
        m_model->setData(index, Transfers::Paused, TransferItem::StatusRole);
    }
}

void TransferListWindow::cancelTransfer() {
    m_model->setData(m_listView->currentIndex(), Transfers::Cancelled, TransferItem::StatusRole);
}

void TransferListWindow::onCountChanged(int count) {
    m_noResultsLabel->setVisible(count == 0);
    m_startAction->setEnabled(count > 0);
    m_pauseAction->setEnabled(count > 0);
}
