#ifndef YOUTUBE_H
#define YOUTUBE_H

#include "enums.h"
#include "feedurls.h"
#include "videoitem.h"
#include "playlistitem.h"
#include "useritem.h"
#include "commentitem.h"
#include <QObject>
#include <QByteArray>
#include <QString>
#include <QUrl>
#include <QVariantMap>
#include <QSharedPointer>

class QNetworkAccessManager;
class QNetworkReply;

class YouTube : public QObject
{
    Q_OBJECT

    Q_PROPERTY(QUrl uploadsFeed
               READ uploadsFeed
               CONSTANT)
    Q_PROPERTY(QUrl favouritesFeed
               READ favouritesFeed
               CONSTANT)
    Q_PROPERTY(QUrl playlistsFeed
               READ playlistsFeed
               CONSTANT)
    Q_PROPERTY(QUrl subscriptionsFeed
               READ subscriptionsFeed
               CONSTANT)
    Q_PROPERTY(QUrl newSubscriptionVideosFeed
               READ newSubscriptionVideosFeed
               CONSTANT)
    Q_PROPERTY(QUrl recommendedFeed
               READ recommendedFeed
               CONSTANT)
    Q_PROPERTY(QUrl watchLaterFeed
               READ watchLaterFeed
               CONSTANT)
    Q_PROPERTY(QUrl watchHistoryFeed
               READ watchHistoryFeed
               CONSTANT)
    Q_PROPERTY(QString username
               READ username
               NOTIFY usernameChanged)
    Q_PROPERTY(bool userSignedIn
               READ userSignedIn
               NOTIFY userSignedInChanged)
    Q_PROPERTY(QUrl authUrl
               READ authUrl
               CONSTANT)
    Q_PROPERTY(bool playlistsLoaded
               READ playlistsLoaded
               NOTIFY playlistsLoadedChanged)
    Q_PROPERTY(bool subscriptionsLoaded
               READ subscriptionsLoaded
               NOTIFY subscriptionsLoadedChanged)
    Q_PROPERTY(bool busy
               READ busy
               NOTIFY busyChanged)

public:
    explicit YouTube(QObject *parent = 0);
    ~YouTube();
    inline QNetworkAccessManager* networkAccessManager() const { return m_nam; }
    inline void setNetworkAccessManager(QNetworkAccessManager *manager) { m_nam = manager; }
    inline QString username() const { return m_user; }
    inline bool userSignedIn() const { return !this->accessToken().isEmpty(); }
    inline QUrl uploadsFeed() const { return YOUTUBE_UPLOADS_FEED; }
    inline QUrl favouritesFeed() const { return YOUTUBE_FAVOURITES_FEED; }
    inline QUrl recommendedFeed() const { return YOUTUBE_RECOMMENDED_FEED; }
    inline QUrl playlistsFeed() const { return YOUTUBE_PLAYLISTS_FEED; }
    inline QUrl subscriptionsFeed() const { return YOUTUBE_SUBSCRIPTIONS_FEED; }
    inline QUrl newSubscriptionVideosFeed() const { return YOUTUBE_NEW_SUBSCRIPTION_VIDEOS_FEED; }
    inline QUrl watchLaterFeed() const { return YOUTUBE_WATCH_LATER_FEED; }
    inline QUrl watchHistoryFeed() const { return YOUTUBE_WATCH_HISTORY_FEED; }
    QNetworkReply* createReply(QUrl feed, int offset = 0);
    QNetworkReply* createSearchReply(
            int queryType,
            const QString &query,
            int offset = 1,
            int order = Queries::Relevance,
            int time = Queries::AllTime,
            int duration = Queries::Any,
            const QString &language = "all");
    QNetworkReply* createUploadReply(const QVariantMap &metadata);
    QUrl authUrl() const;
    inline bool safeSearch() const { return m_safeSearch; }
    inline QList< QSharedPointer<PlaylistItem> > * playlists() const { return m_playlistCache; }
    inline QList< QSharedPointer<UserItem> > * subscriptions() const { return m_subscriptionCache; }
    inline bool playlistsLoaded() const { return m_playlistCacheLoaded; }
    inline bool subscriptionsLoaded() const { return m_subscriptionCacheLoaded; }
    inline bool busy() const { return m_busy; }
    inline bool cancelled() const { return m_cancelled; }

    static YouTube* instance();

public slots:
    void signIn(const QString &displayName, const QString &code);
    void signInFromDevice(const QString &displayName);
    void signOut();
    void getPlaylists(int offset = 1);
    void getSubscriptions(int offset = 1);
    inline void setSafeSearch(bool safe) { m_safeSearch = safe; }
    void setAccount(const QString &user = QString(), const QString &token = QString(), const QString &refresh = QString());
    inline void setUsername(const QString &user) { m_user = user; emit usernameChanged(user); }
    void deleteFromUploads(const QStringList &videoIds);
    void addToFavourites(const QStringList &videoIds);
    void deleteFromFavourites(const QStringList &videoIds);
    void addToPlaylist(const QStringList &videoIds, const QString &playlistId);
    void deleteFromPlaylist(const QStringList &videoIds, const QString &playlistId);
    void addToWatchLaterPlaylist(const QStringList &videoIds);
    void deleteFromWatchLaterPlaylist(const QStringList &videoIds);
    void createPlaylist(const QVariantMap &playlist, const QStringList &videoIds = QStringList());
    void deletePlaylist(const QString &playlistId);
    void subscribe(const QString &userId);
    void unsubscribe(const QString &userId);
    void likeVideo(const QString &videoId);
    void dislikeVideo(const QString &videoId);
    void addComment(const QVariantMap &comment);
    void replyToComment(const QVariantMap &comment);
    void updateVideoMetadata(const QVariantMap &metadata);
    void refreshAccessToken();
    void getFullVideo(QString id);
    void getCurrentUserProfile();
    void getUserProfile(const QString &id = QString());
    void linkGoogleAccount(QString username);
    void checkUsernameAvailability(QString username);
    void cancelLinkGoogleAccount();
    bool subscribedToChannel(const QString &userId = QString());
    void getVideosFromIds(QStringList ids);
    void getVideoMetadata(const QString &id = QString());
    void getFullComment(const QString &videoId, const QString &commentId);
    void cancelCurrentOperation();

private:
    void clearLoginInfo();
    inline QString accessToken() const { return m_token; }
    inline void setAccessToken(const QString &token) { m_token = token; emit userSignedInChanged(); }
    inline QString refreshToken() const { return m_refreshToken; }
    inline void setRefreshToken(const QString &token) { m_refreshToken = token; }
    void clearCache();
    void setPlaylistsLoaded(bool loaded) { m_playlistCacheLoaded = loaded; emit playlistsLoadedChanged(); }
    void setSubscriptionsLoaded(bool loaded) { m_subscriptionCacheLoaded = loaded; emit subscriptionsLoadedChanged(); }
    void addNewPlaylistToCache(QSharedPointer<PlaylistItem> playlist);
    QSharedPointer<PlaylistItem> removePlaylistFromCache(const QString &id);
    void addNewSubscriptionToCache(QSharedPointer<UserItem> user);
    QSharedPointer<UserItem> removeSubscriptionFromCache(const QString &id);
    void updatePlaylistVideoCount(const QString &id, int change);
    void updatePlaylistThumbnail(const QString &id, const QUrl &thumbnailUrl);
    void getPlaylistForCache(const QString &id);
    void getAddedComment(const QString &videoId, const QString &commentId);
    void setBusy(bool isBusy, const QString &message = QString(), int numberOfOperations = 1);
    inline void setCancelled(bool cancelled) { m_cancelled = cancelled; }

private slots:
    void postRequest(const QUrl &url, const QByteArray &xml);
    void putRequest(const QUrl &url, const QByteArray &xml);
    void deleteRequest(const QUrl &url);
    void postFinished();
    void checkDeviceSignInInfo();
    void pollServerForDeviceToken();
    void checkDeviceToken();
    void checkIfSignedIn();
    void checkIfSignedOut();
    void checkTokenRefresh();
    void deleteFromUploads();
    void onVideoDeleted();
    void onVideoMetadataUpdated();
    void addToFavourites();
    void onAddedToFavourites(const QString &response);
    void deleteFromFavourites();
    void onDeletedFromFavourites();
    void addToPlaylist();
    void onAddedToPlaylist(const QString &response);
    void deleteFromPlaylist();
    void onDeletedFromPlaylist();
    void addToWatchLaterPlaylist();
    void onAddedToWatchLaterPlaylist(const QString &response);
    void deleteFromWatchLaterPlaylist();
    void onDeletedFromWatchLaterPlaylist();
    void createPlaylist();
    void deletePlaylist();
    void subscribe();
    void unsubscribe();
    void likeVideo();
    void dislikeVideo();
    void addComment();
    void replyToComment();
    void updateVideoMetadata();
    void onPlaylistCreated(const QString &response);
    void onPlaylistDeleted();
    void onSubscribed(const QString &response);
    void onUnsubscribed();
    void onVideoLiked();
    void onVideoDisliked();
    void onCommentAdded(const QString &response);
    void onVideoActionError(const QString &errorString);
    void onPlaylistActionError(const QString &errorString);
    void onUserActionError(const QString &errorString);
    void onCommentActionError(const QString &errorString);
    void checkFullVideo();
    void checkUserProfile();
    void checkCurrentUserProfile();
    void checkSuggestedUsernames();
    void onGoogleAccountLinked(const QString &response);
    void addPlaylists();
    void addSubscriptions();
    void checkFullVideos();
    void checkVideoMetadata();
    void checkFullComment();
    void checkCachePlaylist();
    void checkAddedComment();

signals:
    void alert(const QString &message);
    void busy(const QString &message, int numberOfOperations = 1);
    void busyProgressChanged(int progress);
    void busyChanged(bool isBusy);
    void info(const QString &message);
    void error(const QString &errorString);
    void usernameChanged(const QString &username);
    void postSuccessful(const QString &response);
    void postFailed(const QString &errorString);
    void videoMetadataUpdated(const QString &videoId, const QVariantMap &metadata);
    void deletedFromUploads(const QString &videoId);
    void addedToPlaylist(const QString &videoId, const QString &playlistId);
    void deletedFromPlaylist(const QString &videoId, const QString &playlistId);
    void addedToWatchLaterPlaylist(const QString &videoId);
    void deletedFromWatchLaterPlaylist(const QString &videoId);
    void subscriptionChanged(const QString &userId, bool subscribed, const QString &subscriptionId);
    void favouriteChanged(const QString &videoId, bool favourite, const QString &favouriteId);
    void commentAdded(QSharedPointer<CommentItem> comment);
    void videoLiked(const QString &videoId);
    void videoDisliked(const QString &videoId);
    void userSignedInChanged();
    void signedIn(const QString &displayName, const QString &token, const QString &refresh);
    void newAccountSet();
    void accessTokenRefreshed(const QString &token);
    void refreshError();
    void requestToLinkGoogleAccount();
    void googleAccountLinked();
    void gotSuggestedUsernames(const QStringList &usernames);
    void usernameUnavailable();
    void usernameAvailable();
    void gotVideo(QSharedPointer<VideoItem> video);
    void gotVideo(VideoItem *video);
    void gotUser(QSharedPointer<UserItem> user);
    void gotUser(UserItem *user);
    void playlistsLoadedChanged();
    void subscriptionsLoadedChanged();
    void allPlaylistsLoaded();
    void allSubscriptionsLoaded();
    void playlistAddedToCache(int row);
    void playlistRemovedFromCache(int row);
    void playlistUpdated(int row);
    void subscriptionAddedToCache(int row);
    void subscriptionRemovedFromCache(int row);
    void gotVideosFromIds(QList< QSharedPointer<VideoItem> > videos);
    void gotVideoMetadata(const QVariantMap &metadata);
    void gotComment(QSharedPointer<CommentItem> comment);
    void currentOperationCancelled();

private:
    static YouTube *self;

    QNetworkAccessManager *m_nam;

    struct Login {
        QString displayName;
        QString deviceCode;
        int deviceInterval;
    } m_login;

    QString m_user;
    QString m_token;
    QString m_refreshToken;
    bool m_safeSearch;
    QStringList m_actionIdList;
    QString m_actionId;
    QVariantMap m_metadataAction;
    int m_actionsProcessed;
    QList< QSharedPointer<PlaylistItem> > *m_playlistCache;
    QList< QSharedPointer<UserItem> > *m_subscriptionCache;
    bool m_playlistCacheLoaded;
    bool m_subscriptionCacheLoaded;
    bool m_busy;
    bool m_cancelled;
    QHash<int, QString> m_queryOrders;
    QHash<int, QString> m_timeFilters;
    QHash<int, QString> m_durationFilters;
};

#endif // YOUTUBE_H
