#include "videoitem.h"
#include "utils.h"
#include "dailymotion.h"
#include "youtube.h"
#include "vimeo.h"

VideoItem::VideoItem(QObject *parent) :
    QObject(parent),
    m_likes(0),
    m_dislikes(0),
    m_ratingCount(0),
    m_rating(0),
    m_viewCount(0),
    m_liked(false),
    m_disliked(false),
    m_favourite(false),
    m_downloaded(false),
    m_private(false),
    m_service(Services::NoService)
{
}

VideoItem::VideoItem(VideoItem *video, QObject *parent) :
    QObject(parent),
    m_id(video->id()),
    m_videoId(video->videoId()),
    m_favouriteId(video->favouriteId()),
    m_playlistId(video->playlistVideoId()),
    m_url(video->url()),
    m_embedUrl(video->embedUrl()),
    m_title(video->title()),
    m_artist(video->artist()),
    m_artistId(video->artistId()),
    m_artistAvatarUrl(video->artistAvatarUrl()),
    m_description(video->description()),
    m_duration(video->duration()),
    m_date(video->date()),
    m_tags(video->tags()),
    m_likes(video->likes()),
    m_dislikes(video->dislikes()),
    m_ratingCount(video->ratingCount()),
    m_rating(video->rating()),
    m_viewCount(video->viewCount()),
    m_thumbUrl(video->thumbnailUrl()),
    m_largeThumbUrl(video->largeThumbnailUrl()),
    m_subtitlesUrl(video->subtitlesUrl()),
    m_liked(video->liked()),
    m_disliked(video->disliked()),
    m_favourite(video->favourite()),
    m_downloaded(video->downloaded()),
    m_private(video->isPrivate()),
    m_service(video->service())
{
    switch (this->service()) {
    case Services::YouTube:
        this->connect(YouTube::instance(), SIGNAL(favouriteChanged(QString,bool,QString)), this, SLOT(onFavouriteChanged(QString,bool,QString)));
        this->connect(YouTube::instance(), SIGNAL(videoLiked(QString)), this, SLOT(onVideoLiked(QString)));
        this->connect(YouTube::instance(), SIGNAL(videoDisliked(QString)), this, SLOT(onVideoDisliked(QString)));
        this->connect(YouTube::instance(), SIGNAL(videoMetadataUpdated(QString,QVariantMap)), this, SLOT(onMetadataUpdated(QString,QVariantMap)));
        break;
    case Services::Dailymotion:
        this->connect(Dailymotion::instance(), SIGNAL(favouriteChanged(QString,bool)), this, SLOT(onFavouriteChanged(QString,bool)));
        this->connect(Dailymotion::instance(), SIGNAL(videoMetadataUpdated(QString,QVariantMap)), this, SLOT(onMetadataUpdated(QString,QVariantMap)));
        break;
    case Services::Vimeo:
        this->connect(Vimeo::instance(), SIGNAL(likeChanged(QString,bool)), this, SLOT(onFavouriteChanged(QString,bool)));
        break;
    default:
        break;
    }

    emit favouriteChanged(this->favourite());
    emit downloadedChanged(this->downloaded());
}

void VideoItem::loadVideo(VideoItem *video) {
    this->setId(video->id());
    this->setVideoId(video->videoId());
    this->setFavouriteId(video->favouriteId());
    this->setPlaylistVideoId(video->playlistVideoId());
    this->setUrl(video->url());
    this->setEmbedUrl(video->embedUrl());
    this->setTitle(video->title());
    this->setArtist(video->artist());
    this->setArtistId(video->artistId());
    this->setArtistAvatarUrl(video->artistAvatarUrl());
    this->setDescription(video->description());
    this->setDuration(video->duration());
    this->setDate(video->date());
    this->setTags(video->tags());
    this->setLikes(video->likes());
    this->setDislikes(video->dislikes());
    this->setRatingCount(video->ratingCount());
    this->setRating(video->rating());
    this->setViewCount(video->viewCount());
    this->setThumbnailUrl(video->thumbnailUrl());
    this->setLargeThumbnailUrl(video->largeThumbnailUrl());
    this->setSubtitlesUrl(video->subtitlesUrl());
    this->setLiked(video->liked());
    this->setDisliked(video->disliked());
    this->setFavourite(video->favourite());
    this->setDownloaded(video->downloaded());
    this->setPrivate(video->isPrivate());
    this->setService(video->service());

    switch (this->service()) {
    case Services::YouTube:
        this->connect(YouTube::instance(), SIGNAL(favouriteChanged(QString,bool,QString)), this, SLOT(onFavouriteChanged(QString,bool,QString)));
        this->connect(YouTube::instance(), SIGNAL(videoLiked(QString)), this, SLOT(onVideoLiked(QString)));
        this->connect(YouTube::instance(), SIGNAL(videoDisliked(QString)), this, SLOT(onVideoDisliked(QString)));
        this->connect(YouTube::instance(), SIGNAL(videoMetadataUpdated(QString,QVariantMap)), this, SLOT(onMetadataUpdated(QString,QVariantMap)));
        break;
    case Services::Dailymotion:
        this->connect(Dailymotion::instance(), SIGNAL(favouriteChanged(QString,bool)), this, SLOT(onFavouriteChanged(QString,bool)));
        this->connect(Dailymotion::instance(), SIGNAL(videoMetadataUpdated(QString,QVariantMap)), this, SLOT(onMetadataUpdated(QString,QVariantMap)));
        break;
    case Services::Vimeo:
        this->connect(Vimeo::instance(), SIGNAL(likeChanged(QString,bool)), this, SLOT(onFavouriteChanged(QString,bool)));
        break;
    default:
        break;
    }
}

void VideoItem::loadDailymotionVideo(const QVariantMap &video, bool favourite) {
    this->setId(video.value("id").toString());
    this->setVideoId(this->id());
    this->setFavouriteId(this->id());
    this->setPlaylistVideoId(this->id());
    this->setUrl(QUrl("http://dai.ly/" + this->videoId()));
    this->setEmbedUrl(QUrl("http://dailymotion.com/swf/video/" + this->videoId()));
    this->setTitle(video.value("title").toString());
    this->setArtist(video.value("owner.username").toString());
    this->setArtistId(video.value("owner.id").toString());
    this->setArtistAvatarUrl(video.value("owner.avatar_medium_url").toUrl());
    this->setDescription(video.value("description").toString());
    this->setDuration(video.value("duration").toInt());
    this->setDate(Utils::dateFromSecs(video.value("created_time").toLongLong()));
    this->setTags(video.value("tags").toStringList());
    this->setRatingCount(video.value("ratings_total").toInt());
    this->setRating(video.value("rating").toInt());
    this->setViewCount(video.value("views_total").toInt());
    this->setThumbnailUrl(video.value("thumbnail_medium_url").toUrl());
    this->setLargeThumbnailUrl(video.value("thumbnail_large_url").toUrl());
    this->setFavourite(favourite);
    this->setDownloaded(false);
    this->setPrivate(video.value("private").toBool());
    this->setService(Services::Dailymotion);
    this->connect(Dailymotion::instance(), SIGNAL(favouriteChanged(QString,bool)), this, SLOT(onFavouriteChanged(QString,bool)));
    this->connect(Dailymotion::instance(), SIGNAL(videoMetadataUpdated(QString,QVariantMap)), this, SLOT(onMetadataUpdated(QString,QVariantMap)));
}

void VideoItem::loadYouTubeVideo(const QDomNode &video, bool favourite) {
    this->setId(video.firstChildElement("id").text());
    this->setVideoId(video.firstChildElement("media:group").firstChildElement("yt:videoid").text());
    this->setFavouriteId(this->id().section(':', -1));
    this->setPlaylistVideoId(this->favouriteId());
    this->setUrl(QUrl("http://youtu.be/" + this->videoId()));
    this->setEmbedUrl(QUrl("http://www.youtube.com/e/" + this->videoId()));
    this->setTitle(video.firstChildElement("media:group").firstChildElement("media:title").text());
    this->setArtist(video.firstChildElement("media:group").firstChildElement("media:credit").attribute("yt:display"));
    this->setArtistId(video.firstChildElement("media:group").firstChildElement("yt:uploaderId").text().section("UC", 1, -1));
    this->setArtistAvatarUrl(QUrl(QString("http://i2.ytimg.com/i/%1/1.jpg").arg(this->artistId())));
    this->setDescription(video.firstChildElement("media:group").firstChildElement("media:description").text());
    this->setDuration(video.firstChildElement("media:group").firstChildElement("yt:duration").attribute("seconds").toInt());
    this->setDate(Utils::localDateTimeFromString(video.firstChildElement("media:group").firstChildElement("yt:uploaded").text()));
    this->setLikes(video.firstChildElement("yt:rating").attribute("numLikes").toInt());
    this->setDislikes(video.firstChildElement("yt:rating").attribute("numDislikes").toInt());
    this->setRatingCount(this->likes() + this->dislikes());
    this->setRating(this->likes() == 0 ? 0 : this->dislikes() == 0 ? 100 : this->likes() * 100 / this->ratingCount());
    this->setViewCount(video.firstChildElement("yt:statistics").attribute("viewCount").toInt());
    this->setThumbnailUrl(QUrl(video.firstChildElement("media:group").firstChildElement("media:thumbnail").attribute("url")));
    this->setLargeThumbnailUrl(QUrl(this->thumbnailUrl().toString().section('/', 0, -2).append("/hqdefault.jpg")));
    this->setFavourite(favourite);
    this->setDownloaded(false);
    this->setPrivate(!video.firstChildElement("yt:private").isNull());
    this->setService(Services::YouTube);

    QDomElement tagElement = video.firstChildElement("category");

    while (!tagElement.isNull()) {
        m_tags << tagElement.attribute("term");
        tagElement = tagElement.nextSiblingElement("category");
    }

    this->connect(YouTube::instance(), SIGNAL(favouriteChanged(QString,bool,QString)), this, SLOT(onFavouriteChanged(QString,bool,QString)));
    this->connect(YouTube::instance(), SIGNAL(videoLiked(QString)), this, SLOT(onVideoLiked(QString)));
    this->connect(YouTube::instance(), SIGNAL(videoDisliked(QString)), this, SLOT(onVideoDisliked(QString)));
    this->connect(YouTube::instance(), SIGNAL(videoMetadataUpdated(QString,QVariantMap)), this, SLOT(onMetadataUpdated(QString,QVariantMap)));
}

void VideoItem::loadVimeoVideo(const QDomElement &video) {
    this->setId(video.attribute("id"));
    this->setVideoId(this->id());
    this->setFavouriteId(this->id());
    this->setPlaylistVideoId(this->id());
    this->setUrl(QUrl("http://vimeo.com/" + this->videoId()));
    this->setEmbedUrl(QUrl("http://player.vimeo.com/" + this->videoId()));
    this->setTitle(video.firstChildElement("title").text());
    this->setDescription(video.firstChildElement("description").text());
    this->setDuration(video.firstChildElement("duration").text().toInt());
    this->setDate(Utils::localDateTimeFromString(video.firstChildElement("upload_date").text()));
    this->setLikes(video.firstChildElement("number_of_likes").text().toInt());
    this->setRatingCount(this->likes());
    this->setRating(this->likes());
    this->setViewCount(video.firstChildElement("number_of_plays").text().toInt());
    this->setFavourite(video.attribute("is_like").toInt());
    this->setDownloaded(false);
    this->setPrivate(video.attribute("privacy") != "anybody");
    this->setService(Services::Vimeo);

    QDomElement artistNode = video.firstChildElement("owner");

    this->setArtist(artistNode.attribute("display_name"));
    this->setArtistId(artistNode.attribute("id"));

    QDomNodeList avatars = artistNode.firstChildElement("portraits").childNodes();

    if (avatars.size() > 1) {
        this->setArtistAvatarUrl(QUrl(avatars.at(1).toElement().text()));
    }

    QDomNodeList thumbnails = video.firstChildElement("thumbnails").childNodes();

    if (thumbnails.size() > 2) {
        this->setThumbnailUrl(QUrl(thumbnails.at(1).toElement().text()));
        this->setLargeThumbnailUrl(QUrl(thumbnails.at(2).toElement().text()));
    }

    QDomNodeList tags = video.firstChildElement("tags").childNodes();

    for (int i = 0; i < tags.size(); i++) {
        m_tags << tags.at(i).toElement().text();
    }

    this->connect(Vimeo::instance(), SIGNAL(likeChanged(QString,bool)), this, SLOT(onFavouriteChanged(QString,bool)));
}

void VideoItem::onFavouriteChanged(const QString &videoId, bool favourite, const QString &favouriteId) {
    if ((videoId == this->videoId()) || (videoId == this->favouriteId())) {
        this->setFavourite(favourite);
        this->setFavouriteId(favouriteId);

        if (this->service() == Services::Vimeo) {
            this->setLikes(favourite ? this->likes() + 1 : this->likes() - 1);
        }
    }
}

void VideoItem::onVideoLiked(const QString &videoId) {
    if (videoId == this->videoId()) {
        this->like();
    }
}

void VideoItem::onVideoDisliked(const QString &videoId) {
    if (videoId == this->videoId()) {
        this->dislike();
    }
}

void VideoItem::onMetadataUpdated(const QString &videoId, const QVariantMap &metadata) {
    if (videoId == this->videoId()) {
        this->setTitle(metadata.value("title").toString());
        this->setDescription(metadata.value("description").toString());
        this->setTags(metadata.value("tags").toStringList());
        this->setPrivate(metadata.value("isPrivate").toBool());
    }
}
