
#include <clutter/clutter.h>
#include <clutter-cairo.h>
#include <math.h>

#define N_BUBBLES 50
#define BUBBLE_R 128
#define SCREEN_W 640
#define SCREEN_H 480

static ClutterActor *
create_bubble ()
{
  cairo_t *cr;
  cairo_pattern_t *pattern;
  ClutterActor *bubble;
  
  bubble = clutter_cairo_new (BUBBLE_R*2, BUBBLE_R*2);
  
  cr = clutter_cairo_create (CLUTTER_CAIRO (bubble));
  
  cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
  cairo_paint (cr);
  cairo_set_operator (cr, CAIRO_OPERATOR_ADD);
  
  cairo_arc (cr, BUBBLE_R, BUBBLE_R, BUBBLE_R, 0.0, 2*M_PI);
  
  pattern = cairo_pattern_create_radial (BUBBLE_R, BUBBLE_R, 0,
                                         BUBBLE_R, BUBBLE_R, BUBBLE_R);
  cairo_pattern_add_color_stop_rgba (pattern, 0, 0.88, 0.95, 0.99, 0.1);
  cairo_pattern_add_color_stop_rgba (pattern, 0.6, 0.88, 0.95, 0.99, 0.1);
  cairo_pattern_add_color_stop_rgba (pattern, 0.8, 0.67, 0.83, 0.91, 0.2);
  cairo_pattern_add_color_stop_rgba (pattern, 0.9, 0.5, 0.67, 0.88, 0.7);
  cairo_pattern_add_color_stop_rgba (pattern, 1.0, 0.3, 0.43, 0.69, 0.8);
  
  cairo_set_source (cr, pattern);
  cairo_fill_preserve (cr);
  
  cairo_pattern_destroy (pattern);
  
  pattern = cairo_pattern_create_linear (0, 0, BUBBLE_R*2, BUBBLE_R*2);
  cairo_pattern_add_color_stop_rgba (pattern, 0.0, 1.0, 1.0, 1.0, 0.0);
  cairo_pattern_add_color_stop_rgba (pattern, 0.15, 1.0, 1.0, 1.0, 0.95);
  cairo_pattern_add_color_stop_rgba (pattern, 0.3, 1.0, 1.0, 1.0, 0.0);
  cairo_pattern_add_color_stop_rgba (pattern, 0.7, 1.0, 1.0, 1.0, 0.0);
  cairo_pattern_add_color_stop_rgba (pattern, 0.85, 1.0, 1.0, 1.0, 0.95);
  cairo_pattern_add_color_stop_rgba (pattern, 1.0, 1.0, 1.0, 1.0, 0.0);

  cairo_set_source (cr, pattern);
  cairo_fill (cr);
  
  cairo_pattern_destroy (pattern);
  
  cairo_destroy (cr);
  
  return bubble;
}

static void
bubble_y_notify (ClutterActor     *actor,
                 GParamSpec       *pspec,
                 ClutterContainer *stage)
{
  gint size;
  gdouble *angular, *linear;
  
  if (clutter_actor_get_y (actor) > -(gint)clutter_actor_get_height (actor))
    return;
  
  size = g_random_int_range (BUBBLE_R/4, BUBBLE_R*2);
  clutter_actor_set_size (actor, size, size);
  clutter_actor_set_rotation (actor, CLUTTER_Z_AXIS,
                              g_random_double_range (0.0, 360.0),
                              size/2, size/2, 0);
  clutter_actor_set_position (actor,
                              g_random_int_range (0, SCREEN_W-BUBBLE_R),
                              g_random_int_range (SCREEN_H*2, SCREEN_H*3));
  clutter_actor_set_opacity (actor, g_random_int_range (80, 255));
  
  linear = g_object_get_data (G_OBJECT (actor), "linear");
  if (!linear)
    linear = g_slice_new (gdouble);
  *linear = g_random_double_range (0.5, 3.0);
  g_object_set_data (G_OBJECT (actor), "linear", linear);
  
  angular = g_object_get_data (G_OBJECT (actor), "angular");
  if (!angular)
    angular = g_slice_new (gdouble);
  *angular = g_random_double_range (-0.5, 0.5);
  g_object_set_data (G_OBJECT (actor), "angular", angular);
}

static void
timeline_new_frame_cb (ClutterTimeline *timeline,
                       gint             frame_num,
                       ClutterActor    *actor)
{
  gint radius;
  guint delta = clutter_timeline_get_delta (timeline, NULL);
  gdouble *linear = g_object_get_data (G_OBJECT (actor), "linear");
  gdouble angular =
    *((gdouble *)g_object_get_data (G_OBJECT (actor), "angular"));
  
  angular *= (gdouble)delta;

  clutter_actor_set_yu (actor, clutter_actor_get_yu (actor) -
                        (CLUTTER_UNITS_FROM_FLOAT (*linear) * delta));
  clutter_actor_set_xu (actor, clutter_actor_get_xu (actor) +
                        (CLUTTER_UNITS_FROM_FLOAT (angular)));

  angular += clutter_actor_get_rotation (actor, CLUTTER_Z_AXIS,
                                         NULL, NULL, NULL);
  while (angular > 360)
    angular -= 360.0;
  while (angular < 0)
    angular += 360.0;
  
  radius = clutter_actor_get_width (actor)/2;
  clutter_actor_set_rotation (actor, CLUTTER_Z_AXIS,
                              angular, radius, radius, 0);
}

int
main (int argc, char **argv)
{
  const ClutterColor bg_color = { 0xe0, 0xf2, 0xfc, 0xff };
  ClutterActor *stage, *bubble;
  ClutterTimeline *timeline;
  gint i;
  
  clutter_init (&argc, &argv);
  
  stage = clutter_stage_get_default ();
  clutter_actor_set_size (stage, SCREEN_W, SCREEN_H);
  clutter_stage_set_color (CLUTTER_STAGE (stage), &bg_color);
  
  timeline = clutter_timeline_new_for_duration (1000);
  clutter_timeline_set_loop (timeline, TRUE);

  bubble = create_bubble ();
  for (i = 0; i < N_BUBBLES; i++)
    {
      ClutterActor *actor;
      
      actor = clutter_clone_texture_new (CLUTTER_TEXTURE (bubble));
      
      clutter_actor_set_position (actor, i * BUBBLE_R*2, -BUBBLE_R*2);
      clutter_container_add_actor (CLUTTER_CONTAINER (stage), actor);
      
      g_signal_connect (actor, "notify::y",
                        G_CALLBACK (bubble_y_notify), stage);
      g_signal_connect (timeline, "new-frame",
                        G_CALLBACK (timeline_new_frame_cb), actor);  
      
      bubble_y_notify (actor, NULL, CLUTTER_CONTAINER (stage));
    }
  g_object_unref (bubble);
  
  clutter_actor_show_all (stage);
  
  clutter_timeline_start (timeline);
  
  clutter_main ();
  
  return 0;
}

