/*
 *  Copyright 2010 Ruediger Gad
 *
 *  This file is part of Binary Clock.
 *
 *  Binary Clock is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Binary Clock is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Binary Clock.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QCoreApplication>
#include <QSettings>

#include "mainwidget.h"
#include "settingsdialog.h"

MainWidget::MainWidget(QWidget *parent) :
    QWidget(parent)
{
    clockWidget = NULL;

    QCoreApplication::setOrganizationName("Binary Clock");
    QCoreApplication::setOrganizationDomain("https://garage.maemo.org/projects/binaryclock");
    QCoreApplication::setApplicationName("Binary Clock");

    setAttribute(Qt::WA_TranslucentBackground);

    layout = new QVBoxLayout();
    layout->setSpacing(0);
    layout->setMargin(0);
    setLayout(layout);

    QSettings settings;

    setClock(settings.value(SettingsDialog::CLOCK_STYLE,
                            SettingsDialog::CLOCK_STYLE_DEFAULT).toInt());
    clockWidget->setLedColor(settings.value(SettingsDialog::CLOCK_LED_COLOR,
                                            SettingsDialog::CLOCK_LED_COLOR_DEFAULT).value<QColor>());

    layout->addWidget(clockWidget);
#ifndef Q_WS_MAEMO_5
    connect(clockWidget, SIGNAL(clicked()), this, SLOT(switchClockStyle()));
#endif
}

MainWidget::~MainWidget(){
    delete clockWidget;
    delete layout;
}

void MainWidget::mouseReleaseEvent(QMouseEvent */*event*/){
    emit clicked();
}

void MainWidget::openSettingsDialog(){
    SettingsDialog dialog;

    if(dialog.exec()){
        QSettings settings;
        setClock(settings.value(SettingsDialog::CLOCK_STYLE,
                                SettingsDialog::CLOCK_STYLE_DEFAULT).toInt());
        clockWidget->setLedColor(settings.value(SettingsDialog::CLOCK_LED_COLOR,
                                                SettingsDialog::CLOCK_LED_COLOR_DEFAULT).value<QColor>());
    }
}

void MainWidget::paintEvent(QPaintEvent */*event*/){
    QPainter painter(this);
    painter.setRenderHint(QPainter::Antialiasing, true);
    painter.setBrush(QColor(255, 255, 255, 128));
    painter.setPen(Qt::NoPen);
    painter.drawRoundedRect(rect(), 25, 25);
    painter.end();
}

void MainWidget::setClock(int clockStyle){
    if(clockWidget != NULL){
#ifndef Q_WS_MAEMO_5
    disconnect(clockWidget, SIGNAL(clicked()), this, SLOT(switchClockStyle()));
#endif
        delete clockWidget;
    }

    switch(clockStyle){
    case 0:
        clockWidget = new BinaryClockWidget();
        break;
    case 1:
        clockWidget = new BcdClockWidget();
        break;
    default:
        clockWidget = new BinaryClockWidget();
        break;
    }

    setFixedSize(clockWidget->sizeHint());
    layout->addWidget(clockWidget);

#ifndef Q_WS_MAEMO_5
    connect(clockWidget, SIGNAL(clicked()), this, SLOT(switchClockStyle()));
#endif
}

void MainWidget::switchClockStyle(){
    currentClockStyle = ++currentClockStyle % 2;
    setClock(currentClockStyle);
}
