/*
 *  Copyright 2010 Ruediger Gad
 *
 *  This file is part of Binary Clock.
 *
 *  Binary Clock is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Binary Clock is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Binary Clock.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QPainterPath>

#include "binarydisplaywidget.h"

BinaryDisplayWidget::BinaryDisplayWidget(int bits, int value, int height, QWidget *parent) :
    QWidget(parent)
{
    this->bits = bits;
    this->value = value;
    baseSize = DEFAULT_BASE_SIZE;
    border = DEFAULT_BORDER;

    if(height != 0){
        this->height = height;
    }else{
        this->height = bits;
    }

    ledColor = QColor(DEFAULT_LED_COLOR);
    ledPen = QPen();
    ledPen.setWidth(DEFAULT_LED_PEN_WIDTH);
}

void BinaryDisplayWidget::drawLed(int x, int y, bool on, QPainter *painter){
    QRadialGradient gradient(QPointF(x + (baseSize * 0.5),
                                     y + (baseSize * 0.5)),
                             baseSize * 0.5);
    gradient.setFocalPoint(x + (baseSize * 0.42),
                           y + (baseSize * 0.4));
    gradient.setColorAt(0, QColor(Qt::white));

    if(on){
        gradient.setColorAt(0.2, ledColor);
        gradient.setColorAt(0.55, ledColor);
        gradient.setColorAt(1, QColor(Qt::black));
    }else{
        gradient.setColorAt(0.2, QColor(Qt::black));
    }

    QPainterPath path;
    path.arcTo(x + (border * 0.5),
               y + (border * 0.5),
               baseSize - border,
               baseSize - border,
               0, 5760);
    painter->fillPath(path, gradient);

    painter->setPen(ledPen);
    painter->drawArc(x + (border * 0.5),
                     y + (border * 0.5),
                     baseSize - border,
                     baseSize - border,
                     0, 5760);
}

int BinaryDisplayWidget::getBaseSize(){
    return baseSize;
}

void BinaryDisplayWidget::paintEvent(QPaintEvent */*event*/){
    QPainter painter(this);

    painter.setRenderHint(QPainter::Antialiasing, true);
    painter.setBackgroundMode(Qt::TransparentMode);

    draw(&painter);

    painter.end();
}

void BinaryDisplayWidget::updateValue(int value){
    if(this->value != value){
        this->value = value;
        emit update();
    }
}

void BinaryDisplayWidget::setBaseSize(int size){
    if(baseSize != size){
        baseSize = size;
        emit update();
    }
}

void BinaryDisplayWidget::setBorder(int border){
    if(this->border != border){
        this->border = border;
        emit update();
    }
}

void BinaryDisplayWidget::setLedColor(const QColor &color){
    ledColor = color;
    emit update();
}

void BinaryDisplayWidget::setLedPen(const QPen &pen){
    ledPen = pen;
    emit update();
}
