/**
 * Copyright (C) 2009 Floriano Scioscia.
 *
 * This file is part of 100 Boxes.
 *
 * 100 Boxes is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * 100 Boxes is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with 100 Boxes. If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <hildon/hildon.h>
#include <libosso.h>
#include <curl/curl.h>
#include <string.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include "sound.h"
#include "100boxes.h"

static gboolean box_clicked(GtkWidget *b, GdkEventButton *event, gpointer data);
static void new_clicked(GtkWidget *b, gpointer data);
static void guide_clicked(GtkWidget *b, gpointer data);
static void set_box_image(struct game_data *d, gint box_index, gint img_index);
static gboolean restore_box_image(gpointer data);
static gboolean show_end_dialog(gpointer data);
static gchar* create_end_message(gint score);
static void submit_score(struct game_data *d, gchar *name);
static size_t store_post_reply(void *buffer, size_t size, size_t nmemb, void *userp);
static void show_high_score(struct game_data *d);
static void show_error_message(struct game_data *d, const gchar *msg);

int main (int argc, char **argv) {
	osso_context_t *osso_context;
	HildonProgram *program;
	GtkWidget *win;
	GtkWidget *box_table;
	GtkWidget *box[100];
	GtkWidget *container;
	GtkWidget *panel;
	GtkWidget *new_button;
	GtkWidget *score_label;
	GtkWidget *guide_button;
	gint i,j;
	struct state s;
	struct game_data d;
	d.s = &s;
	d.base = box;
	gchar file[8];

	sound_init(&argc, &argv);

	// Initialize OSSO and Hildon
	osso_context = osso_initialize(OSSO_SERVICE, VERSION, TRUE, NULL);
	if (osso_context == NULL) {
        	return OSSO_ERROR;
	}
	hildon_gtk_init(&argc, &argv);
	program = hildon_program_get_instance();

	// Create the main window
	win = hildon_window_new();
	gtk_window_set_title(GTK_WINDOW(win), "100 Boxes");
	d.window = (GtkWindow*) win;

	// Create the box table
	box_table = gtk_table_new(10, 10, TRUE);
	gtk_widget_set_size_request(GTK_WIDGET(box_table), HEIGHT, -1);
	for (i=0; i<10; i++) {
		for (j=0; j<10; j++) {
			box[10*i+j] = gtk_event_box_new();
			gtk_table_attach_defaults(GTK_TABLE(box_table), GTK_WIDGET(box[10*i+j]), j, j+1, i, i+1);
			g_signal_connect(G_OBJECT(box[10*i+j]), "button_press_event", G_CALLBACK(box_clicked), (gpointer)&d);
		}
	}

	// Create other GUI elements
	container = gtk_hbox_new(FALSE, PADDING);
	panel = gtk_vbox_new(TRUE, 2*PADDING);
	gtk_widget_set_size_request(GTK_WIDGET(panel), WIDTH-HEIGHT-PADDING, -1);

	new_button = gtk_button_new_with_label("New game");
	hildon_helper_set_logical_font(GTK_WIDGET(new_button), "LargeSystemFont");
	g_signal_connect(GTK_WIDGET(new_button), "clicked", G_CALLBACK(new_clicked), (gpointer)&d);
	gtk_container_add(GTK_CONTAINER(panel), GTK_WIDGET(new_button));

	score_label = gtk_label_new("Score: 0");
	hildon_helper_set_logical_font(GTK_WIDGET(score_label), "LargeSystemFont");
	gtk_container_add(GTK_CONTAINER(panel), GTK_WIDGET(score_label));
	d.score = score_label;

	guide_button = gtk_button_new_with_label("Guide");
	//hildon_helper_set_logical_font(GTK_WIDGET(guide_button), "SmallSystemFont");
	g_signal_connect(GTK_WIDGET(guide_button), "clicked", G_CALLBACK(guide_clicked), (gpointer)&d);
	gtk_container_add(GTK_CONTAINER(panel), GTK_WIDGET(guide_button));

	new_game(&d);

	// Assemble GUI elements
	gtk_container_add(GTK_CONTAINER(container), GTK_WIDGET(box_table));
	gtk_container_add(GTK_CONTAINER(container), GTK_WIDGET(panel));
	gtk_container_add(GTK_CONTAINER(win), GTK_WIDGET(container));
	g_signal_connect(win, "destroy", G_CALLBACK(gtk_main_quit), NULL);

	gtk_widget_show_all(win);
	gtk_main();

	// Deinitialize OSSO
	osso_deinitialize(osso_context);
	return 0;
}

static gboolean box_clicked(GtkWidget *b, GdkEventButton *event, gpointer data) {
	gint distx, disty, x, y, poss, n, r, c, old;
	gchar label[4], score_label[11];
	struct game_data *d = (struct game_data *)data;

	for (n = 0; n<100; n++) {
		if (b == d->base[n]) {
			break;
		}
	}
	r = n / 10;
	c = n % 10;
	if (d->s->active) {
		d->s->last = n;
		if ( d->s->b[n] == CLICKABLE) {
			play_sound(SOUND_CLICK);
			d->s->score++;
			d->s->b[n] = d->s->score;
			sprintf(label, "%d", d->s->score);
			set_box_image(d, n, d->s->score);
			sprintf(score_label, "Score: %d", d->s->score);
			gtk_label_set_text(GTK_LABEL(d->score), score_label);
			poss = 0;
			for (x=0; x<10; x++) {
				for (y=0; y<10; y++) {
					if (d->s->b[10*x+y] < 1 || d->s->b[10*x+y] > 100) {
						distx = x-r > 0 ? x-r : r-x;
						disty = y-c > 0 ? y-c : c-y;
						if ( (distx==3 && disty==0) || (disty==3 && distx==0) || (distx==2 && disty==2) ) {
							poss++;
							old = d->s->b[10*x+y];
							d->s->b[10*x+y] = CLICKABLE;
							if (old != CLICKABLE) {
								set_box_image(d, 10*x+y, CLICKABLE);
							}
						} else {
							old = d->s->b[10*x+y];
							d->s->b[10*x+y] = UNUSABLE;
							if (old != UNUSABLE) {
								set_box_image(d, 10*x+y, UNUSABLE);
							}
						}
					}
				}
			}
			// poss = 0; // DEBUG
			if (poss == 0) {
				d->s->active = FALSE;
				if (d->s->score < 100) {
					play_sound(SOUND_LOSE);
				} else {
					play_sound(SOUND_WIN);
				}
				g_timeout_add_seconds(2, show_end_dialog, (gpointer)d);
			}
		} else {
			play_sound(SOUND_ERROR);
			set_box_image(d, n, ERROR);
			d->s->active = FALSE;
			g_timeout_add_seconds(1, restore_box_image, (gpointer)d);
		}
	}
	return TRUE;
}

static void new_clicked(GtkWidget *b, gpointer data) {
	struct game_data *d = (struct game_data *)data;
	new_game(d);
}

static void guide_clicked(GtkWidget *b, gpointer data) {
	system("dbus-send --print-reply --dest=com.nokia.osso_browser /com/nokia/osso_browser/service com.nokia.osso_browser.open_new_window string:file:///usr/share/100boxes/index.html");
}

void new_game(struct game_data *d) {
	gint i;
	d->s->active = TRUE;
	d->s->score = 0;
	d->s->last = -1;
	play_sound(SOUND_START);
	for (i=0; i<100; i++) {
		d->s->b[i] = CLICKABLE;
		set_box_image(d, i, CLICKABLE);
	}
	gtk_label_set_text(GTK_LABEL(d->score), "Score: 0");
}

static void set_box_image(struct game_data *d, gint box_index, gint img_index) {
	gchar file[40];
	sprintf(file, "/usr/share/pixmaps/100boxes/%d.png", img_index);
	GtkWidget *img = gtk_image_new_from_file(file);
	GtkWidget *old = gtk_bin_get_child(GTK_BIN(d->base[box_index]));
	if (old != NULL) {
		gtk_widget_destroy(GTK_WIDGET(old));
	}
	gtk_container_add(GTK_CONTAINER(d->base[box_index]), GTK_WIDGET(img));
	gtk_widget_show(img);
}

static gboolean show_end_dialog(gpointer data) {
	gint ans, len;
	gchar *name, *s;
	GtkWidget *area;
	GtkWidget *comment;
	GtkWidget *icon;
	GtkWidget *caption;
	GtkWidget *label;
	GtkWidget *entry;
	GtkWidget *dialog;
	gchar file[40];
	gchar *message;
	struct game_data *d = (struct game_data *)data;
	
	dialog = gtk_dialog_new_with_buttons (NULL, GTK_WINDOW(d->window),
                                                  GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                                  GTK_STOCK_OK,
                                                  GTK_RESPONSE_ACCEPT,
                                                  GTK_STOCK_CANCEL,
                                                  GTK_RESPONSE_REJECT,
                                                  NULL);

	if (d->s->score < 100) {
		gtk_window_set_title(GTK_WINDOW(dialog), "You lose!");
	} else {
		gtk_window_set_title(GTK_WINDOW(dialog), "You win!");
	}
	area = gtk_dialog_get_content_area(GTK_DIALOG(dialog));

	comment = gtk_hbox_new(FALSE, 5);
	sprintf(file, "/usr/share/pixmaps/100boxes/%d.png", d->s->score);
	icon = gtk_image_new_from_file(file);
	gtk_container_add(GTK_CONTAINER(comment), GTK_WIDGET(icon));	
	message = create_end_message(d->s->score);
	caption = gtk_label_new(message);
	gtk_container_add(GTK_CONTAINER(comment), GTK_WIDGET(caption));
	gtk_container_add(GTK_CONTAINER(area), GTK_WIDGET(comment));		

	label = gtk_label_new("Enter your name:");
	gtk_container_add(GTK_CONTAINER(area), GTK_WIDGET(label));
	entry = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(entry), 12); // Limit of online DB field
	gtk_container_add(GTK_CONTAINER(area), GTK_WIDGET(entry));	
	
	gtk_widget_show_all(GTK_WIDGET(dialog));
	ans = gtk_dialog_run(GTK_DIALOG(dialog));
	s = (gchar*) gtk_entry_get_text(GTK_ENTRY(entry));
	len = strlen(s);
	if (len > 0) {
		name = (gchar*) g_malloc(len+1);
		strcpy(name, s);
	}
	gtk_widget_destroy(dialog);
	g_free(message);
	switch (ans) {
		case GTK_RESPONSE_ACCEPT:
			if (len > 0) {
				submit_score(d, name);
			}
         		break;
		default:
			break;
	}
	return FALSE; // no repeat
}

static gchar* create_end_message(gint score) {
	gint i;
	gint l;
	gint n = 7;
	gint s[] = {40, 60, 75, 90, 97, 99, 100};
	gchar* m[] = { 	"You can do better than this!", 
			"Train some more to improve.",
			"Nice try, the basics are there.",
			"You are approaching the solution.",
			"Good, you are near the goal!",
			"Oh, you were so close!",
			"You are a 100 Boxes champion!" };
	gchar *msg;
	for (i=0; i<n; i++) {
		if (score <= s[i]) {
			l = strlen(m[i]);
			msg = (gchar*) g_malloc(l + 1);
			strcpy(msg, m[i]);
			break;
		}
	}
	return msg;
}

static gboolean restore_box_image(gpointer data) {
	struct game_data *d = (struct game_data *)data;
	if (d->s->score > 0) {	// Cannot restore if game not started
		gint n = d->s->last;
		set_box_image(d, n, d->s->b[n]);
		d->s->active = TRUE;
	}
	return FALSE; // no repeat		
}

static void submit_score(struct game_data *d, gchar *name) {
	CURLcode res;
	gchar post[MAX_QUERY_LENGTH+1];

	d->curl = curl_easy_init();
	d->post_reply = NULL;

	if (d->curl) {
		// POST data: show_scores callback function will receive the reply
		sprintf(post, "n=%s&s=%d&src=%s", name, d->s->score, SCORE_SRC);
		//g_debug("POST:\n%s", post);
		curl_easy_setopt(d->curl, CURLOPT_VERBOSE, 1);		// Set POST query
		curl_easy_setopt(d->curl, CURLOPT_POST, 1);		// Set POST method
		curl_easy_setopt(d->curl, CURLOPT_POSTFIELDS, post);	// Set POST data
		curl_easy_setopt(d->curl, CURLOPT_URL, POST_URL);	// Set POST URL
		curl_easy_setopt(d->curl, CURLOPT_WRITEFUNCTION, store_post_reply); // Set callback function
		curl_easy_setopt(d->curl, CURLOPT_WRITEDATA, d);	// Set data passed to callback
		res = curl_easy_perform(d->curl); 			// Do the POST
		if (res == 0) { // success
			//g_debug("REPLY:\n%s", d->post_reply);
			show_high_score(d);
		} else {
			show_error_message(d, "Submission failed.");
		}
		curl_easy_cleanup(d->curl);
	} else {
		show_error_message(d, "Connection failed.");
	}
}

static size_t store_post_reply(void *buffer, size_t size, size_t nmemb, void *userp) {
	struct game_data *d = (struct game_data *)userp;
	gchar *part = (gchar *)buffer;
	size_t len = size * nmemb;
	//g_debug("In store_post_reply. Size=%d:", len);
	//g_debug("In store_post_reply. post_reply=%s:", d->post_reply);
	if (d->post_reply == NULL) {
		d->post_reply = (gchar*) g_malloc(len + 1);
		strcpy(d->post_reply, part);
	} else {
		d->post_reply = (gchar*) g_realloc(d->post_reply, strlen(d->post_reply) + len);
		strcat(d->post_reply, part);
	}
	//g_debug("Exiting store_post_reply. post_reply=%s:", d->post_reply);
	return len;
}

static void show_error_message(struct game_data *d, const gchar *msg) {
	GtkWidget *note = hildon_note_new_information(d->window, msg);
	gtk_dialog_run(GTK_DIALOG(note));
	gtk_widget_destroy(note);			
}

static void show_high_score(struct game_data *d) {
	xmlDoc *doc;
	xmlNode *root, *pos, *tot, *top, *rec;
	gchar *line, *pos_v, *tot_v;
	gchar *n_v[10], *name_v[10], *score_v[10], *date_v[10];
	gint i, j;
	GtkWidget *text_l, *n_h, *name_h, *score_h, *date_h;
	GtkWidget *n_l[10], *name_l[10], *score_l[10], *date_l[10];
	GtkWidget *area;
	GtkWidget *table;
	GtkWidget *dialog;

	doc = xmlReadDoc((xmlChar*)d->post_reply, NULL, NULL, 0);
	if (doc == NULL) {
		show_error_message(d, "Bad response received");
	} else {
		dialog = gtk_dialog_new();
		dialog = gtk_dialog_new_with_buttons ("High scores", GTK_WINDOW(d->window),
                                                  GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                                  GTK_STOCK_OK,
                                                  GTK_RESPONSE_CLOSE,
                                                  NULL);

		area = gtk_dialog_get_content_area(GTK_DIALOG(dialog));

		//g_debug("Getting root XML element");
		root = xmlDocGetRootElement(doc);
		pos = root->children;
		pos_v = (gchar*) pos->children->content;
		tot = pos->next;
		tot_v = (gchar*) tot->children->content;
		top = tot->next;

		line = (gchar*) g_malloc(100);
		sprintf(line, "<big>You got place no. <b>%s</b> of <b>%s</b>!</big>", pos_v, tot_v);

		text_l = gtk_label_new("");
 		gtk_container_add(GTK_CONTAINER(area), GTK_WIDGET(text_l));
		gtk_label_set_markup(GTK_LABEL(text_l), line);

		// Extract top scores
		for (rec=top->children, i=0; rec!=NULL && i<10; rec = rec->next, i++) {
			n_v[i] = (gchar*) g_malloc(3);
			sprintf(n_v[i], "%d", (i+1));
			name_v[i] = xmlGetProp(rec, "name");
			score_v[i] = xmlGetProp(rec, "score");
			date_v[i] = xmlGetProp(rec, "date");
		}

		// Create table and header row
		table = gtk_table_new(4, i+1, FALSE);
 		gtk_container_add(GTK_CONTAINER(area), GTK_WIDGET(table));
		n_h = gtk_label_new("");
 		gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(n_h), 0, 1, 0, 1);
		gtk_label_set_markup(GTK_LABEL(n_h), "<b>Place</b>");
		name_h = gtk_label_new("");
 		gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(name_h), 1, 2, 0, 1);
		gtk_label_set_markup(GTK_LABEL(name_h), "<b>Name</b>");
		score_h = gtk_label_new("");
 		gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(score_h), 2, 3, 0, 1);
		gtk_label_set_markup(GTK_LABEL(score_h), "<b>Score</b>");
		date_h = gtk_label_new("");
 		gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(date_h), 3, 4, 0, 1);
		gtk_label_set_markup(GTK_LABEL(date_h), "<b>Date</b>");

		// Create data rows
		for (j=0; j<i; j++) {
			n_l[j] = gtk_label_new(n_v[j]);
			gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(n_l[j]), 0, 1, j+1, j+2);
			name_l[j] = gtk_label_new(name_v[j]);
			gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(name_l[j]), 1, 2, j+1, j+2);
			score_l[j] = gtk_label_new(score_v[j]);
			gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(score_l[j]), 2, 3, j+1, j+2);
			date_l[j] = gtk_label_new(date_v[j]);
			gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(date_l[j]), 3, 4, j+1, j+2);
		}

		gtk_widget_show_all(GTK_WIDGET(dialog));
		gtk_dialog_run(GTK_DIALOG(dialog));

		// Cleanup
		gtk_widget_destroy(GTK_WIDGET(dialog));
		g_free(line);
		for (j=0; j<i; j++) {
			g_free(n_v[j]);
		}
		xmlFreeDoc(doc);
	}	
	xmlCleanupParser();
	g_free(d->post_reply);
	d->post_reply = NULL;
}
