import gnome.gconf as gconf
import gtk, gobject

from wifi_assistant.settings.network_setting import NetworkSetting


class NetworkSettings(object):
    '''
    Reads and Writes NetworkSettings from/to GConf.
    '''

    __DEFAULT_SETTINGS = 'default_settings'
    
    def __init__(self, gconf_client, root_path):
        self._gc = gconf_client
        self._root_path = root_path
        
    
    def delete(self, network_setting):
        key = self._assembleRootKeyForSsid(network_setting.getNetworkName())
        self._recursiveUnset(key)
        
        
    def getDefaultSettings(self):
        settings = self.get(self.__DEFAULT_SETTINGS)
        if settings is None:
            settings = NetworkSetting()
            settings.setNetworkName(self.__DEFAULT_SETTINGS)
            settings.setLaunchingOfBrowserEnabled(True)
            settings.setBrowserOptions({'url':'http://wifi-assistant.garage.maemo.org/'})
            self.save(settings)
        return settings
    

    def getListStore(self):
        """Returns a ListStore with one Row per Network Setting"""

        store = gtk.ListStore(str, gobject.TYPE_BOOLEAN)
        
        settings = self._getAllNetworkSettings()
        for setting in settings:
            store.append([setting.getNetworkName(), setting.getLaunchingOfBrowserEnabled()])

        # sorting
        store.set_sort_column_id(0, gtk.SORT_ASCENDING)

        return store


    def get(self, ssid):
        key = self._assembleRootKeyForSsid(ssid)
        if self._gc.dir_exists(key):
            return NetworkSetting(self._loadValuesFromGConf(key))
        print "WARNING: No settings to return for network name", ssid
        return None
    
    
    def save(self, network_setting):
        self._saveValuesToGConf(network_setting)
    
    
    def _assembleRootKeyForSsid(self, ssid):
        safe = self._replaceForbiddenCharacters(ssid)
        return self._root_path + '/' + safe
    
    
    def _getAllNetworkSettings(self):
        dirs = self._gc.all_dirs(self._root_path)
        settings = []
        key_offset = len(self._root_path) + 1
        for dir in dirs:
            key = dir[key_offset:]
            if self.__DEFAULT_SETTINGS != key:
                settings.append(self.get(key))
        return settings
    
    
    def _loadValuesFromGConf(self, path):
        """Loads all values under a given path in gconf recursively into a dict"""
        
        values = {}
        path_length = len(path) + 1 # remove trailing / too
        
        dirs = self._gc.all_dirs(path)
        for sub_dir_path in dirs:
            key = sub_dir_path[path_length:]
            values[key] = self._loadValuesFromGConf(sub_dir_path)
        
        entries = self._gc.all_entries(path)
        for entry in entries:
            full_key = entry.get_key()
            key = full_key[path_length:]
            gvalue = entry.get_value()
            if gvalue.type == gconf.VALUE_BOOL:
                values[key] = gvalue.get_bool()
            elif gvalue.type == gconf.VALUE_STRING:
                values[key] = gvalue.get_string()
            else:
                print 'ga'
        
        return values
    
    
    def _recursiveUnset(self, dir):
        # there's no recursive_unset available in gconf so we'll have to do it ourselves
        all_entries = self._gc.all_entries(dir)
        for entry in all_entries:
            self._gc.unset(entry.get_key())
        for sub_dir in self._gc.all_dirs(dir):
            self._recursiveUnset(sub_dir)
        self._gc.suggest_sync()

    
    def _replaceForbiddenCharacters(self, str):
        allowed = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-_'
        result = ''
        for c in str:
            if c in allowed:
                result = result + c
            else:
                result = result + '__'
        return result
    
    
    def _saveValuesToGConf(self, network_setting):
        """Saves all values "recursively" from NetworkSetting"""
        
        network_name = network_setting.getNetworkName()
        key = self._assembleRootKeyForSsid(network_name)
        browser = network_setting.getNameOfBrowserToLaunch()
        launch = network_setting.getLaunchingOfBrowserEnabled()
        options = network_setting.getBrowserOptions()
        
        self._gc.set_string(key + "/name", network_name)
        self._gc.set_string(key + '/browser', browser)
        self._gc.set_bool(key + '/launch', launch)
        for option_key in options:
            self._gc.set_string(key + "/options/" + option_key, options[option_key])
        