/*
** Copyright (c) 2009  Kimmo 'Rainy' Pekkola
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see http://www.gnu.org/licenses.
*/

#include "rotator.h"
#include "debugtext.h"
#include <QGraphicsSceneMouseEvent>
#include <QDebug>

#define DRAG_TIMEOUT_RATE 50
#define NUMBER_COUNT 5

//-----------------------------------------------------------------------------
/**
** Constructor.
**
** \param pParent The parent object.
*/
CRotator::CRotator(QObject* pParent) : QObject(pParent)
{
    m_Value = 0;
    m_MaxValue = 60;
    m_Offset = 0;
    m_bEditMode = false;
    m_DragStartOffset = 0;
    m_Speed = 0;

    bool bOk = false;
    bOk = connect(&m_DragTimer, SIGNAL(timeout()), this, SLOT(onTimeout()));
    Q_ASSERT(bOk);
}

//-----------------------------------------------------------------------------
/**
** Creates the number items for the rotator and adds them to the scene.
**
** \param pScene The scene where the numbers are added.
** \param rect The position of the rotator.
** \param pixmapNumbers The image containing the numbers.
*/
void CRotator::buildScene(QGraphicsScene* pScene, const QRect& rect, QPixmap pixmapNumbers)
{
    if (pScene)
    {
        int count = NUMBER_COUNT;
        int yStart = -count / 2 * NUMBER_ITEM_HEIGHT - NUMBER_ITEM_HEIGHT / 2;
        for (int i = 0; i < count; i++)
        {
            CNumberItem* pItem = new CNumberItem(this, rect.left(), yStart, false, pixmapNumbers);
            pScene->addItem(pItem);
            m_Numbers.append(pItem);
            yStart += NUMBER_ITEM_HEIGHT;
        }
    }
}

//-----------------------------------------------------------------------------
/**
** Changes the position of the rotator. This gets called when the window area changes.
** Repositions the number items.
**
** \param rect The new area for the rotator.
*/
void CRotator::adjustScene(const QRect& rect)
{
    int count = m_Numbers.size();
    int yStart = -count / 2 * NUMBER_ITEM_HEIGHT - NUMBER_ITEM_HEIGHT / 2;
    for (int i = 0; i < count; i++)
    {
        m_Numbers[i]->setPos(rect.left(), yStart);
        yStart += NUMBER_ITEM_HEIGHT;
    }
}

//-----------------------------------------------------------------------------
/**
** Enables the edit mode in all contained numbers. Sets the background visible
** and the top and bottom numbers.
**
** \param bEnable Set to true to enable, false to disable.
*/
void CRotator::enableEditMode(bool bEnable)
{
    m_bEditMode = bEnable;
    for (int i = 0; i < m_Numbers.size(); i++)
    {
        m_Numbers[i]->showBackground(bEnable);
        if (i - m_Numbers.size() / 2 != 0)
        {
            m_Numbers[i]->setVisible(bEnable);
        }
    }

    if (!bEnable)
    {
        m_Speed = 0;    // Stop rotation when start is clicked
    }
}

//-----------------------------------------------------------------------------
/**
** Sets the value. This just changes the offset to match the height of the number
** item times the value.
**
** \param value The new value which is shown as the central number.
*/
void CRotator::setValue(int value)
{
    // The offset changes also the value
    int offset = value * NUMBER_ITEM_HEIGHT;
    setOffset(offset);
}

//-----------------------------------------------------------------------------
/**
** Sets the new offset for the rotator. Changes the locations of the contained
** numbers with the offset. The number position changes just within the height
** and the value they contain is changed if the offset is greater than the height.
**
** \param offset The new offset for the rotator.
*/
void CRotator::setOffset(int offset)
{
    m_Offset = offset;
    m_Offset += NUMBER_ITEM_HEIGHT * m_MaxValue;
    m_Offset %= NUMBER_ITEM_HEIGHT * m_MaxValue;

    int remainder = offset % NUMBER_ITEM_HEIGHT;
    int value = offset / NUMBER_ITEM_HEIGHT;

    for (int i = 0; i < m_Numbers.size(); i++)
    {
        int index = i - m_Numbers.size() / 2;
        int v = value - index;
        v = (v + m_MaxValue) % m_MaxValue;
        m_Numbers[i]->setValue(v);

        m_Numbers[i]->setOffset(remainder);
    }
}

//-----------------------------------------------------------------------------
/**
** Calculates new speed for the rotator.
**
** \param pos The new mouse position.
*/
void CRotator::calculateSpeed(QPointF pos)
{
    int offset = pos.y() - m_DragStartPos.y();
    setOffset(m_DragStartOffset + offset);

    TimePos tp;
    tp.pos =pos;
    tp.time = QTime::currentTime();
    m_TimePosArray.append(tp);

    if (m_TimePosArray.size() > 3)
    {
        m_TimePosArray.removeFirst();
    }

    // Calculate the speed
    qreal elapsed = m_TimePosArray.first().time.msecsTo(m_TimePosArray.last().time);
    int dist = m_TimePosArray.last().pos.y() - m_TimePosArray.first().pos.y();

    m_Speed = dist / elapsed;
/*
    qDebug() << QTime::currentTime().toString("HH:mm:ss.zzz") <<
            QString("Mouse: %1,%2").arg(pos.x()).arg(pos.y()) <<
            QString("Speed: %1").arg(m_Speed) <<
            QString("Dist: %1").arg(dist) <<
            QString("Elapsed: %1").arg(elapsed);
*/
    DEBUGTEXT("Speed", QString("%1").arg(m_Speed));
    DEBUGTEXT("Offset", QString("%1").arg(m_Offset));
}

//-----------------------------------------------------------------------------
/**
** Handler for the mouse movement. Changes the speed accoring to the previous
** mouse position.
**
*/
void CRotator::mouseMoveEvent(QGraphicsSceneMouseEvent* pEvent)
{
    if (pEvent && m_bEditMode)
    {
        calculateSpeed(pEvent->pos());
    }
}

//-----------------------------------------------------------------------------
/**
** Handles the mouse press events. Stores the position where the mouse was pressed
** for the dragging.
*/
void CRotator::mousePressEvent(QGraphicsSceneMouseEvent* pEvent)
{
/*
    qDebug() << QTime::currentTime().toString("HH:mm:ss.zzz") <<
            QString("Mouse: %1,%2 (press)").arg(pEvent->pos().x()).arg(pEvent->pos().y()) <<
            QString("Speed: %1").arg(m_Speed);
*/
    if (pEvent && m_bEditMode)
    {
        m_DragStartOffset = m_Offset;
        m_DragStartPos = pEvent->pos();
        m_DragTimer.stop();

        m_TimePosArray.clear();
        TimePos tp;
        tp.pos = pEvent->pos();
        tp.time = QTime::currentTime();
        m_TimePosArray.append(tp);
    }
}

//-----------------------------------------------------------------------------
/**
** Handler for the mouse release events. Starts timer to adjust the speed
** of the rotator.
*/
void CRotator::mouseReleaseEvent(QGraphicsSceneMouseEvent* pEvent)
{
    if (pEvent && m_bEditMode)
    {
        if (m_TimePosArray.size() > 1)      // Filter events without any "Move"
        {
            calculateSpeed(pEvent->pos());
        }
        else if (m_TimePosArray.size() == 1 && m_TimePosArray[0].pos == pEvent->pos())
        {
            m_Speed = 0;    // Tap stops
        }
    }

    // Start the timer to align the items
    m_DragTimer.start(DRAG_TIMEOUT_RATE);
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the drag timer time outs. Adjusts the offset
** for the current speed. If the speed is low enough the offset is changed
** to the nearest item.
*/
void CRotator::onTimeout()
{
    DEBUGTEXT("Speed", QString("%1").arg(m_Speed));
    DEBUGTEXT("Offset", QString("%1").arg(m_Offset));

    if (m_Speed > 0.3 || m_Speed < -0.3)
    {
        // Add the speed to the offset to make the rotator go
        setOffset(m_Offset + m_Speed * DRAG_TIMEOUT_RATE);     // Speed is pixels / millisecond and the timer runs in every DRAG_TIMEOUT_RATE ms
//        m_Speed *= 0.95;    // Slow down
    }
    else
    {
        // If the speed is low enough just move to the nearest full item

        int remainder = m_Offset % NUMBER_ITEM_HEIGHT;

        if (remainder > 1 && remainder < NUMBER_ITEM_HEIGHT - 1)
        {
            // Prefer the scroll direction
            int limit = NUMBER_ITEM_HEIGHT / 2;
            if (m_Speed > 0)
            {
                limit -= NUMBER_ITEM_HEIGHT / 6;
            }
            else
            {
                limit += NUMBER_ITEM_HEIGHT / 6;
            }

            // Adjust the offset
            if (remainder > limit)
            {
                int adjust = (NUMBER_ITEM_HEIGHT - remainder) * 0.3;
                adjust = qMax(1, adjust);
                setOffset(m_Offset + adjust);
            }
            else
            {
                int adjust = remainder * 0.3;
                adjust = qMax(1, adjust);
                setOffset(m_Offset - adjust);
            }
        }
        else
        {
            // The offset is set for a full item
            int offset = m_Offset + 3;
            offset -= offset % NUMBER_ITEM_HEIGHT;
            setOffset(offset);
            m_DragTimer.stop();
        }
    }

    emit valueChanged((m_Offset + NUMBER_ITEM_HEIGHT / 2) / NUMBER_ITEM_HEIGHT);
}

// EOF
