/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

/* 
 * File:   Preferences.hpp
 * Author: ivan
 *
 * Created on April 22, 2009, 2:31 PM
 */

#pragma once

#include <string>
#include <ting/types.hpp>
#include <ting/Singleton.hpp>
#include <ting/Thread.hpp>
#include <ting/utils.hpp>


#include "utils.hpp"


inline unsigned DHighestOctave(){
	return 7;
}



class Preferences : public ting::Singleton<Preferences>{
	std::string prefsFileName;

public:
	Preferences(const std::string& fileName) :
			prefsFileName(fileName),
			
			serverPort(0),
			
			numOctaves(1),
			basePitch(12 * 3),//A3
			lockCurrentOctave(true),
			timbre(SINE_PLUS_ADSR),

			kbScrPos(14 * 3),// 3-rd octave
			kbTimbre(SINE_PLUS_ADSR),

			angTimbre(SINE_PLUS_ADSR),

			autologin(false),
			broadcast(false)
	{}

	void LoadFromFile();
	void SaveToFile();


	

	//Non-saveable settings
	std::string serverHost;
	ting::u16 serverPort;





	//Theremin settings
private:
	unsigned numOctaves;

	unsigned basePitch;//step number starting from A0 (27.5Hz)

	bool lockCurrentOctave;

private:
	E_Timbre timbre;
public:

	inline unsigned GetNumOctaves()const{
		return this->numOctaves;
	}

	void SetNumOctaves(unsigned numberOfOctaves){
		this->numOctaves = numberOfOctaves;
		this->ClampBasePitch();
	}

	inline unsigned GetBasePitch()const{
		return this->basePitch;
	}

	void SetBasePitch(unsigned newBasePitch){
		this->basePitch = newBasePitch;
		this->ClampBasePitch();
	}

	inline bool GetLockCurrentOctave()const{
		return this->lockCurrentOctave;
	}

	inline void SetLockCurrentOctave(bool lockOctave){
		this->lockCurrentOctave = lockOctave;
	}

	inline E_Timbre GetTimbre()const{
		return this->timbre;
	}

	inline void SetTimbre(E_Timbre newTimbre){
		this->timbre = newTimbre;
	}



private:
	void ClampBasePitch();





	//keyboard view settings
private:
	unsigned kbScrPos;//screen position

	E_Timbre kbTimbre;

public:
	inline unsigned GetKbScrPos()const{
		return this->kbScrPos;
	}

	void SetKbScrPos(unsigned newScrPos){
		this->kbScrPos = newScrPos;
	}

	inline E_Timbre GetKbTimbre()const{
		return this->kbTimbre;
	}

	inline void SetKbTimbre(E_Timbre newTimbre){
		this->kbTimbre = newTimbre;
	}






	//Anglephone view settings
private:
	E_Timbre angTimbre;

public:
	inline E_Timbre GetAngTimbre()const{
		return this->angTimbre;
	}

	inline void SetAngTimbre(E_Timbre newTimbre){
		this->angTimbre = newTimbre;
	}




	//Network settings
private:
	std::string username;
	std::string password;

	//username and password can be get and set in 2 different threads,
	//thus protect them by mutex, as it is not simple types.
	ting::Mutex usernamePasswordMutex;

public:
	const std::string GetUsername()const{
		ting::Mutex::Guard mutexGuard(const_cast<ting::Mutex&>(this->usernamePasswordMutex));
		return std::string(this->username.c_str());//return copy
	}

	void SetUsername(const std::string& un){
		ting::Mutex::Guard mutexGuard(this->usernamePasswordMutex);
		this->username = std::string(un.c_str());//copy
	}

	const std::string GetPassword()const{
		ting::Mutex::Guard mutexGuard(const_cast<ting::Mutex&>(this->usernamePasswordMutex));
		return std::string(this->password.c_str());//return copy
	}

	void SetPassword(const std::string& pw){
		ting::Mutex::Guard mutexGuard(this->usernamePasswordMutex);
		this->password = std::string(pw.c_str());//copy
	}



private:
	bool autologin;
	bool broadcast;

public:
	bool GetAutologin()const{
		return this->autologin;
	}

	void SetAutologin(bool al){
		this->autologin = al;
	}

	bool GetBroadcast()const{
		return this->broadcast;
	}

	void SetBroadcast(bool bc){
		this->broadcast = bc;
	}

	//~Network settings
};

