/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

/*
* aumiks 0.1
*/


#pragma once

#include <ting/debug.hpp>
#include <ting/math.hpp>

#include "config.hpp"


namespace aumiks{

/**
 * @brief discrete analog of simple RC high pass filter.
 * Calculation formula:
 *     y[i] = alfa * y[i-1] + alfa * (x[i] - x[i-1])
 * where
 *     alfa = R * C / (R * C + deltaT)   ,
 *     deltaT - discretization time step
 * corner frequency:
 *     f = 1 / (2 * pi * R * C)
 */
class FilterHighPass{
	aumiks::real alfa;
	aumiks::real xold;//holds x[i-1]
	aumiks::real yold;//holds y[i-1]
public:

	/**
	 * @brief create instance of filter.
	 * @param cornerFreq - requested corner frequency in rounds per sample
	 *                     (sample is a discretization time step).
	 */
	FilterHighPass(aumiks::real cornerFreq = 100){
		this->Init();
		this->SetCornerFrequency(cornerFreq);
	}

	void Init(){
		this->xold = 0;
		this->yold = 0;
	}

	/**
	 * @brief sets filter coffecients to maintain given corner frequency.
	 * @param cornerFreq - requested corner frequency in rounds per sample
	 *                     (sample is a discretization time step).
	 */
	void SetCornerFrequency(aumiks::real cornerFreq){
		aumiks::real rc = 1 / (ting::D2Pi<aumiks::real>() * cornerFreq);
		this->alfa = rc / (rc + 1);
	}

	inline aumiks::real Tick(aumiks::real sample){
		this->yold = this->alfa * (this->yold + sample - this->xold);
		this->xold = sample;
		return this->yold;
	}
};

}//~namespace
