/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

/*
* aumiks 0.1
*/

/*
 * STK library (http://ccrma.stanford.edu/software/stk/index.html) was used as
 * a reference while creating this class.
 */

#pragma once

#include <ting/Array.hpp>
#include <ting/debug.hpp>
#include <ting/math.hpp>
#include <ting/Exc.hpp>
#include <ting/types.hpp>

#include "config.hpp"

namespace aumiks{

class DelayAllPass1{
	ting::Array<aumiks::real> line;
	ting::uint inIndex, outIndex;

	aumiks::real delay;
	aumiks::real eta;
	aumiks::real delta;
	aumiks::real y;
	aumiks::real prevX;

	void SetMaxDelay(aumiks::real delayToSet){
		ASSERT(delayToSet > 0.5)
		this->line.Init(ting::uint(delayToSet + 1));
	}
public:
	//default constructor
	DelayAllPass1(){};

	/**
	@brief delay line constructor.
	@param maxDelay - maximum delay in samples.
	@param delayToSet - delay in samples, cannot be larger than maxDelay.
	*/
	DelayAllPass1(aumiks::real maxDelay, aumiks::real delayToSet = 0.5){
		this->Init(maxDelay, delayToSet);
	}

	void Init(aumiks::real maxDelay, aumiks::real delayToSet = 0.5){
		if(delayToSet > maxDelay)
			throw ting::Exc("DelayAllPass1: delay longer than maxDelay requested");

		if(delayToSet < 0.5)
			throw ting::Exc("DelayAllPass1: delay shorter than 0.5 samples requested");

		this->SetMaxDelay(maxDelay);
		this->inIndex = 0;
		this->Clear();
		this->SetDelay(delayToSet);
	}

	void Clear(){
		for(ting::uint i = 0; i < this->line.Size(); ++i)
			line[i] = 0;
		this->y = 0;
		this->prevX = 0;
	}

	inline real MaxDelay()const{
		return real(this->line.Size());
	}

	inline real Delay()const{
		return this->delay;
	}

	void SetDelay(aumiks::real delayToSet){
		aumiks::real outIdx;

		ASSERT(this->line.Size() > 0)

		ting::uint lineLength = this->line.Size() - 1;

		if(delayToSet > lineLength){
			ASSERT(this->inIndex < this->line.Size())
			//set maximum delay
			outIdx = this->inIndex + 1;
			this->delay = lineLength;
		}else if(delayToSet < 0.5){
			outIdx = aumiks::real(this->inIndex) + 0.4999999999;
			this->delay = 0.5;
		}else{
			outIdx = aumiks::real(this->inIndex) - delayToSet + 1;//outIndex chases inIndex
			this->delay = delayToSet;
		}

		if(outIdx < 0)
			outIdx += this->line.Size();

		this->outIndex = ting::uint(outIdx);//integer part of the delay

		this->delta = aumiks::real(1) + this->outIndex - outIdx;//fractional part of the delay

		if(this->delta < 0.5){
			// The optimal range for delta is about 0.5 - 1.5 in order to
			// achieve the flattest phase delay response.
			this->outIndex += 1;
			this->delta += 1;
		}

		if(this->outIndex >= this->line.Size())
			this->outIndex -= this->line.Size();

		//coefficient for all-pass
		this->eta = (aumiks::real(1) - this->delta) / (aumiks::real(1) + this->delta);
	}

	inline aumiks::real LastOutValue()const{
		return this->y;
	}

	inline aumiks::real Tick(aumiks::real sample){
		this->line[this->inIndex] = sample;
		++this->inIndex;
		if(this->inIndex == this->line.Size())
			this->inIndex = 0;

		//y[n] = eta * x[n] + x[n-1] - eta * y[n-1] =
		//     = eta * (x[n] - y[n-1]) + x[n-1]
		this->y = this->eta * (this->line[this->outIndex] - this->y) + this->prevX;

		// Save the allpass input and increment modulo length.
		this->prevX = this->line[this->outIndex];
		++this->outIndex;
		if(this->outIndex == this->line.Size())
			this->outIndex = 0;

		return this->y;
	}
};

}//~namespace
