/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

/*
 * aumiks 0.1
 * File:   Synth.hpp
 * Author: ivan
 *
 * Created on February 21, 2009, 1:50 PM
 */

#pragma once


#include <ting/Array.hpp>
#include <ting/types.hpp>
#include <ting/Fixed.hpp>

#include "../aumiks.hpp"

#include "PluckedString.hpp"

namespace aumiks{


using namespace ting;



class Instrument : public aumiks::Sound{
public:

	class Channel : public aumiks::Channel{
	protected:
		int lastVolume;

	public:
		inline Channel() :
				lastVolume(-1)
		{}

		virtual void SetFrequency(float freq){}
	};

	virtual Ref<Instrument::Channel> CreateChannel() = 0;
};


class PluckedStringInstrument : public Instrument{
public:
	class Channel : public Instrument::Channel{
		PluckedString pluckedString;

		enum E_State{
			STOPPED,
			PLAYING,
			RELEASING
		} state;

		float curTime;

	public:
		Channel() :
				pluckedString(27.5),
				state(STOPPED)
		{
			this->pluckedString.SetFrequency(220);
			this->pluckedString.SetLoopGain(1);
			this->pluckedString.SetLoopFilterCoefs(0.95, 0.05);
		}

		//override
		void SetFrequency(float freq){
			this->pluckedString.SetFrequency(freq);
		}

	protected:
		//override
		void OnStart(){
			this->pluckedString.Pluck(1.0);
			this->state = PLAYING;
		}
	private:
		//override
		bool MixToMixBuf(Array<s32>& mixBuf);
	};

	//override
	Ref<Instrument::Channel> CreateChannel();

	static inline ting::Ref<PluckedStringInstrument> New(){
		return ting::Ref<PluckedStringInstrument>(new PluckedStringInstrument());
	}
};



class SineWaveInstrument : public Instrument{
public:

#ifdef M_USE_FIXED_POINT
	typedef Fixed<13> real;
#else
	typedef float real;
#endif

	class Channel : public Instrument::Channel{
		real curPhase;

		real curTime; //in milliseconds
		enum EEnvelopePhase{
			ATTACK,
			DECAY,
			SUSTAIN,
			RELEASE
		} envPhase;

		real phaseStep;// (Hz / 44100) * 2pi
		real attackTime;//ms
		real decayTime;//ms
		real sustainLevel;
		real releaseTime;//ms

		real releaseFromLevel;
	public:

		Channel() :
				curPhase(0),
				curTime(0),
				envPhase(ATTACK)
		{
			this->SetFrequency(440);
		}

		//override
		// freq in Hz
		void SetFrequency(float freq){
			this->phaseStep = D2Pi<real>() * real(freq) / real(44100);
			this->attackTime = 1000.0f * 30.0f / freq;
			this->decayTime = 1000.0f * 50.0f / freq;
			this->sustainLevel = 0.8f;
			this->releaseTime = 1000.0f * 30.0f / freq;
		}

	private:
		//override
		bool MixToMixBuf(Array<s32>& mixBuf);
	};

	//override
	Ref<Instrument::Channel> CreateChannel();

	static inline ting::Ref<SineWaveInstrument> New(){
		return ting::Ref<SineWaveInstrument>(new SineWaveInstrument());
	}
};



class SampledInstrument : public Instrument{
	friend class Channel;

public:

#ifdef M_USE_FIXED_POINT
	typedef Fixed<16> real;
#else
	typedef float real;
#endif


protected:

	struct ADSRParams{
		real attack;//periods
		real decay;//periods
		real sustainLevel;
		real release;//periods
	};

public:

	class Channel : public Instrument::Channel{
		Ref<SampledInstrument> instr;

		real curPhase;
//
		real curTime; //in milliseconds
		enum EEnvelopePhase{
			ATTACK,
			DECAY,
			SUSTAIN,
			RELEASE
		} envPhase;

		//position in sample buffer of the instrument
		real phaseStep;// (Hz / 44100) * 2pi

		real attackTime;
		real decayTime;
		real releaseTime;

		real releaseFromLevel;
	public:

		Channel(Ref<SampledInstrument> instrument) :
				instr(instrument),
				curPhase(0),
				curTime(0),
				envPhase(ATTACK)
		{
			this->SetFrequency(440);
		}

		//override
		// freq in Hz
		void SetFrequency(float freq){
			real frequency(freq);
			ADSRParams &p = this->instr->adsrParams;

			this->phaseStep = frequency / this->instr->sample->freq;
			
			this->attackTime = real(1000.0f) * p.attack / frequency;
			this->decayTime = real(1000.0f) * p.decay / frequency;
			this->releaseTime = real(1000.0f) * p.release / frequency;
		}

	private:
		//override
		bool MixToMixBuf(Array<s32>& mixBuf);
	};

	//override
	Ref<Instrument::Channel> CreateChannel();


protected:
	class Sample : public RefCounted{
	public:
		Array<real> data;//sampling rate 44100Hz
		real freq;//base frequency of the sound
	};

	SampledInstrument(
			const Ref<Sample> sampleData,
			ADSRParams adsrParameters
		) :
			sample(sampleData),
			adsrParams(adsrParameters)
	{}

private:
	const Ref<Sample> sample;

	ADSRParams adsrParams;
};



class TromboneInstrument : public SampledInstrument{
	TromboneInstrument() :
			SampledInstrument(GenerateSample(), GetADSRParams())
	{}


	static Ref<Sample> GenerateSample();

	static ADSRParams GetADSRParams();

public:
	static inline ting::Ref<TromboneInstrument> New(){
		return ting::Ref<TromboneInstrument>(new TromboneInstrument());
	}
};



class ClarinetInstrument : public SampledInstrument{
	ClarinetInstrument() :
			SampledInstrument(GenerateSample(), GetADSRParams())
	{}


	static Ref<Sample> GenerateSample();

	static ADSRParams GetADSRParams();

public:
	static inline ting::Ref<ClarinetInstrument> New(){
		return ting::Ref<ClarinetInstrument>(new ClarinetInstrument());
	}
};



}//~namespace
