// (c) Ivan Gagis
// e-mail: igagis@gmail.com
// Version: 1

// Description:
//          Main Thread class

#pragma once

#include <vector>

#include <ting/debug.hpp>
#include <ting/types.hpp>
#include <ting/Array.hpp>
#include <ting/Singleton.hpp>
#include <ting/Timer.hpp>
#include <ting/Thread.hpp>
#include <ting/utils.hpp>

#include <cliser/clt/NetworkThread.hpp>

#include "PlaybackThread.hpp"



//forward declarations
class MainThread;
class MainWindow;



//==============================================================================
//==============================================================================
//==============================================================================
class ThNetworkThread : public cliser::NetworkThread{
	MainThread *mt;
public:
	ThNetworkThread(MainThread* mainThread) :
			mt(mainThread)
	{}

	//override
	void OnDataReceived(ting::Array<ting::byte> data);

	//override
	void OnConnect(EConnectFailReason failReason);

	//override
	void OnDisconnect();


	void SendLoginToServer_ts(
			const std::string& username,
			const std::string& password
		);

	void SendStopListeningToServer_ts();
	
	void SendListenNextToServer_ts();
};



class MainThread :
		public ting::Thread,
		public ting::Singleton<MainThread>
{
	friend class ConnectedMessage;
	friend class NetworkDataFromServerMessage;
	friend class DisconnectedMessage;
	friend class MusicActionMessage;
	friend class SendMusicActionsToServerMessage;
	friend class SendListenNextCommandToServerMessage;
	friend class ConnectRequestMessage;
	friend class LoginRequestMessage;
	friend class ListenNextMessage;
	friend class StopListeningMessage;

	

private:
	ThNetworkThread networkThread;
public:
	enum ENetworkState{
		OFFLINE,

		//Indicates that we went offline because login has failed
		//(incorrect login/password supplied).
		OFFLINE_LOGIN_FAILED,

		CONNECTING,
		CONNECTED,
		LOGGING_IN,
		LOGGED_IN
	};

	class NetworkStateListener : public ting::Listener{
	public:
		virtual void OnStateChange(ENetworkState newState) = 0;
	};

	//NOTE: after adding the listener it will be notified about the current state,
	//even if no state change occured.
	void AddNetworkStateListener_ts(NetworkStateListener* listener);//_ts stands for 'threadsafe'

	void RemoveNetworkStateListener_ts(NetworkStateListener* listener);

private:
	class NetworkStateListenersList : public ting::ListenersList<NetworkStateListener>{
		//override
		void NotificationRoutine(){
			while(this->HasNext()){
				this->Next()->OnStateChange(this->stateToNotifiy);
			}
		}
	public:
		ENetworkState stateToNotifiy;
	} networkStateListeners;

	ting::Mutex networkStateMutex;
	ENetworkState networkState;

	void ChangeNetworkState(ENetworkState newState);


	void ConnectInternal_ts(bool manuallyRequested);

public:
	inline void Connect_ts(){
		this->ConnectInternal_ts(true);
	}

	void Disconnect_ts();

	void Login_ts();


	
private:
	//Reconnect facilities

	static inline ting::u32 DMinReconnectTime(){
		return 1000 * 15; //15 seconds
	}

	static inline ting::u32 DMaxReconnectTime(){
		return 1000 * 60 * 20; //20 minutes
	}

	ting::u32 reconnectTime;

	friend class ReconnectTimer;
	class ReconnectTimer : public ting::Timer{
	public:
		//override
		ting::u32 OnExpire();
	} reconnectTimer;

	void SetUpReconnectTimerIfAutologinIsOn();

	//~Reconnect facilities





public:
	class CurrentPlayerListener : public ting::Listener{
	public:
		virtual void CurrentPlayer(const std::string& playerUsername) = 0;
	};

	inline void AddCurrentPlayerListener_ts(CurrentPlayerListener* l){
		this->currentPlayerListeners.Add_ts(l);
	}

	inline void RemoveCurrentPlayerListener_ts(CurrentPlayerListener* l){
		this->currentPlayerListeners.Remove_ts(l);
	}

private:
	class CurrentPlayerListenersList : public ting::ListenersList<CurrentPlayerListener>{
		//override
		void NotificationRoutine(){
			while(this->HasNext()){
				this->Next()->CurrentPlayer(this->curPlayer);
			}
		}
		std::string curPlayer;
	public:
		inline void SetCurrentPlayer(const std::string& playerUsername){
			this->curPlayer = std::string(playerUsername.c_str()); //copy
		}
	} currentPlayerListeners;




private:
	PlaybackThread playbackThread;

	//indicates listening state
	bool isListening;

	ting::uint numActionsInBuffer;
	std::vector<ting::byte> musicActionsBuffer;

	void SendMusicActionsBufferToServer();

	class SendMusicActionsTimer : public ting::Timer{
	public:
		//override
		ting::u32 OnExpire();
	} sendMusicActionsTimer;

public:
	MainThread();
	~MainThread();

	//override
	void Run();

	void ListenNext_ts();//_ts stands for 'threadsafe'

	void StopListening_ts();

private:

	void HandleLoginResultFromServer(ting::Array<ting::byte> p);

	void HandleListeningMusicDataFromServer(ting::Array<ting::byte> p);

	void HandlePlayerSwitchMessageFromServer(ting::Array<ting::byte> p);

	void HandleStopListeningMessageFromServer(ting::Array<ting::byte> p);
};
//==============================================================================
//==============================================================================
//==============================================================================

//This message is sent by NetworkThread, indicating that TCP session was estabilished
class ConnectedMessage : public ting::Message{
	MainThread *mt;
	cliser::NetworkThread::EConnectFailReason reason;
public:
	ConnectedMessage(
			MainThread* clientMainThread,
			cliser::NetworkThread::EConnectFailReason failReason
		) :
			mt(clientMainThread),
			reason(failReason)
	{
		ASSERT(this->mt)
	}

	//override
	void Handle();
};



//This message is sent by NetworkThread, indicating that TCP session was terminated
class DisconnectedMessage : public ting::Message{
	MainThread *mt;
public:
	DisconnectedMessage(MainThread* clientMainThread) :
			mt(clientMainThread)
	{
		ASSERT(this->mt)
	}

	//override
	void Handle();
};



//This message is sent by NetworkThread
class NetworkDataFromServerMessage : public ting::Message{
	MainThread *mt;//this message should hold reference to the thread this message is sent to

	ting::Array<ting::byte> data;

public:
	NetworkDataFromServerMessage(MainThread* clientMainThread, ting::Array<ting::byte> d) :
			mt(clientMainThread),
			data(d)
	{
		ASSERT(this->mt)
		ASSERT(this->data.Size() != 0)
	}

	//override
	void Handle();
};



//This message is sent to MainThread by UI thread.
class MusicActionMessage : public ting::Message{
	MainThread *mt;//this message should hold reference to the thread this message is sent to

public:
	
	enum E_Type{
		NOTE_ON = 0,
		NOTE_OFF = 1
	} type;

	ting::u32 timeStamp;
	ting::u8 volume;
	float freq;

	MusicActionMessage(MainThread* clientMainThread) :
			mt(clientMainThread)
	{
		ASSERT(this->mt)
	}

	//override
	void Handle();
};



//This message is sent to the MainThread by Timer when it expires
class SendMusicActionsToServerMessage : public ting::Message{
	MainThread *mt;//this message should hold reference to the thread this message is sent to

public:

	SendMusicActionsToServerMessage(MainThread* clientMainThread) :
			mt(clientMainThread)
	{
		ASSERT(this->mt)
	}

	//override
	void Handle();
};



//reequest to connect to the server
class ConnectRequestMessage : public ting::Message{
	MainThread *t;

	bool manuallyRequestedConnect;

public:
	ConnectRequestMessage(MainThread* mt, bool manuallyRequested = false) :
			t(mt),
			manuallyRequestedConnect(manuallyRequested)
	{
		ASSERT(this->t)
	}

	//override
	void Handle();
};



class LoginRequestMessage : public ting::Message{
	MainThread *t;

public:
	LoginRequestMessage(MainThread *mt) :
			t(mt)
	{
		ASSERT(this->t)
	}

	//override
	void Handle();
};



class ListenNextMessage : public ting::Message{
	MainThread *t;
public:
	ListenNextMessage(MainThread* thread) :
			t(thread)
	{
		ASSERT(this->t)
	}

	//override
	void Handle();
};



class StopListeningMessage : public ting::Message{
	MainThread *t;
public:
	StopListeningMessage(MainThread* thread) :
			t(thread)
	{
		ASSERT(this->t)
	}

	//override
	void Handle();
};


