/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

/*
* aumiks 0.1
*/

/*
 * STK library (http://ccrma.stanford.edu/software/stk/index.html) was used as
 * a reference while creating this class.
 */

#ifndef M_Filter_hpp
#define M_Filter_hpp

#include <ting/Array.hpp>
#include <ting/debug.hpp>

#include "config.hpp"

namespace aumiks{

/**
@brief Filter class.
Calculates y[n] = b[1]*y[n-1] + ... + b[k]*y[n-k] + a[0]*x[n] + ... + a[i]*x[n-i]
NOTE: the coefficient b[0] is not used anywhere in calculations, it is ignored.
*/
class Filter{
protected:
	ting::Array<real> a; //a[0] .. a[i]
	ting::Array<real> b; //b[0] .. b[k]
	ting::Array<real> x; //x[n] .. x[n-i]
	ting::Array<real> y; //y[n] .. y[n-k]
public:
	Filter();

	Filter(
			ting::Array<real> aCoefs = ting::Array<real>(),
			ting::Array<real> bCoefs = ting::Array<real>()
		) :
			a(aCoefs),
			b(bCoefs),
			x(this->a.Size()),
			y(this->b.Size())
	{
		this->Clear();
	}

	Filter(ting::uint numACoefs, ting::uint numBCoefs) :
			a(numACoefs),
			b(numBCoefs),
			x(a.Size()),
			y(b.Size())
	{
		this->Clear();
	}

	virtual ~Filter(){}

	void Clear(){
		for(ting::uint i = 0; i < this->x.Size(); ++i)
			this->x[i] = aumiks::real(0);
		for(ting::uint i = 0; i < this->y.Size(); ++i)
			this->y[i] = aumiks::real(0);
	}

	inline aumiks::real LastTickResult()const{
		return this->y[0];
	}

	inline aumiks::real Tick(aumiks::real sample){
		this->y[0] = aumiks::real(0);
		this->x[0] = sample;

		//TODO: use iterator pointers instead of indexes
		for(ting::uint i = this->a.Size() - 1; i > 0; --i){
			this->y[0] += this->a[i] * this->x[i];
			this->x[i] = this->x[i - 1];
		}
		this->y[0] += this->a[0] * this->x[0];

		//TODO: use iterator pointers instead of indexes
		for(ting::uint k = this->b.Size() - 1; k > 0; --k){
			this->y[0] += this->b[k] * this->y[k];
			this->y[k] = this->y[k - 1];
		}
		return this->y[0];
	}
};

}//~namespace
#endif //~once
