/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */


#include <cmath>
#include <gtkmm/dialog.h>
#include <gtkmm/main.h>
#include <cstdlib>

#include <ting/Exc.hpp>
#include <ting/math.hpp>
#include <ting/Array.hpp>
#include <ting/Ref.hpp>
#include <ting/Timer.hpp>
#include <ting/Thread.hpp>
#include <ting/utils.hpp>
#include <pugixml/pugixml.hpp>

#include "stdafx.hpp"
#include "Preferences.hpp"
#include "OptionsDialog.hpp"
#include "ThereminView.hpp"
#include "File.hpp"
#include "FSFile.hpp"
#include "MainWindow.hpp"
#include "MainThread.hpp"
#include "utils.hpp"



using namespace ting;



ThereminView::ThereminView() :
		optionsDialog(MainWindow::Inst()),
		tappedX(0)
{
	this->add_events(
			Gdk::BUTTON_PRESS_MASK |
			Gdk::BUTTON_RELEASE_MASK |
			Gdk::POINTER_MOTION_MASK |
			Gdk::BUTTON_MOTION_MASK
		);

	this->signal_button_press_event().connect(
			sigc::mem_fun(*this, &ThereminView::OnButtonPressEvent)
		);
	this->signal_button_release_event().connect(
			sigc::mem_fun(*this, &ThereminView::OnButtonReleaseEvent)
		);
	this->signal_motion_notify_event().connect(
			sigc::mem_fun(*this, &ThereminView::OnMotionEvent)
		);

	//set CAN_FOCUS flag to be able to receive keyboard events
	this->set_flags(Gtk::CAN_FOCUS);

	//NOTE: if use simple connect() then Arrow Keys press events does not come,
	//this is the bug in Gtkmm. Thus, process arrow keys in OnKeyPress_Notify()
	//while process all other keys in OnKeyPress().
	this->signal_key_press_event().connect_notify(
			sigc::mem_fun(*this, &ThereminView::OnKeyPress_Notify)
		);
	this->signal_key_press_event().connect(
			sigc::mem_fun(*this, &ThereminView::OnKeyPress)
		);
	this->signal_expose_event().connect(
			sigc::mem_fun(*this, &ThereminView::OnExposeEvent)
		);

	//set default timbre
	this->ApplyTimbre(Preferences::Inst().GetTimbre());
}



void ThereminView::ShowOptionsDialog(){
//	TRACE(<< "ThereminView::ShowOptionsDialog(): enter" << std::endl)
	this->optionsDialog.SetNumOctaves(Preferences::Inst().GetNumOctaves());
	this->optionsDialog.SetTimbre(Preferences::Inst().GetTimbre());
	this->optionsDialog.SetLockToTappedOctave(Preferences::Inst().GetLockCurrentOctave());

	switch(this->optionsDialog.run()){
		case Gtk::RESPONSE_OK: //OK
			Preferences::Inst().SetNumOctaves(this->optionsDialog.GetNumOctaves());
			Preferences::Inst().SetTimbre(this->optionsDialog.GetTimbre());
			Preferences::Inst().SetLockCurrentOctave(this->optionsDialog.GetLockToTappedOctave());
			this->ApplyTimbre(Preferences::Inst().GetTimbre());

			Preferences::Inst().SaveToFile();
			break;
		default: //Cancel and all other cases
		case Gtk::RESPONSE_CANCEL:
			this->optionsDialog.SetNumOctaves(Preferences::Inst().GetNumOctaves());
			this->optionsDialog.SetTimbre(Preferences::Inst().GetTimbre());
			this->optionsDialog.SetLockToTappedOctave(Preferences::Inst().GetLockCurrentOctave());
			break;
	}

	this->optionsDialog.hide();
	MainWindow::Inst().queue_draw();//request redrawing entire window
//	TRACE(<< "ThereminView::ShowOptionsDialog(): exit" << std::endl)
}



void ThereminView::ApplyTimbre(Preferences::E_Timbre timbre){
	switch(timbre){
		default:
		case Preferences::SINE_PLUS_ADSR:
			this->instr = aumiks::SineWaveInstrument::New();
			break;
		case Preferences::PLUCKED_STRING:
			this->instr = aumiks::PluckedStringInstrument::New();
			break;
		case Preferences::TROMBONE:
			this->instr = aumiks::TromboneInstrument::New();
			break;
		case Preferences::CLARINET:
			this->instr = aumiks::ClarinetInstrument::New();
			break;
	}
}



Preferences::E_Timbre ThereminView::GetCurrentTimbreType()const{
	if(this->instr.DynamicCast<aumiks::PluckedStringInstrument>().IsValid()){
		return Preferences::PLUCKED_STRING;
	}else{
		return Preferences::SINE_PLUS_ADSR;
	}
}



bool ThereminView::OnExposeEvent(GdkEventExpose* event){
//	TRACE(<< "ThereminView::OnExposeEvent(): enter" << std::endl)

	Glib::RefPtr<Gdk::Window> window = this->get_window();
	if(!window)
		return true;

	Gtk::Allocation allocation = get_allocation();
	const int width = allocation.get_width();
	const int height = allocation.get_height();

	Cairo::RefPtr<Cairo::Context> c = window->create_cairo_context();

	if(event){
		// clip to the area indicated by the expose event so that we only
		// redraw the portion of the window that needs to be redrawn
		c->rectangle(event->area.x, event->area.y, event->area.width, event->area.height);
		c->clip();
	}


	c->set_line_width(1.0);
	c->set_source_rgb(0, 0, 0);//black
	c->select_font_face("Sans", Cairo::FONT_SLANT_NORMAL, Cairo::FONT_WEIGHT_NORMAL);
	c->set_font_size(50.0);

	for(ting::uint i = 0; i < Preferences::Inst().GetNumOctaves(); ++i){
		c->save();
		c->translate(0, (i * height) / Preferences::Inst().GetNumOctaves());
		ThereminView::DrawOctave(
				c,
				width,
				height / Preferences::Inst().GetNumOctaves(),
				Preferences::Inst().GetBasePitch() + (Preferences::Inst().GetNumOctaves() - 1 - i) * 12
			);
		c->restore();
	}

	//draw tapped indicator
//	c->set_line_width(2.0);
//	c->set_source_rgb(0.7f, 0, 0);//darker red
//	c->move_to(this->tappedX, 0);
//	c->line_to(this->tappedX, width);
//	c->stroke();


//	TRACE(<< "ThereminView::OnExposeEvent(): exit" << std::endl)

	return true;
}



static const char* notes[] = {
	"A", 0, "B", "C", 0, "D", 0, "E", "F", 0, "G", 0
};



//static
void ThereminView::DrawOctave(
		Cairo::RefPtr<Cairo::Context> c,
		ting::uint w,
		ting::uint h,
		ting::uint basePitch)
{
	float step = float(w) / 12.0f;
	float curX = 0;
	float height = float(h);
	c->set_source_rgb(0, 0, 0);//black

	for(ting::uint i = 0, n = basePitch % 12; i < 12; ++i, ++n){
		if(n >= 12)
			n -= 12;

		c->move_to(curX, 0);
		c->line_to(curX, height);
		c->stroke();

		ASSERT(n < 12)
		//draw notes
		if(notes[n]){
			Cairo::TextExtents extents;
			c->get_text_extents(notes[n], extents);
			c->move_to(curX + step / 2 - (extents.width / 2) - extents.x_bearing, height / 2);
			c->show_text(notes[n]);
		}

		curX += step;
	}
	c->move_to(0, h);
	c->line_to(w, h);
	c->stroke();

	//draw octave number
	c->set_source_rgb(0.5f, 0, 0);//dark red
	c->move_to(0, h);
	char buf[33];
	snprintf(buf, sizeof(buf)-1, "%u", basePitch / 12);
	buf[32] = 0;
	c->show_text(buf);
}



struct MenuPlacer{
	int menuX, menuY;
	void onPopupMenuPosition(int& x, int& y, bool& pushIn){
		x = this->menuX;
		y = this->menuY;
		pushIn = true;
	}
};



bool ThereminView::OnKeyPress(GdkEventKey* event){
	TRACE(<<"ThereminView::OnKeyPress(): keyval = " << event->keyval << std::endl)
	switch(event->keyval){
		case GDK_Escape:
			MainWindow::Inst().ShowMainMenuScreen();
			break;
		default:
			return false;
			break;
	}
	return true;
}



void ThereminView::OnKeyPress_Notify(GdkEventKey* event){
	TRACE(<<"OnKeyPress_Notify(): keyval = " << event->keyval << std::endl)
	switch(event->keyval){
		case GDK_Up:
			//DHighestOctave() + 1 is the number of octaves
			if(Preferences::Inst().GetBasePitch() + this->PitchRange() >= DHighestOctave() * 12){
				Preferences::Inst().SetBasePitch(
						(DHighestOctave() + 1 - Preferences::Inst().GetNumOctaves()) * 12
					);
			}else{
				Preferences::Inst().SetBasePitch(
						Preferences::Inst().GetBasePitch() + 12
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		case GDK_Down:
			if(Preferences::Inst().GetBasePitch() <= 12){
				Preferences::Inst().SetBasePitch(0);
			}else{
				Preferences::Inst().SetBasePitch(
						Preferences::Inst().GetBasePitch() - 12
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		case GDK_Left:
			if(Preferences::Inst().GetBasePitch() != 0){
				Preferences::Inst().SetBasePitch(
						Preferences::Inst().GetBasePitch() - 1
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		case GDK_Right:
			//octaves index starts from 0, here need the number of octaves,
			//i.e. DHighestOctave() + 1
			if(Preferences::Inst().GetBasePitch() + this->PitchRange() < 12 * (DHighestOctave() + 1)){
				Preferences::Inst().SetBasePitch(
						Preferences::Inst().GetBasePitch() + 1
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		default:
			break;
	}
}



//static
ting::u8 ThereminView::CalcVol(real y, real height){
	real factor = real(1) - y / height;
	//quadric dependency on factor allows more precise volume control by finger
	return ting::u8(s32(real(0xff) * factor * factor));
}



ThereminView::FreqVol ThereminView::CalcFreqAndVol(
		real width,
		real height,
		real x,
		real y,
		bool isPressEvent
	)
{
	FreqVol ret;
	x -= width / real(24); //subtract half of halfstep onscreen width

	ting::uint octHeight = height / real(Preferences::Inst().GetNumOctaves());

	ting::ClampBottom(y, real(0));
	ASSERT(y >= real(0))

	//find the screen octave number of the touched octave
	ting::uint touchedOctave = 0;
	for(ting::uint i = 1; i <= Preferences::Inst().GetNumOctaves(); ++i){
		if(ting::uint(y) < i * octHeight){
			touchedOctave = i;
			break;
		}
	}

	if(Preferences::Inst().GetLockCurrentOctave()){
		//if press event then update current playing octave, this is to prevent volume leaps
		if(isPressEvent)
			this->curOctave = touchedOctave;
	}else{
		this->curOctave = touchedOctave;
	}

	real pow = 1;
	for(ting::uint j = 0; j < (Preferences::Inst().GetNumOctaves() - this->curOctave); ++j)
		pow *= 2;

	ret.freq = CalcFreq(
			real(Preferences::Inst().GetBaseFreq()) * pow,
			x / width
		);

	if(touchedOctave == this->curOctave)
		ret.vol = ThereminView::CalcVol(y - real(octHeight * (this->curOctave - 1)), octHeight);
	else if(touchedOctave < this->curOctave)
		ret.vol = 0xff;//max volume
	else //touchedOctave > this->curOctave
		ret.vol = 0;

	return ret;
}



bool ThereminView::OnButtonPressEvent(GdkEventButton* event){
//	TRACE(<<"onButtonPressEvent(): invoked, x = "<< event->x << " y = "<< event->y << std::endl)
//	TRACE(<<"onButtonPressEvent(): width = "<< this->get_width() << " height = "<< this->get_height() << std::endl)

	this->tappedX = float(event->x);
//	this->queue_draw_area (this->tappedX - 2, 0, this->tappedX + 2, this->get_height());

	FreqVol fv = this->CalcFreqAndVol(
			float(this->get_width()),
			float(this->get_height()),
			float(event->x),
			float(event->y),
			true //press event
		);

	if(this->ch.IsValid())
		this->ch->Stop();

#ifndef M_NO_AUDIO
	this->ch = instr->CreateChannel();
	this->ch->SetFrequency(fv.freq);
	this->ch->SetVolume(fv.vol);
	this->ch->Play();
#endif

	if(Preferences::Inst().GetBroadcast()){
		//send NOTE_ON music action event to main thread
		Ptr<MusicActionMessage> m(new MusicActionMessage(&MainThread::Inst()));
		m->type = MusicActionMessage::NOTE_ON;
		m->timeStamp = ting::GetTicks();
		m->volume = fv.vol;
		m->freq = fv.freq;
		MainThread::Inst().PushMessage(m);
	}

	return true;
}



bool ThereminView::OnButtonReleaseEvent(GdkEventButton* event){
//	TRACE(<<"onButtonPressEvent(): invoked, x = "<< event->x << " y = "<< event->y << std::endl)
//	TRACE(<<"onButtonPressEvent(): width = "<< window->get_width() << " height = "<< window->get_height() << std::endl)

//	this->queue_draw_area(this->tappedX - 2, 0, this->tappedX + 2, this->get_height());
	this->tappedX = -1;

	if(this->ch.IsValid())
		this->ch->Stop();

	if(Preferences::Inst().GetBroadcast()){
		//send NOTE_OFF music action event to main thread
		Ptr<MusicActionMessage> m(new MusicActionMessage(&MainThread::Inst()));
		m->type = MusicActionMessage::NOTE_OFF;
		m->timeStamp = ting::GetTicks();
		MainThread::Inst().PushMessage(m);
	}

	return true;
}



bool ThereminView::OnMotionEvent(GdkEventMotion* event){
	if(event->state & GDK_BUTTON1_MASK){
		this->tappedX = float(event->x);
//		this->queue_draw_area (this->tappedX - 2, 0, this->tappedX + 2, this->get_height());

		if(this->ch.IsValid()){
			FreqVol fv = this->CalcFreqAndVol(
					float(this->get_width()),
					float(this->get_height()),
					float(event->x),
					float(event->y),
					false //not a press event
				);

			this->ch->SetFrequency(fv.freq);
			this->ch->SetVolume(fv.vol);

			if(Preferences::Inst().GetBroadcast()){
				//send NOTE_ON music action event to main thread
				Ptr<MusicActionMessage> m(new MusicActionMessage(&MainThread::Inst()));
				m->type = MusicActionMessage::NOTE_ON;
				m->timeStamp = ting::GetTicks();
				m->volume = fv.vol;
				m->freq = fv.freq;
				MainThread::Inst().PushMessage(m);
			}
		}
	}else{
		this->tappedX = -1;
	}
	return true;
}


