/*
 * haze-plugin: An account plugin for telepathy-haze
 * Copyright (C) 2009 Collabora, Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <gtk/gtk.h>

#include "haze-plugin.h"

#include <librtcom-accounts-widgets/rtcom-param-string.h>
#include <librtcom-accounts-widgets/rtcom-param-int.h>
#include <librtcom-accounts-widgets/rtcom-param-bool.h>
#include <librtcom-accounts-widgets/rtcom-login.h>
#include <librtcom-accounts-widgets/rtcom-edit.h>

#include <hildon/hildon-entry.h>

#include <telepathy-glib/connection-manager.h>
#include <telepathy-glib/util.h>

ACCOUNT_DEFINE_PLUGIN (PecanPlugin, pecan_plugin, LONGCAT_TYPE);

static void
pecan_plugin_init (PecanPlugin *self)
{
}

static gchar *
get_param_name_formatted (const gchar *param_name)
{
  gchar *str;
  gchar *p;

  str = g_strdup (param_name);

  if (str && g_ascii_isalpha (str[0]))
    str[0] = g_ascii_toupper (str[0]);

  while ((p = strchr (str, '-')) != NULL
      || (p = strchr (str, '_')) != NULL)
    {
      if (p[1] != '\0' && g_ascii_isalpha (p[1]))
        {
          p[0] = ' ';
          p[1] = g_ascii_toupper (p[1]);
        }

      p++;
    }

  return str;
}

static void
fill_advanced_table (LongcatPn *plugin,
    RtcomAccountService *service,
    AccountItem *account,
    GtkWidget *table)
{
  TpConnectionManagerParam *params, *p;
  guint row;

  if (service == NULL || service->protocol == NULL)
    {
      g_warning ("Failed to find protocol, so can't fill table with advanced settings");
      return;
    }

  params = service->protocol->params;

  for (p = params, row = 0; p != NULL && p->name != NULL; p++, row++)
    {
      GtkWidget *label, *widget;
      gchar *name_formatted;
      gboolean widget_fill_row = FALSE;

      if (!tp_strdiff (p->name, "account")
          || !tp_strdiff (p->name, "password"))
        continue;

      name_formatted = get_param_name_formatted (p->name);

      if (p->dbus_signature[0] == 's')
        {
          widget = GTK_WIDGET (g_object_new (RTCOM_TYPE_PARAM_STRING,
                  "field", p->name,
                  NULL));

          /* Disable auto-capitalisation */
          g_object_set (widget, "hildon-input-mode",
              HILDON_GTK_INPUT_MODE_FULL | HILDON_GTK_INPUT_MODE_DICTIONARY,
              NULL);

          if (p->flags & TP_CONN_MGR_PARAM_FLAG_HAS_DEFAULT)
            {
              hildon_entry_set_text (HILDON_ENTRY (widget),
                  g_value_get_string (&p->default_value));
            }
        }
      else if (p->dbus_signature[0] == 'y'
          || p->dbus_signature[0] == 'n'
          || p->dbus_signature[0] == 'q'
          || p->dbus_signature[0] == 'i'
          || p->dbus_signature[0] == 'u'
          || p->dbus_signature[0] == 'x'
          || p->dbus_signature[0] == 't'
          || p->dbus_signature[0] == 'd')
        {
          gint64 minint = 0;
          guint64 maxint = 0;
          gchar *range;
          gchar *text = NULL;

          switch (p->dbus_signature[0])
            {
            case 'y':
              minint = 0;
              maxint = G_MAXUINT8;

              if (p->flags & TP_CONN_MGR_PARAM_FLAG_HAS_DEFAULT)
                text = g_strdup_printf ("%u",
                    g_value_get_uint (&p->default_value));

              break;
            case 'n':
              minint = G_MININT16;
              maxint = G_MAXINT16;

              if (p->flags & TP_CONN_MGR_PARAM_FLAG_HAS_DEFAULT)
                text = g_strdup_printf ("%" G_GINT16_FORMAT,
                    g_value_get_int (&p->default_value));

              break;
            case 'q':
              minint = 0;
              maxint = G_MAXUINT16;

              if (p->flags & TP_CONN_MGR_PARAM_FLAG_HAS_DEFAULT)
                text = g_strdup_printf ("%" G_GUINT16_FORMAT,
                    g_value_get_uint (&p->default_value));

              break;
            case 'i':
              minint = G_MININT32;
              maxint = G_MAXINT32;

              if (p->flags & TP_CONN_MGR_PARAM_FLAG_HAS_DEFAULT)
                text = g_strdup_printf ("%" G_GINT32_FORMAT,
                    g_value_get_int (&p->default_value));

              break;
            case 'u':
              minint = 0;
              maxint = G_MAXUINT32;

              if (p->flags & TP_CONN_MGR_PARAM_FLAG_HAS_DEFAULT)
                text = g_strdup_printf ("%" G_GUINT32_FORMAT,
                    g_value_get_uint (&p->default_value));

              break;
            case 'x':
              minint = G_MININT64;
              maxint = G_MAXINT64;

              if (p->flags & TP_CONN_MGR_PARAM_FLAG_HAS_DEFAULT)
                text = g_strdup_printf ("%" G_GINT64_FORMAT,
                    g_value_get_int64 (&p->default_value));

              break;
            case 't':
              minint = 0;
              maxint = G_MAXUINT64;

              if (p->flags & TP_CONN_MGR_PARAM_FLAG_HAS_DEFAULT)
                text = g_strdup_printf ("%" G_GUINT64_FORMAT,
                    g_value_get_uint64 (&p->default_value));

              break;
            case 'd':
              minint = G_MININT32;
              maxint = G_MAXINT32;

              if (p->flags & TP_CONN_MGR_PARAM_FLAG_HAS_DEFAULT)
                text = g_strdup_printf ("%" G_GINT32_FORMAT,
                    g_value_get_int (&p->default_value));

              break;
            }

          range = g_strdup_printf ("%" G_GINT64_FORMAT ":%" G_GUINT64_FORMAT,
              minint, maxint);

          widget = GTK_WIDGET (g_object_new (RTCOM_TYPE_PARAM_INT,
                  "field", p->name,
                  "range", range,
                  NULL));

          if (text != NULL)
            hildon_entry_set_text (HILDON_ENTRY (widget), text);

          g_free (text);
          g_free (range);
        }
      else if (p->dbus_signature[0] == 'b')
        {
          widget = GTK_WIDGET (g_object_new (RTCOM_TYPE_PARAM_BOOL,
                  "label", name_formatted,
                  "can_focus", TRUE,
                  "field", p->name,
                  "xalign", 0.0,
                  NULL));

          if (p->flags & TP_CONN_MGR_PARAM_FLAG_HAS_DEFAULT)
            {
              hildon_check_button_set_active (HILDON_CHECK_BUTTON (widget),
                  g_value_get_boolean (&p->default_value));
            }

          widget_fill_row = TRUE;
        }
      else
        {
          g_warning ("Unknown signature for param %s: %s",
              p->name, p->dbus_signature);
          goto clean;
        }

      if (!widget_fill_row)
        {
          label = gtk_label_new (name_formatted);
          g_object_set (label, "xalign", 0.0, NULL);
          gtk_widget_show (label);

          gtk_table_attach (GTK_TABLE (table), label,
              0, /* left */
              1, /* right */
              row, /* top */
              (row + 1), /* bottom */
              GTK_SHRINK | GTK_FILL, 0, 0, 0);
        }

      rtcom_widget_set_account (RTCOM_WIDGET (widget),
          RTCOM_ACCOUNT_ITEM (account));
      gtk_widget_show (widget);

      gtk_table_attach (GTK_TABLE (table), widget,
          widget_fill_row ? 0 : 1, /* left */
          2, /* right */
          row, /* top */
          (row + 1), /* bottom */
          GTK_EXPAND | GTK_FILL, 0, 0, 0);

clean:
      g_free (name_formatted);
    }
}

static GtkWidget *
get_login_page (LongcatPn *plugin,
    RtcomAccountItem *account)
{
  const static gchar *invalid_chars_re = "[:'\"<>&;#\\s]";

  return g_object_new (RTCOM_TYPE_LOGIN,
      "username-field", "account",
      "username-invalid-chars-re", invalid_chars_re,
      "username-prefill", RTCOM_ACCOUNT_PLUGIN (plugin)->username_prefill,
      "items-mask", RTCOM_ACCOUNT_PLUGIN (plugin)->capabilities,
      "account", account,
      NULL);
}

static GtkWidget *
get_edit_page (LongcatPn *plugin,
    RtcomAccountItem *account)
{
  const static gchar *invalid_chars_re = "[:'\"<>&;#\\s]";

  return g_object_new (RTCOM_TYPE_EDIT,
      "username-field", "account",
      "username-invalid-chars-re", invalid_chars_re,
      "items-mask", RTCOM_ACCOUNT_PLUGIN (plugin)->capabilities,
      "account", account,
      NULL);
}

static void
pecan_plugin_constructed (GObject *object)
{
  LongcatPn *plugin = LONGCAT (object);
  guint caps;

  caps = RTCOM_PLUGIN_CAPABILITY_ADVANCED
    | RTCOM_PLUGIN_CAPABILITY_SCREEN_NAME;

  longcat_pn_set_values (plugin, "pecan", caps,
      get_login_page, get_edit_page, fill_advanced_table);

  G_OBJECT_CLASS (pecan_plugin_parent_class)->constructed (object);
}

static void
pecan_plugin_class_init (PecanPluginClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);
  object_class->constructed = pecan_plugin_constructed;
}
