#include "summarywindowmaemo.h"
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QGridLayout>
#include <QDebug>
#include <QMenuBar>
#include <QActionGroup>
#include <QMaemo5InformationBox>
#include <QDate>
#include "maemo/datechooserdialogmaemo.h"

SummaryWindowMaemo::SummaryWindowMaemo(int minYear,
                                       int minMonth,
                                       int maxYear,
                                       int maxMonth,
                                       QWidget *parent)
        : SummaryWindow (parent)
{
        setAttribute(Qt::WA_Maemo5StackedWindow);
        setAttribute(Qt::WA_DeleteOnClose);
        setWindowFlags(windowFlags() | Qt::Window);
        setWindowTitle(tr("Month Summary"));

        QActionGroup *actionGroup = new QActionGroup(this);
        QAction *statistics = new QAction(tr("Statistics"), actionGroup);
        connect(statistics, SIGNAL(triggered()), this, SLOT(statisticsSelected()));
        QAction *items = new QAction(tr("Items"), actionGroup);
        connect(items, SIGNAL(triggered()), this, SLOT(itemsSelected()));
        QAction *evolution = new QAction(tr("Evolution"), actionGroup);
        connect(evolution, SIGNAL(triggered()), this, SLOT(evolutionSelected()));

        QMenuBar *menuBar = new QMenuBar(this);
        menuBar->addAction(statistics);
        menuBar->addAction(items);
        menuBar->addAction(evolution);

        statistics->setCheckable(true);
        items->setCheckable(true);
        evolution->setCheckable(true);

        summaryData = NULL;
        _minYear = minYear;
        _minMonth = minMonth;
        _maxYear = maxYear;
        _maxMonth = maxMonth;
        _startYear = 0;
        _startMonth = 0;
        _endYear = 0;
        _endMonth = 0;

        /* main vbox */
        vbox = new QVBoxLayout(this);

        /* period stuff */
        QHBoxLayout *hbox = new QHBoxLayout();
        vbox->addLayout(hbox, 0);

        fromLabel = new QLabel(tr("Months from"));
        hbox->addWidget(fromLabel, 0, Qt::AlignCenter);

        startDate = new QMaemo5ValueButton();
        startDate->setValueLayout(QMaemo5ValueButton::ValueBesideText);
        hbox->addWidget(startDate, 1);
        connect (startDate, SIGNAL(clicked()), this, SLOT(startDateClicked()));

        toLabel = new QLabel(tr("to"));
        hbox->addWidget(toLabel, 0, Qt::AlignCenter);

        endDate = new QMaemo5ValueButton();
        endDate->setValueLayout(QMaemo5ValueButton::ValueBesideText);
        hbox->addWidget(endDate, 1);
        connect (endDate, SIGNAL(clicked()), this, SLOT(endDateClicked()));

        loadButton = new QPushButton(tr("Update"));
        connect(loadButton, SIGNAL(clicked()), this, SLOT(loadButtonClicked()));
        hbox->addWidget(loadButton);

        /* statistics mode stuff */
        statsBox = new QFrame();
        statsBox->setFrameStyle(QFrame::NoFrame);
        vbox->addWidget(statsBox, 1);

        QGridLayout *statsLayout = new QGridLayout();
        statsBox->setLayout(statsLayout);

        statsLayout->addWidget(new QLabel(tr("Period totals")), 0, 0, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Budget")), 1, 0, 1, 1, Qt::AlignCenter);
        totalBudget = new QLabel(QString("0"));
        statsLayout->addWidget(totalBudget, 2, 0, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Expenses")), 3, 0, 1, 1, Qt::AlignCenter);
        totalExpense = new QLabel(QString("0"));
        statsLayout->addWidget(totalExpense, 4, 0, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Saved")), 5, 0, 1, 1, Qt::AlignCenter);
        totalSaved = new QLabel(QString("0"));
        statsLayout->addWidget(totalSaved, 6, 0, 1, 1, Qt::AlignCenter);

        statsLayout->addWidget(new QLabel(tr("Period expenses")), 0, 1, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Minimum")), 1, 1, 1, 1, Qt::AlignCenter);
        minExpense = new QLabel(QString("0"));
        statsLayout->addWidget(minExpense, 2, 1, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Maximum")), 3, 1, 1, 1, Qt::AlignCenter);
        maxExpense = new QLabel(QString("0"));
        statsLayout->addWidget(maxExpense, 4, 1, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Average")), 5, 1, 1, 1, Qt::AlignCenter);
        averageExpense = new QLabel(QString("0"));
        statsLayout->addWidget(averageExpense, 6, 1, 1, 1, Qt::AlignCenter);

        statsLayout->addWidget(new QLabel(tr("Period saves")), 0, 2, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Minimum")), 1, 2, 1, 1, Qt::AlignCenter);
        minSaved = new QLabel(QString("0"));
        statsLayout->addWidget(minSaved, 2, 2, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Maximum")), 3, 2, 1, 1, Qt::AlignCenter);
        maxSaved = new QLabel(QString("0"));
        statsLayout->addWidget(maxSaved, 4, 2, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Average")), 5, 2, 1, 1, Qt::AlignCenter);
        averageSaved = new QLabel(QString("0"));
        statsLayout->addWidget(averageSaved, 6, 2, 1, 1, Qt::AlignCenter);

        chart = new ConceptsChart();
        vbox->addWidget(chart, 1);

        viewLabel = new QLabel(tr("Monthly accumulated view"));
        vbox->addWidget(viewLabel, 0, Qt::AlignCenter);
        graph = new ExpensesGraph();
        vbox->addWidget(graph, 1);

        /* set statistics mode by default */
        statistics->setChecked(true);
        chart->hide();
        graph->hide();
        viewLabel->hide();
}

SummaryWindowMaemo::~SummaryWindowMaemo()
{
        if (summaryData != NULL) {
                delete summaryData;
        }
}

void SummaryWindowMaemo::loadButtonClicked()
{
        if (_startMonth == 0 || _startYear == 0) {
                QMaemo5InformationBox::information(this, tr("Choose a period start date"));
                return;
        }

        if (_endMonth == 0 || _endYear == 0) {
                QMaemo5InformationBox::information(this, tr("Choose a period finish date"));
                return;
        }

        /* starting date must be lower or equal to end date */
        if (QDate(_startYear, _startMonth, 1) > QDate(_endYear, _endMonth, 1)) {
                QMaemo5InformationBox::information(this, tr("Start date must be prior to end date"));
                return;
        }

        emit periodChanged(_startYear, _startMonth, _endYear, _endMonth);
}

void SummaryWindowMaemo::startDateClicked()
{
        int year = _startYear;
        int month = _startMonth;

        if (DateChooserDialogMaemo::askDate(_minYear, _minMonth, _maxYear, _maxMonth, year, month)) {
                _startYear = year;
                _startMonth = month;

                startDate->setValueText(QString("%1 %2").arg(QDate::longMonthName(_startMonth)).arg(_startYear));
        }
}

void SummaryWindowMaemo::endDateClicked()
{
        int year = _endYear;
        int month = _endMonth;

        if (DateChooserDialogMaemo::askDate(_minYear, _minMonth, _maxYear, _maxMonth, year, month)) {
                _endYear = year;
                _endMonth = month;

                endDate->setValueText(QString("%1 %2").arg(QDate::longMonthName(_endMonth)).arg(_endYear));
        }
}

void SummaryWindowMaemo::statisticsSelected()
{
        statsBox->show();
        chart->hide();
        graph->hide();
        viewLabel->hide();
}

void SummaryWindowMaemo::itemsSelected()
{
        statsBox->hide();
        chart->show();
        graph->hide();
        viewLabel->hide();
}

void SummaryWindowMaemo::evolutionSelected()
{
        statsBox->hide();
        chart->hide();
        graph->show();
        viewLabel->show();
}

void SummaryWindowMaemo::setSummaryData(SummaryData *data)
{
        if (summaryData != NULL) {
                delete summaryData;
        }

        summaryData = data;

        totalBudget->setText(QString("%1").arg(summaryData->totalBudget()));
        totalExpense->setText(QString("%1").arg(summaryData->totalExpense()));

        if (summaryData->totalSaved() > 0) {
                totalSaved->setText(QString("<font color='green'>%1</font>")
                                    .arg(summaryData->totalSaved()));
        } else {
                totalSaved->setText(QString("<font color='red'>%1</font>")
                                    .arg(summaryData->totalSaved()));
        }

        minExpense->setText(QString("%1").arg(summaryData->minExpense()));
        maxExpense->setText(QString("%1").arg(summaryData->maxExpense()));
        averageExpense->setText(QString("%1").arg(summaryData->averageExpense()));

        minSaved->setText(QString("%1").arg(summaryData->minSaved()));
        maxSaved->setText(QString("%1").arg(summaryData->maxSaved()));

        if (summaryData->averageSaved() > 0) {
                averageSaved->setText(QString("<font color='green'>%1</font>")
                                    .arg(summaryData->averageSaved()));
        } else {
                averageSaved->setText(QString("<font color='red'>%1</font>")
                                    .arg(summaryData->averageSaved()));
        }

        chart->setSummaryData(summaryData);
        if (chart->isVisible()) {
                chart->repaint();
        }

        graph->setModel(summaryData);
        if (graph->isVisible()) {
                graph->repaint();
        }

        if (summaryData->totalExpense() == 0) {
                QMaemo5InformationBox::information(this, tr("Warning: there are no expenses stored in the selected period"));
        }
}

void SummaryWindowMaemo::mousePressEvent(QMouseEvent *event)
{
        if (graph->isVisible()) {
                graph->changeView();
                if (graph->accumulated()) {
                        viewLabel->setText(tr("Monthly accumulated view"));
                } else {
                        viewLabel->setText(tr("Month expenses view"));
                }
        }
}
