#include "scheduleddetailsdialogmaemo.h"
#include "confirmationdialogmaemo.h"
#include "global.h"
#include <QFormLayout>
#include <QLabel>
#include <QDebug>
#include <QStandardItem>
#include <QMaemo5InformationBox>
#include <QDate>
#include <QHBoxLayout>

ScheduledDetailsDialogMaemo::ScheduledDetailsDialogMaemo(QStandardItemModel *concepts,
                                                         QWidget *parent,
                                                         Qt::WindowFlags f)
        : ScheduledDetailsDialog (parent, f)
{
        monthSelectionAsInt = 0;
        daySelectionAsInt = 0;
        newScheduled = true;
        scheduledId = 0;

        setWindowTitle(tr("New scheduled expense"));

        buildUI(concepts);


        QPushButton *saveButton = new QPushButton(tr("Save"));
        box->addButton(saveButton, QDialogButtonBox::ActionRole);

        connect(saveButton, SIGNAL(clicked()), this, SLOT(checkValues()));
}

ScheduledDetailsDialogMaemo::ScheduledDetailsDialogMaemo(QStandardItemModel *concepts,
                                                         QStandardItemModel *scheduled,
                                                         int row,
                                                         QWidget *parent,
                                                         Qt::WindowFlags f)
        : ScheduledDetailsDialog (parent, f)
{
        scheduledId = scheduled->item(row, SCHEDULED_ID)->text().toInt();
        monthSelectionAsInt = scheduled->item(row, SCHEDULED_MONTH)->text().toInt();
        daySelectionAsInt = scheduled->item(row, SCHEDULED_DAY)->text().toInt();
        previousFrecuency = monthSelectionAsInt;
        newScheduled = false;

        setWindowTitle(tr("Scheduled expense details"));

        buildUI(concepts);

        QPushButton *updateButton = new QPushButton(tr("Save"));
        box->addButton(updateButton, QDialogButtonBox::ActionRole);
        QPushButton *deleteButton = new QPushButton(tr("Delete"));
        box->addButton(deleteButton, QDialogButtonBox::ActionRole);

        /* amount */
        amount->setText(scheduled->item(row, SCHEDULED_AMOUNT)->text());

        /* description */
        description->setText(scheduled->item(row, SCHEDULED_DESCRIPTION)->text());

        /* concept */
        /* concept may be empty, so we need to check it */
        if (!scheduled->item(row, SCHEDULED_CONCEPT)->text().isEmpty()) {
                QList<QStandardItem*> list = concepts->findItems(scheduled->item(row, SCHEDULED_CONCEPT)->text(),
                                                                 Qt::MatchFixedString,
                                                                 0);
                conceptSelector->setCurrentIndex(list[0]->index().row());
        }

        /* frecuency */
        month->setValueText(Expense::monthFrecuencyAsString(monthSelectionAsInt));

        /* day */
        day->setValueText(Expense::dayFrecuencyAsString(daySelectionAsInt));

        connect(updateButton, SIGNAL(clicked()), this, SLOT(checkValues()));
        connect(deleteButton, SIGNAL(clicked()), this, SLOT(deleteClicked()));
}

void ScheduledDetailsDialogMaemo::buildUI (QStandardItemModel *concepts)
{
        QHBoxLayout *layout = new QHBoxLayout (this);

        QFormLayout *form = new QFormLayout ();
        form->setHorizontalSpacing(50);
        layout->addLayout (form, 1);

        day = new QMaemo5ValueButton(tr("Day").toLatin1().data());
        day->setValueLayout(QMaemo5ValueButton::ValueUnderText);
        connect (day, SIGNAL(clicked()), this, SLOT(launchDayDialog()));
        form->addRow (day);

        month = new QMaemo5ValueButton(tr("Repeat").toLatin1().data());
        month->setValueLayout(QMaemo5ValueButton::ValueUnderText);
        connect (month, SIGNAL(clicked()), this, SLOT(launchMonthDialog()));
        form->addRow (month);

        concept = new QMaemo5ValueButton(tr("Item").toLatin1().data());
        concept->setValueLayout(QMaemo5ValueButton::ValueBesideText);
        conceptSelector = new QMaemo5ListPickSelector();
        conceptSelector->setModel(concepts);
        conceptSelector->setModelColumn(0);
        concept->setPickSelector(conceptSelector);
        form->addRow (concept);

        connect (conceptSelector, SIGNAL(selected(const QString&)), this, SLOT(conceptSelected(const QString&)));

        amount = new QLineEdit();
        form->addRow (new QLabel(tr("Amount")), amount);

        description = new QLineEdit();
        form->addRow (new QLabel(tr("Description")), description);

        box = new QDialogButtonBox (QDialogButtonBox::Cancel, Qt::Vertical);
        connect(box, SIGNAL(rejected()), this, SLOT(reject()));
        layout->addWidget (box, 0, Qt::AlignBottom);
}

ScheduledDetailsDialogMaemo::~ScheduledDetailsDialogMaemo()
{
}

void ScheduledDetailsDialogMaemo::checkValues()
{
        Expense *e;

        if (daySelectionAsInt == 0) {
                QMaemo5InformationBox::information(this, tr("Select at least one day for the expense"));
                return;
        }

        QMaemo5ListPickSelector *conceptSelector = dynamic_cast<QMaemo5ListPickSelector*>(concept->pickSelector());
        int conceptIndex = conceptSelector->currentIndex();
        QString conceptName("");
        if (conceptIndex >= 0) {
                QStandardItemModel *m = dynamic_cast<QStandardItemModel*>(conceptSelector->model());
                conceptName.append(m->item(conceptIndex, 0)->text());
        }

        QString _description(description->text());
        if (conceptIndex < 0 && _description.isEmpty()) {
                QMaemo5InformationBox::information(this, tr("Choose a concept or enter a description"));
                return;
        }

        double _amount(amount->text().toDouble());
        if (_amount <= 0) {
                QMaemo5InformationBox::information(this, tr("Invalid amount"));
                return;
        }

        if (monthSelectionAsInt == 0) {
                QMaemo5InformationBox::information(this, tr("Select at least one month for the expense"));
                return;
        }

        /* check that day-month combinations are ok */
        int i,j;
        for (i = 0; i < 12; i++) {
            for (j = 0; j < 31; j++) {
                if ((monthSelectionAsInt & (1 << i))
                    && (daySelectionAsInt & (1 << j))) {
                        if (!QDate::isValid(QDate::currentDate().year(), i + 1, j + 1)
                            && !(j == 28 && i == 1)) {
                                QMaemo5InformationBox::information(this,
                                                                   QString(tr("Day %1 in month %2 is not a valid date"))
                                                                   .arg(j + 1)
                                                                   .arg(QDate::longMonthName(i + 1)));
                                return;
                        }
                }
            }
        }

        e = new Expense (scheduledId, daySelectionAsInt, _amount, conceptName, _description, monthSelectionAsInt);

        if (newScheduled) {
                if (monthSelectionAsInt & (1 << (QDate::currentDate().month() - 1))) {
                        /* current month fits into the just created scheduled expense. Ask the user
                           if it should be applied to current month */
                        QString message(tr("Current month fits in the scheduled expense description.\n\n"
                                           "Do you want the expense to be created in it?"));
                        if (ConfirmationDialogMaemo::askConfirmation(message, this)) {
                                emit newScheduledConfirmed(e, true);
                        } else {
                                emit newScheduledConfirmed(e, false);
                        }
                } else {
                        emit newScheduledConfirmed(e, false);
                }
        } else {
                if (monthSelectionAsInt & (1 << (QDate::currentDate().month() - 1))
                    && !(previousFrecuency & (1 << (QDate::currentDate().month() - 1)))) {
                        /* current month has been added to the scheduled expense in the modification. Ask the user
                           if it should be applied to current month */
                        QString message(tr("Current month fits in the scheduled expense description after the modification.\n\n"
                                           "Do you want the expense to be created in it?"));
                        if (ConfirmationDialogMaemo::askConfirmation(message, this)) {
                                emit updateScheduledConfirmed(e, true);
                        } else {
                                emit updateScheduledConfirmed(e, false);
                        }
                } else {
                        emit updateScheduledConfirmed(e, false);
                }
        }
        accept();
}

void ScheduledDetailsDialogMaemo::deleteClicked()
{
        accept();
        emit deleteScheduledConfirmed (new Expense (scheduledId,0,0,"","",0));
}

void ScheduledDetailsDialogMaemo::conceptSelected(const QString&)
{
        if (amount->text().isEmpty()) {
                QMaemo5ListPickSelector *conceptSelector = dynamic_cast<QMaemo5ListPickSelector*>(concept->pickSelector());
                int conceptIndex = conceptSelector->currentIndex();
                if (conceptIndex >= 0) {
                        QStandardItemModel *m = dynamic_cast<QStandardItemModel*>(conceptSelector->model());
                        amount->setText(m->item(conceptIndex, 1)->text());
                }
        }
}

void ScheduledDetailsDialogMaemo::launchMonthDialog()
{
        monthDialog = new QDialog(this);

        monthDialog->setWindowTitle(tr("Select months to perform the expense"));
        monthDialog->setMinimumHeight(400);

        QHBoxLayout *layout = new QHBoxLayout (monthDialog);

        monthModel = new QStandardItemModel (0, 1, monthDialog);
        for (int i=1; i<= 12; i++) {
                QStandardItem *item = new QStandardItem (QDate::longMonthName(i));
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                monthModel->appendRow(item);
        }

        monthList = new QListView();
        monthList->setModel(monthModel);
        monthList->setProperty("FingerScrollable", true);
        monthList->setSelectionMode(QAbstractItemView::MultiSelection);
        monthSelection = monthList->selectionModel();

        int i;
        for (i = 0; i < 12; i++) {
                if (monthSelectionAsInt & (1 << i)) {
                        QModelIndex index = monthModel->index(i, 0);
                        monthSelection->select(index, QItemSelectionModel::Select);
                }
        }

        layout->addWidget(monthList, 1);

        QDialogButtonBox *box = new QDialogButtonBox();
        box->addButton(new QPushButton(tr("Save")), QDialogButtonBox::AcceptRole);
        box->addButton(QDialogButtonBox::Cancel);
        layout->addWidget (box, 0, Qt::AlignBottom);

        connect(box, SIGNAL(accepted()), this, SLOT(saveMonthSelection()));
        connect(box, SIGNAL(rejected()), this, SLOT(closeMonthDialog()));

        monthDialog->exec();
}

void ScheduledDetailsDialogMaemo::saveMonthSelection()
{
        QModelIndexList indexes = monthSelection->selectedIndexes();
        QModelIndex index;

        monthSelectionAsInt = 0;

        foreach(index, indexes) {
                monthSelectionAsInt |= 1 << index.row();
        }
        month->setValueText(Expense::monthFrecuencyAsString(monthSelectionAsInt));

        closeMonthDialog();
}

void ScheduledDetailsDialogMaemo::closeMonthDialog()
{
        monthDialog->accept();
        delete monthDialog;
}

void ScheduledDetailsDialogMaemo::launchDayDialog()
{
        dayDialog = new QDialog(this);

        dayDialog->setWindowTitle(tr("Select days to perform the expense"));
        dayDialog->setMinimumHeight(400);

        QHBoxLayout *layout = new QHBoxLayout (dayDialog);

        dayModel = new QStandardItemModel (0, 1, dayDialog);
        for (int i=1; i<= 31; i++) {
            QStandardItem *item = new QStandardItem (QString("%1").arg(i));
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                dayModel->appendRow(item);
        }

        dayList = new QListView();
        dayList->setModel(dayModel);
        dayList->setProperty("FingerScrollable", true);
        dayList->setSelectionMode(QAbstractItemView::MultiSelection);
        daySelection = dayList->selectionModel();

        int i;
        for (i = 0; i < 31; i++) {
                if (daySelectionAsInt & (1 << i)) {
                        QModelIndex index = dayModel->index(i, 0);
                        daySelection->select(index, QItemSelectionModel::Select);
                }
        }

        layout->addWidget(dayList, 1);

        QDialogButtonBox *box = new QDialogButtonBox();
        box->addButton(new QPushButton(tr("Save")), QDialogButtonBox::AcceptRole);
        box->addButton(QDialogButtonBox::Cancel);
        layout->addWidget (box, 0, Qt::AlignBottom);

        connect(box, SIGNAL(accepted()), this, SLOT(saveDaySelection()));
        connect(box, SIGNAL(rejected()), this, SLOT(closeDayDialog()));

        dayDialog->exec();
}

void ScheduledDetailsDialogMaemo::saveDaySelection()
{
        QModelIndexList indexes = daySelection->selectedIndexes();
        QModelIndex index;

        daySelectionAsInt = 0;

        foreach(index, indexes) {
                daySelectionAsInt |= 1 << index.row();
        }
        day->setValueText(Expense::dayFrecuencyAsString(daySelectionAsInt));

        closeDayDialog();
}

void ScheduledDetailsDialogMaemo::closeDayDialog()
{
        dayDialog->accept();
        delete dayDialog;
}
