#include "scheduleddetailsdialogdesktop.h"
#include "confirmationdialogdesktop.h"
#include "global.h"
#include <QFormLayout>
#include <QLabel>
#include <QDebug>
#include <QStandardItem>
#include <QDate>
#include <QVBoxLayout>
#include <QPushButton>
#include <QMessageBox>

ScheduledDetailsDialogDesktop::ScheduledDetailsDialogDesktop(QStandardItemModel *concepts,
                                                             QWidget *parent,
                                                             Qt::WindowFlags f)
        : ScheduledDetailsDialog (parent, f)
{
        monthSelectionAsInt = 0;
        newScheduled = true;
        scheduledId = 0;

        setWindowTitle(tr("New scheduled expense"));

        buildUI(concepts);
        concept->setCurrentIndex(-1);


        QPushButton *saveButton = new QPushButton(tr("Save"));
        box->addButton(saveButton, QDialogButtonBox::ActionRole);

        connect(saveButton, SIGNAL(clicked()), this, SLOT(checkValues()));
}

ScheduledDetailsDialogDesktop::ScheduledDetailsDialogDesktop(QStandardItemModel *concepts,
                                                             QStandardItemModel *scheduled,
                                                             int row,
                                                             QWidget *parent,
                                                             Qt::WindowFlags f)
        : ScheduledDetailsDialog (parent, f)
{
        scheduledId = scheduled->item(row, SCHEDULED_ID)->text().toInt();
        monthSelectionAsInt = scheduled->item(row, SCHEDULED_MONTH)->text().toInt();
        daySelectionAsInt = scheduled->item(row, SCHEDULED_DAY)->text().toInt();
        previousFrecuency = monthSelectionAsInt;
        newScheduled = false;

        setWindowTitle(tr("Scheduled expense details"));

        buildUI(concepts);

        QPushButton *updateButton = new QPushButton(tr("Save"));
        box->addButton(updateButton, QDialogButtonBox::ActionRole);
        QPushButton *deleteButton = new QPushButton(tr("Delete"));
        box->addButton(deleteButton, QDialogButtonBox::ActionRole);

        /* day */
        QItemSelectionModel *daySelection = day->selectionModel();
        QAbstractItemModel *dayModel = day->model();
        for (int i = 0; i < 31; i++) {
                if (daySelectionAsInt & (1 << i)) {
                        QModelIndex index = dayModel->index(i, 0);
                        daySelection->select(index, QItemSelectionModel::Select);
                }
        }

        /* amount */
        amount->setText(scheduled->item(row, SCHEDULED_AMOUNT)->text());

        /* description */
        description->setText(scheduled->item(row, SCHEDULED_DESCRIPTION)->text());

        /* concept */
        /* concept may be empty, so we need to check it */
        if (!scheduled->item(row, SCHEDULED_CONCEPT)->text().isEmpty()) {
                QList<QStandardItem*> list = concepts->findItems(scheduled->item(row, SCHEDULED_CONCEPT)->text(),
                                                                 Qt::MatchFixedString,
                                                                 0);
                concept->setCurrentIndex(list[0]->index().row());
        }

        /* frecuency */
        QItemSelectionModel *monthSelection = month->selectionModel();
        QAbstractItemModel *monthModel = month->model();
        for (int i = 0; i < 12; i++) {
                if (monthSelectionAsInt & (1 << i)) {
                        QModelIndex index = monthModel->index(i, 0);
                        monthSelection->select(index, QItemSelectionModel::Select);
                }
        }

        connect(updateButton, SIGNAL(clicked()), this, SLOT(checkValues()));
        connect(deleteButton, SIGNAL(clicked()), this, SLOT(deleteClicked()));
}

void ScheduledDetailsDialogDesktop::buildUI (QStandardItemModel *concepts)
{
        QVBoxLayout *layout = new QVBoxLayout (this);

        QFormLayout *form = new QFormLayout ();
        form->setHorizontalSpacing(50);
        layout->addLayout (form, 1);

        QStandardItemModel *dayModel = new QStandardItemModel (0, 1, this);
        for (int i = 1; i <= 31; i++) {
                QStandardItem *item = new QStandardItem (QString("%1").arg(i));
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                dayModel->appendRow(item);
        }

        day = new QListView();
        day->setModel(dayModel);
        day->setSelectionMode(QAbstractItemView::MultiSelection);
        form->addRow (new QLabel(tr("Day")), day);

        monthModel = new QStandardItemModel (0, 1, this);
        for (int i=1; i<= 12; i++) {
                QStandardItem *item = new QStandardItem (QDate::longMonthName(i));
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                monthModel->appendRow(item);
        }

        month = new QListView();
        month->setModel(monthModel);
        month->setSelectionMode(QAbstractItemView::MultiSelection);
        form->addRow(new QLabel("Repeat"), month);

        concept = new QComboBox();
        concept->setModel(concepts);
        concept->setModelColumn(0);
        form->addRow (new QLabel(tr("Item")), concept);

        amount = new QLineEdit();
        form->addRow (new QLabel(tr("Amount")), amount);

        description = new QLineEdit();
        form->addRow (new QLabel(tr("Description")), description);

        box = new QDialogButtonBox (QDialogButtonBox::Cancel, Qt::Horizontal);
        connect(box, SIGNAL(rejected()), this, SLOT(reject()));
        layout->addWidget (box, 0, Qt::AlignBottom);
}

ScheduledDetailsDialogDesktop::~ScheduledDetailsDialogDesktop()
{
}

void ScheduledDetailsDialogDesktop::checkValues()
{
        Expense *e;

        QItemSelectionModel *selectionModel = day->selectionModel();
        QModelIndexList indexes = selectionModel->selectedIndexes();
        QModelIndex index;

        daySelectionAsInt = 0;
        foreach(index, indexes) {
                daySelectionAsInt |= 1 << index.row();
        }

        if (daySelectionAsInt == 0) {
                message(tr("Select at least one day for the expense"));
                return;
        }

        QString conceptName = concept->currentText();

        QString _description(description->text());
        if (conceptName.isEmpty() && _description.isEmpty()) {
                message(tr("Choose a concept or enter a description"));
                return;
        }

        double _amount(amount->text().toDouble());
        if (_amount <= 0) {
                message(tr("Invalid amount"));
                return;
        }

        selectionModel = month->selectionModel();
        indexes = selectionModel->selectedIndexes();

        monthSelectionAsInt = 0;
        foreach(index, indexes) {
                monthSelectionAsInt |= 1 << index.row();
        }

        if (monthSelectionAsInt == 0) {
                message(tr("Select at least one month for the expense"));
                return;
        }

        /* check that day-month combinations are ok */
        int i,j;
        for (i = 0; i < 12; i++) {
            for (j = 0; j < 31; j++) {
                if ((monthSelectionAsInt & (1 << i))
                    && (daySelectionAsInt & (1 << j))) {
                        if (!QDate::isValid(QDate::currentDate().year(), i + 1, j + 1)
                            && !(j == 28 && i == 1)) {
                            message(QString(tr("Day %1 in month %2 is not a valid date"))
                                    .arg(j + 1)
                                    .arg(QDate::longMonthName(i + 1)));
                            return;
                        }
                }
            }
        }

        e = new Expense (scheduledId, daySelectionAsInt, _amount, conceptName, _description, monthSelectionAsInt);

        if (newScheduled) {
                if (monthSelectionAsInt & (1 << (QDate::currentDate().month() - 1))) {
                        /* current month fits into the just created scheduled expense. Ask the user
                           if it should be applied to current month */
                        QString message(tr("Current month fits in the scheduled expense description.\n\n"
                                           "Do you want the expense to be created in it?"));
                        if (ConfirmationDialogDesktop::askConfirmation(message, this)) {
                                emit newScheduledConfirmed(e, true);
                        } else {
                                emit newScheduledConfirmed(e, false);
                        }
                } else {
                        emit newScheduledConfirmed(e, false);
                }
        } else {
                if (monthSelectionAsInt & (1 << (QDate::currentDate().month() - 1))
                    && !(previousFrecuency & (1 << (QDate::currentDate().month() - 1)))) {
                        /* current month has been added to the scheduled expense in the modification. Ask the user
                           if it should be applied to current month */
                        QString message(tr("Current month fits in the scheduled expense description after the modification.\n\n"
                                           "Do you want the expense to be created in it?"));
                        if (ConfirmationDialogDesktop::askConfirmation(message, this)) {
                                emit updateScheduledConfirmed(e, true);
                        } else {
                                emit updateScheduledConfirmed(e, false);
                        }
                } else {
                        emit updateScheduledConfirmed(e, false);
                }
        }
        accept();
}

void ScheduledDetailsDialogDesktop::deleteClicked()
{
        accept();
        emit deleteScheduledConfirmed (new Expense (scheduledId,0,0,"","",0));
}

void ScheduledDetailsDialogDesktop::message(QString text)
{
        QMessageBox msgBox(this);
        msgBox.setText(text);
        msgBox.exec();
}

QSize ScheduledDetailsDialogDesktop::minimumSizeHint() const
{
        return QSize(400, 400);
}

QSize ScheduledDetailsDialogDesktop::sizeHint() const
{
        return QSize(400, 400);
}
