#include "expensedetailsdialog.h"
#include <QHBoxLayout>
#include <QFormLayout>
#include <QLabel>
#include <QDebug>
#include <QStandardItem>
#include <QMaemo5ListPickSelector>
#include <QMaemo5InformationBox>

ExpenseDetailsDialog::ExpenseDetailsDialog(const QDate &date,
                                           QStandardItemModel *concepts,
                                           QWidget *parent, Qt::WindowFlags f)
        : QDialog (parent, f)
{
        newExpense = true;
        expenseId = 0;

        setWindowTitle(tr("Add new expense"));

        buildUI (date, concepts);

        if (date.year() == QDate::currentDate().year()
            && date.month() == QDate::currentDate().month()) {
                    /* current month. set default day in the picker */
                    daySelector->setCurrentIndex(date.day() - 1);
        }
}


ExpenseDetailsDialog::ExpenseDetailsDialog(const QDate &date,
                                           QStandardItemModel *concepts,
                                           QStandardItemModel *expenses,
                                           int row,
                                           QWidget *parent,
                                           Qt::WindowFlags f)
        : QDialog (parent, f)
{
        newExpense = false;
        expenseId = expenses->item(row, 0)->text().toInt();

        setWindowTitle(tr("Expense details"));

        buildUI(date, concepts);

        QPushButton *deleteButton = new QPushButton(tr("Delete"));
        box->addButton(deleteButton, QDialogButtonBox::ActionRole);
        connect(deleteButton, SIGNAL(clicked()), this, SLOT(deleteClicked()));

        /* day */
        daySelector->setCurrentIndex(expenses->item(row, 1)->text().toInt() - 1);

        /* amount */
        amount->setText(expenses->item(row, 2)->text());

        /* description */
        description->setText(expenses->item(row, 4)->text());

        /* concept */
        /* concept may be empty, so we need to check it */
        if (!expenses->item(row, 3)->text().isEmpty()) {
                QList<QStandardItem*> list = concepts->findItems(expenses->item(row, 3)->text(),
                                                                 Qt::MatchFixedString,
                                                                 0);
                conceptSelector->setCurrentIndex(list[0]->index().row());
        }
}

void ExpenseDetailsDialog::buildUI (const QDate &date,
                                    QStandardItemModel *concepts)
{
        QHBoxLayout *layout = new QHBoxLayout (this);

        QFormLayout *form = new QFormLayout ();
        form->setHorizontalSpacing(50);
        layout->addLayout (form, 1);

        QDate currentDate = QDate::currentDate();
        bool isCurrentMonth;
        if (date.month() == currentDate.month()
            && date.year() == currentDate.year()) {
                isCurrentMonth = true;
        } else {
                isCurrentMonth = false;
        }
        QStandardItemModel *dayModel = new QStandardItemModel (0, 2, this);
        for (int i=1; i <= date.daysInMonth(); i++) {
                const QDate _date = QDate(date.year(), date.month(), i);
                QStandardItem *dayName = new QStandardItem (humanizeDay(_date, currentDate, isCurrentMonth));
                dayName->setTextAlignment(Qt::AlignCenter);
                dayName->setEditable(false);
                QStandardItem *dayNumber = new QStandardItem(QString("%1").arg(i));
                dayModel->setItem(i - 1, 0, dayName);
                dayModel->setItem(i - 1, 1, dayNumber);
        }
        day = new QMaemo5ValueButton(tr("Day").toLatin1().data());
        day->setValueLayout(QMaemo5ValueButton::ValueBesideText);
        daySelector = new QMaemo5ListPickSelector();
        daySelector->setModel(dayModel);
        daySelector->setModelColumn(0);
        day->setPickSelector(daySelector);
        form->addRow (day);

        concept = new QMaemo5ValueButton(tr("Item").toLatin1().data());
        concept->setValueLayout(QMaemo5ValueButton::ValueBesideText);
        conceptSelector = new QMaemo5ListPickSelector();
        conceptSelector->setModel(concepts);
        conceptSelector->setModelColumn(0);
        concept->setPickSelector(conceptSelector);
        form->addRow (concept);

        connect (conceptSelector, SIGNAL(selected(const QString&)), this, SLOT(conceptSelected(const QString&)));

        amount = new QLineEdit();
        form->addRow (new QLabel(tr("Amount")), amount);

        description = new QLineEdit();
        form->addRow (new QLabel(tr("Description")), description);

        box = new QDialogButtonBox (QDialogButtonBox::Cancel, Qt::Vertical);
        connect(box, SIGNAL(rejected()), this, SLOT(reject()));
        layout->addWidget (box, 0, Qt::AlignBottom);

        QPushButton *saveButton = new QPushButton(tr("Save"));
        box->addButton(saveButton, QDialogButtonBox::ActionRole);
        connect(saveButton, SIGNAL(clicked()), this, SLOT(checkValues()));
}

QString ExpenseDetailsDialog::humanizeDay(const QDate &date,
                                          const QDate &currentDate,
                                          bool isCurrentMonth)
{
        int currentDay = currentDate.day();
        int day = date.day();

        QString humanizedDay = QString("%1, %2").arg(date.toString("ddd")).arg(date.day());

        if (isCurrentMonth) {
                if (day == currentDay) {
                        humanizedDay = QString(tr("Today"));
                } else if (currentDay - day == 1) {
                        humanizedDay = QString(tr("Yesterday"));
                } else if (currentDay + 1 == day) {
                                humanizedDay = QString(tr("Tomorrow"));
                }
        }

	return humanizedDay;
}

ExpenseDetailsDialog::~ExpenseDetailsDialog()
{
}

void ExpenseDetailsDialog::checkValues()
{
        Expense *e;

        int dayIndex = daySelector->currentIndex();
        QStandardItemModel *dayModel = dynamic_cast<QStandardItemModel*>(daySelector->model());
        int _day = dayModel->item(dayIndex, 1)->text().toInt();

        if (_day == 0) {
                QMaemo5InformationBox::information(this, tr("Invalid day"));
                return;
        }

        QMaemo5ListPickSelector *conceptSelector = dynamic_cast<QMaemo5ListPickSelector*>(concept->pickSelector());
        int conceptIndex = conceptSelector->currentIndex();
        QString conceptName;
        if (conceptIndex >= 0) {
                QStandardItemModel *m = dynamic_cast<QStandardItemModel*>(conceptSelector->model());
                conceptName.append(m->item(conceptIndex, 0)->text());
        }

        QString _description(description->text());
        if (conceptIndex < 0 && _description.isEmpty()) {
                QMaemo5InformationBox::information(this, tr("Choose a concept or enter a description"));
                return;
        }

        double _amount(amount->text().toDouble());
        if (_amount <= 0) {
                QMaemo5InformationBox::information(this, tr("Invalid amount"));
                return;
        }

        if (conceptIndex < 0) {
                e = new Expense (expenseId, _day, _amount, "", _description);
        } else {
                /* amount and concept */
                e = new Expense (expenseId, _day, _amount, conceptName, _description);
        }

        if (newExpense) {
                emit newExpenseConfirmed(e);
        } else {
                emit updateExpenseConfirmed(e);
        }

        accept();
}

void ExpenseDetailsDialog::deleteClicked()
{
        emit deleteExpenseConfirmed (new Expense (expenseId,0,0,"",""));
        accept();
}

void ExpenseDetailsDialog::conceptSelected(const QString&)
{
        if (amount->text().isEmpty()) {
                QMaemo5ListPickSelector *conceptSelector = dynamic_cast<QMaemo5ListPickSelector*>(concept->pickSelector());
                int conceptIndex = conceptSelector->currentIndex();
                if (conceptIndex >= 0) {
                        QStandardItemModel *m = dynamic_cast<QStandardItemModel*>(conceptSelector->model());
                        amount->setText(m->item(conceptIndex, 1)->text());
                }
        }
}
