/*
 * Copyright (C) 2009, Jamie McCracken (jamiecc at gnome org)
 * Copyright (C) 2010, Bundyo (bundyo at gmail.com)
 *
 * Initially written by Jamie McCracken in Genie. 
 * Converted to Vala and butchered (downgraded to Tracker 0.6, 
 * Hildonized and adapted to Maemo 5) by Bundyo.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA  02110-1301, USA.
 */

using Gtk;
using Gdk;
using Hildon;
using Osso;

public enum Categories {
    Application,
    Contact,
    Email,
    EmailAttachment,
    File,
    Folder,
    Music,
    Video,
    Image,
    Document,
    Text,
    Development,
    Web,
    WebHistory
}

[CCode (cname = "TRACKER_UI_DIR")]
static const string UIDIR = "";

[CCode (cname = "SRCDIR")]
static const string SRCDIR = "";


class TrackerUtils {

	public static string last_selection;
	public static Hildon.Dialog dialog;
	public static unowned FileInfo[] ginfos; // Vala 0.7.9 seems to produce segmentation faults when using local vars in a lambda... :(

	public static int get_info_index( List<AppInfo> haystack, string needle ) {
		for ( var i = 0; i < haystack.length(); i++ ) {
			if (haystack.nth_data(i).get_name() == needle)
				return i;
		}
		
		return -1;
	}

	public static string get_fileinfo_name( FileInfo[] haystack, string needle ) {
		for ( var i = 0; i < haystack.length; i++ ) {
			if (haystack[i].get_display_name() == needle)
				return haystack[i].get_name();
		}
		
		return needle;
	}

	public static bool has_fileinfo_name( FileInfo[] haystack, string needle ) {
		for ( var i = 0; i < haystack.length; i++ ) {
			if (haystack[i].get_display_name() == needle)
				return true;
		}
		
		return false;
	}

	public static string get_textdomain_from_desktop (string id) {
		KeyFile key_file;
		string filename;
		string textdomain = "";

		filename = Path.build_filename ( "applications", id );
		key_file = new KeyFile ();

		if (key_file.load_from_data_dirs ( filename, null, KeyFileFlags.NONE)) {
			string group = key_file.get_start_group ();
			textdomain = key_file.get_string ( group, "X-Text-Domain" );
		}

		return textdomain;
	}

	public static string? get_service_from_desktop (string id) {
		KeyFile key_file;
		string filename;
		string? service_name = null;

		filename = Path.build_filename ( "applications", id );
		key_file = new KeyFile ();

		if (key_file.load_from_data_dirs ( filename, null, KeyFileFlags.NONE)) {
			string group = key_file.get_start_group ();
			service_name = key_file.get_string ( group, "X-Osso-Service" );
		}

		return service_name;
	}

    public static bool OpenUri (string uri, bool is_dir) {
	    
		if (is_dir) {
			Banner.show_information_with_markup( mainwindow, null, "Trying <b>" + uri + "</b>");
		    var service = get_service_from_desktop("hildon-filemanager.desktop").strip();
	    	open_with_service ( service, "/media/mmc1", "mime_open" );
//	    	open_with_service ( service, uri, "open_folder" );
		} else {
			Banner.show_information_with_markup( mainwindow, null, "Opening <b>" + uri + "</b>");
			mime_open_file ( conn.get_connection (), uri );
		}
		
		return true;
    }

    public static bool OpenUriWith (string uri, bool is_dir) {
	    
	    var uriinfo = File.new_for_commandline_arg ( "file://" + uri ).query_info("standard::content-type", 
                                                 								FileQueryInfoFlags.NONE, null);
	    
        var infos = new FileInfo[0];
        unowned List<AppInfo> app_infos;
        
       	app_infos = AppInfo.get_all_for_type(uriinfo.get_content_type());
       	if (app_infos.length() == 0)
        	app_infos = AppInfo.get_all();
        	
        app_infos.sort( (CompareFunc) strcmp );
        
        foreach (AppInfo info in app_infos) {
	        if (info.should_show() && !info.get_id().contains("catorise")) {
		        var file = File.new_for_commandline_arg ( "file://" + info.get_executable() );
		        FileInfo? fileinfo = null;
		        try {
	        		fileinfo = file.query_info ("standard::display-name,standard::content-type", 
                                                 								FileQueryInfoFlags.NONE, null);
                    string display_name = "";

                    if (info.get_name().contains("_") || info.get_name().contains("."))
                    	display_name = dgettext( get_textdomain_from_desktop(info.get_id()), info.get_name());
                    else
	                    display_name = info.get_name().substring(0,1).up() + info.get_name().substring(1);

					if (!has_fileinfo_name(infos, display_name)) {
						fileinfo.set_display_name(display_name);
	                    fileinfo.set_name(info.get_name());
	                    infos += fileinfo;
                 	}
                } catch (GLib.Error e) {
	                //print (e.message + "\n");
                }
        	}
        }
        
		dialog = new Hildon.Dialog.with_buttons ( "Open With...", mainwindow, DialogFlags.MODAL );
		var selector = create_icon_selector( infos, 0 );
		
		var pannable = new PannableArea();
		pannable.set_size_request(800, 350);
		pannable.add ( selector );
		dialog.vbox.add( pannable );
		dialog.show_all();

	    var rd = dialog.run();
	    dialog.hide();
	    if (rd == ResponseType.OK) {
		    var index = get_info_index ( app_infos, last_selection );
		    AppInfo info = (AppInfo) app_infos.nth_data(index);
		    
			Banner.show_information_with_markup ( mainwindow, null, "Opening <b>" + uri+ "</b> with <b>" + last_selection + "</b>.");
			
			print ("Test 1\n");
		    var service = get_service_from_desktop(info.get_id()).strip();
			print ("Test 2\n");
		    if (service == null) 
		    	open_with_launch( info, uri );
		    else
		    	open_with_service ( service, uri);
			print ("Test 3\n");
	    }
		
		return true;
    }

	public static void open_with_service( string srvc, string uri, string method = "mime_open" ) {
		string service, object_path, iface;
		
		service = srvc;
		if (service.contains(".")) {
			object_path = "/%s".printf ( service.replace(".", "/") );
			iface = service;
		} else {
			object_path = "/com/nokia/%s".printf ( service );
			service = "com.nokia.%s".printf ( service );
			iface = service;  
    	}
    	 
//    	print ( service + "  " + object_path + "  " + iface + "\n" );
    	
   		var normalized_uri = uri;
    	if (!uri.has_prefix("file://"))
    		normalized_uri = "file://" + uri;
    	
    	try {
            var conn = DBus.Bus.get (DBus.BusType.SESSION);
			dynamic DBus.Object item = conn.get_object (service, object_path, iface); 
			item.call( method, null, typeof(string), normalized_uri, null );
		} catch (GLib.Error e) {
			//print (e.message + "\n");
		}
	}
	
	public static void open_with_launch( AppInfo appinfo, string uri ) {
		List<File> args = new List<File> ();
		args.append( File.new_for_commandline_arg( uri ) );
		
		try {
			appinfo.launch( args, null );
		} catch (GLib.Error e) {
			//
		}
	}

    public static Gdk.Pixbuf? GetThemePixbufByName (string icon_name, int size, Gdk.Screen screen) {

        var icon = new ThemedIcon (icon_name);

        return GetThemeIconPixbuf (icon, size, screen);
	}

    public static Gdk.Pixbuf? GetThumbNail (FileInfo info, int thumb_size, int icon_size, Gdk.Screen screen) {

        Gdk.Pixbuf? pixbuf = null;

        try {
            var thumbpath = info.get_attribute_byte_string (FILE_ATTRIBUTE_THUMBNAIL_PATH);

            if (thumbpath != null)
                pixbuf = new Gdk.Pixbuf.from_file_at_size (thumbpath, thumb_size, thumb_size);
        } catch (Error e) {
            //pass;
        };

        if (pixbuf == null)
            pixbuf = GetThemeIconPixbuf (info.get_icon (), icon_size, screen);

        if (pixbuf == null)
            pixbuf = GetThemePixbufByName ("text-x-generic", icon_size, screen);

        return pixbuf;
    }


    public static Gdk.Pixbuf? GetThemeIconPixbuf (Icon icon, int size, Gdk.Screen screen) {

        IconInfo icon_info;

        var theme = IconTheme.get_for_screen (screen);

        icon_info = theme.lookup_by_gicon (icon, size, IconLookupFlags.USE_BUILTIN);

        try {
            return icon_info.load_icon ();
        } catch (Error e) {
            return null;
        }
    }
    
    public static string[] join_list(List<string> list) {
	    string[] output = new string[0];
	    list.sort( (CompareFunc) strcmp );
	    
	    foreach (string item in list) {
		    output += item;
	    }
	    
	    return output;
    }

	public static TouchSelector create_selector( string[] items, int active, TouchSelectorSelectionMode selection_mode ) {
		var selector = new TouchSelector.text();
		selector.set_column_selection_mode ( selection_mode );
		int index = 0;
		
		foreach ( string item in items ) {
			selector.append_text( item );
			index++;
		}
		
		if (active >= 0)
			selector.set_active( 0, active );
		else {
			TreeIter iter;
			var model = selector.get_model(0);
			if (model.get_iter_first(out iter)) {
				do {
					selector.select_iter( 0, iter, false );
				} while (model.iter_next( ref iter ));
			}
		}

		return selector;
	}
	
	public static TreeView create_icon_selector( FileInfo[] infos, int active ) {
		var selector = (TreeView) gtk_tree_view_new ( UIMode.NORMAL );
		selector.get_selection().set_mode( SelectionMode.SINGLE );
		int index = 0;
		
		var renderer = new CellRendererText();
		var renderer_pixbuf = new CellRendererPixbuf();
		renderer_pixbuf.width = 100;
		
		var column = new TreeViewColumn.with_attributes("Icon", renderer_pixbuf, "pixbuf", 1);
		selector.append_column ( column );
		column = new TreeViewColumn.with_attributes("Language", renderer, "text", 0);
		selector.append_column ( column );
		
		ListStore store_icons = new ListStore (2, typeof(string), typeof(Gdk.Pixbuf));
                                                		
		foreach ( FileInfo item in infos ) {
			TreeIter iter;
			store_icons.append ( out iter );
			store_icons.set ( iter, 0, item.get_display_name() , 1, icons.lookup(item.get_name()) );
		}
		
		var sortable = new TreeModelSort.with_model( store_icons );
		
		selector.set_model ( sortable );
		column.set( "text-column", 0 );
		sortable.set_sort_column_id( 0, SortType.ASCENDING );
		
		ginfos = infos;

		selector.row_activated += (self, path, column) => {
			Gtk.TreeModel model = self.get_model();
			Gtk.TreeIter iter;
	
			model.get_iter (out iter, path);
		    
		    model.get( iter, 0, out last_selection );
		    last_selection = get_fileinfo_name(ginfos, last_selection);

		    dialog.response( ResponseType.OK );
		    dialog.hide();
		};
		
//		if (active >= 0)
//			selector.set_active( 0, active );
//		else {
//			TreeIter iter;
//			var model = selector.get_model(0);
//			if (model.get_iter_first(out iter)) {
//				do {
//					selector.select_iter( 0, iter, false );
//				} while (model.iter_next( ref iter ));
//			}
//		}

		selector.show_all();
		
		return selector;
	}

	public static TouchSelectorEntry create_selector_entry( string[] items, string text ) {
		var selector = new TouchSelectorEntry.text();
		
		foreach ( string item in items ) {
			selector.append_text( item );
		}
		
		selector.get_entry().set_text( text );
		
		return selector;
	}
}