/*
 * Copyright (c) 2010 David Galindo <nowheremanmail@gmail.com>
 */

#ifndef DATABASEMANAGER_H_
#define DATABASEMANAGER_H_

#include <QObject>
#include <QSqlDatabase>
#include <QSqlError>
#include <QSqlQuery>
#include <QPixmap>
#include <QDate>
#include <QtGui>
#include <QDebug>

#include <openssl/aes.h>

class DatabaseManager;

// ---------------------------------------------------------------------------
// TargetField
// ---------------------------------------------------------------------------
class TargetField
{
public:
	TargetField() {
		id = -1;
		originalName = "";
	}
	TargetField(TargetField * copy) {
		this->id = copy->id;
		this->name = copy->name;
		this->type = copy->type;
		this->encrypted = copy->encrypted;
		this->order = copy->order;
		this->originalName = copy->originalName;
		this->value = copy->value;
	}
	~TargetField();

	TargetField(QString name, int type, bool encrypted, int order) {
		this->id = -1;
		this->name = name;
		this->type = type;
		this->encrypted = encrypted;
		this->order = order;
		this->originalName = name;
	}

	TargetField(QString name, int type, bool encrypted, int order, QString original) {
		this->id = -1;
		this->name = name;
		this->type = type;
		this->encrypted = encrypted;
		this->order = order;
		this->originalName = original;
	}


	int id;
	int targetId;

	int order;
	bool encrypted;
	QString name;
	QString originalName;
	QString value;
	int type;
};
// ---------------------------------------------------------------------------
// target
// ---------------------------------------------------------------------------
class Target
{
public:
	Target();
	~Target();
	int id;
	QString name;
	QString groupName;
	QString type;

	QString creationdate;
	QString modificationdate;

	QImage * getPictureSmall ();
	QImage * getPicture ();
	QImage * getPictureBack ();

	QImage * getPicture (DatabaseManager *);
	QImage * getPictureBack (DatabaseManager *);

	//void setPictureSmall (QImage * a);
	void setPicture (QImage * a);
	void setPictureBack (QImage * a);

	void freeResources ();
	void freeFieldList ();
	QString getSummary ();

	bool existField (QString name);
	int getFieldByName (QString name);

//private:
	QImage * pictureSmall;
	QImage * picture;
	QImage * pictureBack;

	QList<TargetField*> fields; // ?? * or not

	bool listLoaded;

	QString getName ();
};



// ---------------------------------------------------------------------------
// DatabaseManager
// ---------------------------------------------------------------------------
class DatabaseManager: public QObject
{
public:
	DatabaseManager(QObject *parent = 0);
	~DatabaseManager();

public:
	//bool sortTargets (const Target * f, const Target * g);

	bool openDB(const QString & pass);
	bool close ();
	bool hasPassword ();
	bool isReady ();
	void lock ();
	void reset ();
	
	int nextId();

	bool insertTarget(Target* target);
	bool copyTarget(Target* target);
	QList<Target*> getTargets(const QString * filter);
	QList<QString> getGroups ();

	bool deleteTarget(Target* target);
	bool updateTarget(Target* target);
	bool updateTargetPicture(Target* target);
	bool updateTargetPictureBack(Target* target);
	bool updateFields(Target* target);
	bool copyFields(Target* target);

	bool insertTargetField(TargetField* field);
	bool copyTargetField(TargetField* field);
	bool updateTargetField(TargetField* field);
	void getFields(Target * target);

	QSqlError lastError();	

	QImage * getPicture (Target * t);
	QImage * getPictureBack (Target * t);

	bool backupDB(const QString & prefix);
	bool restoreDB(const QString & prefix);
	QString changePassword (const QString & newPassword);
        QString mergeDB (const QString & dbToMerge, const QString & passwordToMerge);

	bool renameGroup (const QString & from, const QString & to);
	bool deleteGroup (const QString & from);

private:
        bool _deleteTarget(int idtarget);
	bool deleteDB(const QString & prefix);
	bool _openDB(const QString & name, const QString & cName = "");
        bool _openFileDB(const QString & name, const QString & cName = "");
        bool initDB();
	bool isInit();

	bool migrationFrom (DatabaseManager * mgr);
        bool _mergeDB (DatabaseManager * mergeDB);

	bool checkPassword (QString a, int v);
	bool createPassword (QString a, int v);

	int getVersion ();
	bool addVersion (int v);

	bool createTargetTable();
	bool createIdTable();
	bool createTargetFieldTable();

	bool createTableVersion ();
	bool migrateVersion (int fromV, int toV, const QString & pas);

	void encrypt (const QString a, QByteArray& b);
	void decrypt (const QByteArray& a, QString& b);

	void encrypt (const QByteArray& a, QByteArray& b);
	void decrypt (const QByteArray& a, QByteArray& b);
	void encryptSHA1 (const QString & a, QByteArray& b);

	QByteArray selectSecretPhrase();
	bool hasSecretPhrase();
	bool updateSecretPhrase(QByteArray a);

private:
	AES_KEY aeskeyE;
	AES_KEY aeskeyD;
	QSqlDatabase db;
	unsigned char key[32];
	int keyLen;
	
	bool opendb;
	bool initdb;

	int currentVersion;
	QString connectionName;
};

#endif /* DATABASEMANAGER_H_ */
