/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: openBossa - INdT (renato.chencarek@openbossa.org)
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** the openBossa stream from INdT (renato.chencarek@openbossa.org).
** $QT_END_LICENSE$
**
****************************************************************************/

#include "utils.h"
#include "bbcweatherresponse.h"

#include <QStringList>


QHash<QString, Forecast::ForecastType> mappedForecastTypes()
{
    QHash<QString, Forecast::ForecastType> result;

    result["sunny"] = Forecast::Sunny;
    result["clear"] = Forecast::Sunny;
    result["clear sky"] = Forecast::Sunny;

    result["heavy snow"] = Forecast::Snow;
    result["heavy snow shower"] = Forecast::Snow;
    result["heavy snow showers"] = Forecast::Snow;
    result["cloudy with heavy snow"] = Forecast::Snow;

    result["cloudy"] = Forecast::Cloudy;
    result["white cloud"] = Forecast::Cloudy;

    result["partly cloudy"] = Forecast::PartlyCloudy;
    result["sunny intervals"] = Forecast::PartlyCloudy;

    result["grey cloud"] = Forecast::MostlyCloudy;

    result["sleet"] = Forecast::Sleet;
    result["sleet shower"] = Forecast::Sleet;
    result["sleet showers"] = Forecast::Sleet;
    result["cloudy with sleet"] = Forecast::Sleet;

    result["thundery shower"] = Forecast::Thunderstorm;
    result["thunder storm"] = Forecast::Thunderstorm;
    result["tropical storm"] = Forecast::Thunderstorm;

    result["drizzle"] = Forecast::ChanceOfRain;
    result["misty"] = Forecast::Mist;
    result["mist"] = Forecast::Mist;
    result["fog"] = Forecast::Fog;
    result["foggy"] = Forecast::Fog;

    result["hazy"] = Forecast::Haze;

    result["light shower"] = Forecast::ChanceOfRain;
    result["light rain shower"] = Forecast::ChanceOfRain;
    result["light showers"] = Forecast::ChanceOfRain;
    result["light rain"] = Forecast::ChanceOfRain;

    result["heavy rain"] = Forecast::Rain;
    result["heavy showers"] = Forecast::Rain;
    result["heavy shower"] = Forecast::Rain;
    result["heavy rain shower"] = Forecast::Rain;

    //result["cloudy with hail"] = Hail;
    //result["hail shower"] = Hail;
    //result["hail showers"] = Hail;
    //result["hail"] = Hail;

    result["light snow"] = Forecast::ChanceOfSnow;
    result["light snow shower"] = Forecast::Flurries;
    result["light snow showers"] = Forecast::Flurries;
    result["cloudy with light snow"] = Forecast::ChanceOfSnow;

    result["na"] = Forecast::UnknownForecast;

    return result;
}


BbcWeatherResponse::BbcWeatherResponse(const QString &locationId,
                                       const QDomElement &element)
    : WeatherResponse(locationId),
      m_isNight(false)
{
    parseDocument(element);
}

void BbcWeatherResponse::parseDocument(const QDomElement &element)
{
    const QDomNodeList &channels = element.elementsByTagName("channel");
    if (channels.count() == 0)
        return;

    QRegExp rxlen("for ([^,]+), (.+)");
    QDomElement channel = channels.item(0).toElement();

    for (int i = 0; i < channel.childNodes().count(); ++i) {
        QDomNode node = channel.childNodes().at(i);

        if (node.nodeName() == "title") {
            const QString &title = getNodeContentText(node);
            if (rxlen.indexIn(title) > -1) {
                location.city = rxlen.cap(1);
                location.country = rxlen.cap(2);
            }
        }

        if (node.nodeName() == "lastBuildDate")
            buildDate = readDateTime(getNodeContentText(node), false);

        if (node.nodeName() == "item")
            readItem(node.toElement());
    }
}

void BbcWeatherResponse::readItem(const QDomElement &item)
{
    QRegExp rxlen("(\\S+) at (\\S+) ([^:]+):([^\\d]+). ([-\\d]+)*");
    static QHash<QString, Forecast::ForecastType> types = mappedForecastTypes();

    int count = item.childNodes().count();

    for (int i = 0; i < count; i++) {
        const QDomNode &node = item.childNodes().at(i);

        if (node.nodeName() == "pubDate") {
            const QString &fullPubDate = getNodeContentText(node);
            pubDate = readDateTime(fullPubDate, false);
            condition.date = pubDate;
            location.timeZone = parseTimeZone(fullPubDate);
        }

        if (node.nodeName() == "geo:lat")
            location.latitude = getNodeContentText(node).toDouble();

        if (node.nodeName() == "geo:long")
            location.longitude = getNodeContentText(node).toDouble();

        if (node.nodeName() == "title") {
            const QString &title = getNodeContentText(node);

            if (rxlen.indexIn(title) < 0)
                continue;

            condition.text = rxlen.cap(4).trimmed();

            const QString &temperature = rxlen.cap(5);
            const QString &conditionKey = condition.text.toLower();

            condition.temperature = temperature.toInt();

            if (types.contains(conditionKey))
                condition.code = types[conditionKey];
            else
                condition.code = Forecast::UnknownForecast;

            // XXX: does not provided
            forecast1.low = 0;
            forecast1.high = 0;
            forecast1.valid = false;
        }
    }

    astronomy.sunrise = calculateSunriseSunset(pubDate.date(), location.latitude,
                                               location.longitude, location.timeZone, false);
    astronomy.sunset = calculateSunriseSunset(pubDate.date(), location.latitude,
                                              location.longitude, location.timeZone, true);
}

bool BbcWeatherResponse::isNight() const
{
    return condition.date.time() < astronomy.sunrise
        || condition.date.time() > astronomy.sunset;
}

bool BbcWeatherResponse::parseForecast(const QDomElement &element)
{
    const QDomNodeList &items = element.elementsByTagName("item");
    if (items.count() == 0)
        return false;

    for (int i = 0; i < items.count(); i++) {
        const QDomNode &node = items.at(i);
        readForecastItem(node.toElement());
    }

    return true;
}

void BbcWeatherResponse::readForecastItem(const QDomElement &item)
{
    static QHash<QString, Forecast::ForecastType> types = mappedForecastTypes();

    QRegExp subMatch("(\\S+): ([^,]+), Max Temp*");
    QRegExp maxMatch("Max Temp: ([+-]?[\\d]+)*");
    QRegExp minMatch("Min Temp: ([+-]?[\\d]+)*");

    int count = item.childNodes().count();

    WeatherResponse::WeekForecast result;

    for (int i = 0; i < count; i++) {
        const QDomNode &node = item.childNodes().at(i);

        if (node.nodeName() == "pubDate") {
            const QString &fullPubDate = getNodeContentText(node);
            result.date = readDateTime(fullPubDate, false).date();
        }

        if (node.nodeName() == "title") {
            const QString &title = getNodeContentText(node);

            QString conditionKey;

            if (subMatch.indexIn(title) >= 0) {
                result.weekdayName = subMatch.cap(1);
                conditionKey = subMatch.cap(2);
            }

            if (maxMatch.indexIn(title) >= 0)
                result.high = maxMatch.cap(1).toInt();

            if (minMatch.indexIn(title) >= 0)
                result.low = minMatch.cap(1).toInt();

            if (types.contains(conditionKey))
                result.code = types[conditionKey];
            else
                result.code = Forecast::UnknownForecast;

            result.valid = true;
        }
    }

    forecasts.append(result);
}
