/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: openBossa - INdT (renato.chencarek@openbossa.org)
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** the openBossa stream from INdT (renato.chencarek@openbossa.org).
** $QT_END_LICENSE$
**
****************************************************************************/

#include "citylist.h"
#include "settings.h"
#include "pixmaploader.h"
#include "forecastmodel.h"

#include <cmath>
#include <QPainter>
#include <QPropertyAnimation>
#include <QParallelAnimationGroup>
#include <QSequentialAnimationGroup>


#define ITEM_BACKGROUND             (PixmapLoader::pixmap("list_item_bg"))
#define SELECTED_ITEM_BACKGROUND    (PixmapLoader::pixmap("list_item_selected_bg"))
#define PRESSED_ITEM_BACKGROUND    (PixmapLoader::pixmap("list_item_pressed_bg"))

#define ITEM_BUTTON_PIXMAP          (PixmapLoader::pixmap("button_list_delete"))
#define ITEM_BUTTON_LEFT            (Settings::scaleWidth(406.0))

#define ITEM_CHECK_PIXMAP           (PixmapLoader::pixmap("list_check"))
#define ITEM_CHECK_LEFT             (Settings::scaleWidth(10.0))

#define TEXT_LEFT                   (2 * ITEM_CHECK_LEFT + ITEM_CHECK_PIXMAP.width())


//static const int maxVisibleItems = 4;
#define maxVisibleItems Settings::intValue("citylist-max-visible")
static const int minListCount = 3;
#define showDeleteButton (Settings::intValue("show-delete-button") != 0)

static inline qreal getCenterVerticalPos(QGraphicsItem *parent, QGraphicsItem *item)
{
    const qreal top = (parent->boundingRect().height() - item->boundingRect().height()) / 2;
    return top - parent->boundingRect().top() - item->boundingRect().top();
}

static inline qreal getCenterVerticalPos(QGraphicsItem *item)
{
    return getCenterVerticalPos(item->parentItem(), item);
}


//  CityListItem

CityListItem::CityListItem(const ForecastData &data, CityContentList *list, bool deleteVisible)
    : ContentListItem(0)
    , m_height(ITEM_BACKGROUND.height())
    , m_data(data)
    , m_list(list)
    , m_background(new QGraphicsPixmapItem(ITEM_BACKGROUND, this))
    , m_check(new QGraphicsPixmapItem(ITEM_CHECK_PIXMAP, m_background))
{
    QFont f("Nokia Sans");
    f.setPixelSize(Settings::intValue("citylist-item-font-size"));

    m_text = new ImtkLabel(this);
    m_text->setFont(f);
    m_text->setText(m_data.cityName());
    m_text->setPos(TEXT_LEFT, Settings::scaleHeight(22));
    //m_text->setCacheMode(ItemCoordinateCache);
    m_text->setMinimumHeight(Settings::scaleHeight(70));
    m_text->setMinimumWidth(ITEM_BUTTON_LEFT - TEXT_LEFT);
    m_text->setMaximumWidth(ITEM_BUTTON_LEFT - TEXT_LEFT);

    m_check->setPos(ITEM_CHECK_LEFT, getCenterVerticalPos(m_check));
    m_check->hide();

    if (showDeleteButton) {
        m_delete = new ImtkButton(ITEM_BUTTON_PIXMAP, m_background);
        m_delete->setPos(ITEM_BUTTON_LEFT, getCenterVerticalPos(m_delete));
        m_delete->setVisible(deleteVisible);
        m_delete->setOpacity(deleteVisible ? 1.0 : 0.0);

        connect(m_delete, SIGNAL(clicked()), this, SLOT(removeFromList()));
    }
}

void CityListItem::select(bool selected)
{
    m_check->setVisible(selected);
}

void CityListItem::mousePressEvent(QGraphicsSceneMouseEvent *ev)
{
    Q_UNUSED(ev);
}

void CityListItem::mouseReleaseEvent(QGraphicsSceneMouseEvent *ev)
{
    Q_UNUSED(ev);
    m_list->doActivateItem(this);
}

void CityListItem::stateChanged(ContentListItem::State state)
{
    switch(state) {
    case ContentListItem::NormalState:
        m_background->setPixmap(ITEM_BACKGROUND);
        break;
    case ContentListItem::PressedState:
        m_background->setPixmap(PRESSED_ITEM_BACKGROUND);
        break;
    case ContentListItem::SelectedState:
        m_background->setPixmap(SELECTED_ITEM_BACKGROUND);
        break;
    }
}

void CityListItem::removeFromList()
{
    m_list->removeItem(this);
}

qreal CityListItem::contentHeight() const
{
    return m_height;
}

QAbstractAnimation *CityListItem::buttonAnimation(bool hide)
{
    if (showDeleteButton) {
        return getFadeAnimation(m_delete, hide, 150);
    } else {
        return 0;
    }
}

QAbstractAnimation *CityListItem::showAnimation()
{
    return getFadeAnimation(this, false, 150);
}

QAbstractAnimation *CityListItem::hideAnimation()
{
    return getFadeAnimation(this, true, 150);
}

template<class T>QAbstractAnimation *CityListItem::getFadeAnimation(T *target, bool hide, int msecs)
{
    target->setOpacity(hide ? 1.0 : 0.0);
    target->show();
    QPropertyAnimation* lResult = new QPropertyAnimation(target, "opacity");
    lResult->setEasingCurve(QEasingCurve::OutExpo);
    lResult->setStartValue(hide ? 1.0 : 0.0);
    lResult->setEndValue(hide ? 0.0 : 1.0);
    lResult->setDuration(msecs);
    return lResult;
}

// CityContentList

CityContentList::CityContentList(QObject *holder)
    : ContentList(0),
      m_holder(holder)
{
    ForecastModel *model = ForecastModel::instance();

    connect(model, SIGNAL(itemAdded(const ForecastData &)),
            this, SLOT(addForecast(const ForecastData &)));

    QList<ContentListItem *> items;

    for (int i = 0; i < model->count(); i++) {
        ForecastData data = model->itemAt(i);
        items.append(new CityListItem(data, this, model->count() > minListCount));
    }

    appendItems(items);
}

void CityContentList::addForecast(const ForecastData &data)
{
    ForecastModel *model = ForecastModel::instance();
    addItem(new CityListItem(data, this, model->count() > minListCount));
}

void CityContentList::select(const QString &selected)
{
    for (int i = 0; i < itemCount(); ++i) {
        CityListItem *item = static_cast<CityListItem*>(itemAt(i));
        if (item)
            item->select(item->data().key() == selected);
    }
}

QAbstractAnimation *CityContentList::insertAnimation(int idx, qreal height)
{
    if (idx < 0 || idx > itemCount())
        return 0;

    QList<QAbstractAnimation*> list;

    if (m_holder && itemCount() < maxVisibleItems)
        list.append(getMoveAnimation(m_holder, -height));

    if (idx < itemCount()) {
        for (int i = idx; i < itemCount(); ++i)
            list.append(getItemMoveAnimation(i, height));
    }

    // move to show item insertion
    ImtkScrollArea *scrollArea = qobject_cast<ImtkScrollArea *>(parentWidget());
    if (scrollArea) {
        QPropertyAnimation *anim = new QPropertyAnimation(scrollArea, "offset");
        anim->setEndValue(idx * height);
        anim->setDuration(200);
        list.append(anim);
    }

    if (itemCount() == minListCount) {
        for (int i = 0; i < minListCount; ++i) {
            CityListItem *item = static_cast<CityListItem*>(itemAt(i));
            if (item) {
                QAbstractAnimation *ab = item->buttonAnimation(false);
                if (ab)
                    list.append(ab);
            }
        }
    }

    return createCompoundAnimation(list);
}

QAbstractAnimation *CityContentList::removeAnimation(int idx)
{
    if (idx < 0 || idx >= itemCount())
        return 0;

    qreal itemHeight = itemAt(idx)->contentHeight();

    qreal offset = -itemHeight;
    QList<QAbstractAnimation*> list;

    QSequentialAnimationGroup *result = new QSequentialAnimationGroup();

    QAbstractAnimation *hideAnim = itemAt(idx)->hideAnimation();
    if (hideAnim)
        result->addAnimation(hideAnim);

    if (m_holder && itemCount() <= maxVisibleItems)
        list.append(getMoveAnimation(m_holder, -offset));

    for (int i = idx + 1; i < itemCount(); ++i)
        list.append(getItemMoveAnimation(i, offset));

    if (itemCount() == minListCount + 1) {
        for (int i = 0; i <= minListCount; ++i) {
            if (i != idx) {
                CityListItem *item = static_cast<CityListItem*>(itemAt(i));
                if (item) {
                    QAbstractAnimation *ab = item->buttonAnimation(true);
                    if (ab)
                        list.append(ab);
                }
            }
        }
    }

    ForecastModel *model = ForecastModel::instance();
    model->remove(model->itemAt(idx));

    QAbstractAnimation *compoundAnim = createCompoundAnimation(list);
    if (compoundAnim)
        result->addAnimation(compoundAnim);

    return result;
}

QAbstractAnimation *CityContentList::createCompoundAnimation(QList<QAbstractAnimation*> list)
{
    if (list.count() == 0)
        return 0;
    if (list.count() == 1)
        return list[0];
    QParallelAnimationGroup *result = new QParallelAnimationGroup();
    for (int i = 0; i < list.count(); ++i)
        result->addAnimation(list[i]);
    return result;

}

QAbstractAnimation *CityContentList::getMoveAnimation(QObject *object, qreal offset)
{
    if (!object)
        return 0;
    qreal itemTop = object->property("y").toReal();
    QPropertyAnimation* lResult = new QPropertyAnimation(object, "y");

    lResult->setEasingCurve(QEasingCurve::OutExpo);
    lResult->setStartValue(itemTop);
    lResult->setEndValue(itemTop + offset);
    lResult->setDuration(150);
    return lResult;
}

QAbstractAnimation *CityContentList::getItemMoveAnimation(int idx, qreal offset)
{
    return getMoveAnimation(itemAt(idx), offset);
}

// CityList

CityList::CityList(QGraphicsItem *parent)
    : QGraphicsWidget(parent),
      m_panelTop(PixmapLoader::pixmap("city_panel_bg")),
      m_itemHeight(ITEM_BACKGROUND.height()),
      m_list(new CityContentList(this))
{
    int t = Settings::scaleHeight(25);
    int w = Settings::windowSize().width();

    resize(w, t + maxVisibleItems * m_itemHeight);

    m_scrollBox = new ImtkScrollArea(this);
    m_scrollBox->setWidget(m_list);
    m_scrollBox->setGeometry(0, t, w, maxVisibleItems * m_itemHeight);

    const QPixmap &scrollPixmap = PixmapLoader::pixmap("scroll");
    const QPixmap &scrollKnobPixmap = PixmapLoader::pixmap("scroll_knob");

    m_scroll = new ImtkScrollBar(scrollPixmap, scrollKnobPixmap, this);
    m_scroll->setFadeEffect(true);
    m_scroll->setKnobBorders(0, 6, 0, 6);

    m_scroll->resize(scrollPixmap.size().width(),
                     Settings::scaleHeight(500));
    m_scroll->setPos(Settings::scaleWidth(465.0), Settings::scaleHeight(50));
    m_scroll->setZValue(1.0);

    updateScrollBar();
    m_scroll->setPageSize(size().height());

    connect(m_scrollBox, SIGNAL(offsetChanged()), SLOT(updateScrollBar()));
    connect(m_scrollBox, SIGNAL(maximumOffsetChanged()), SLOT(updateScrollBar()));
}

void CityList::updateScrollBar()
{
    m_scroll->setValue(m_scrollBox->offset());
    m_scroll->setMaximum(m_scrollBox->maximumOffset());
}

QGraphicsWidget *CityList::scroll() const
{
    return m_scrollBox;
}

CityContentList *CityList::contents() const
{
    return m_list;
}

qreal CityList::initialTop() const
{
    int count = maxVisibleItems - m_list->itemCount();
    return qMax<qreal>(0, count * m_itemHeight);
}

int CityList::loadImages()
{
    PixmapLoader::load("scroll");
    PixmapLoader::load("scroll_knob");
    PixmapLoader::load("list_top");
    PixmapLoader::load("list_item_bg");
    PixmapLoader::load("list_item_selected_bg");
    PixmapLoader::load("button_list_delete");
    PixmapLoader::load("list_check");
    return 5;
}

void CityList::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);
}

void CityList::paint(QPainter *painter, const QStyleOptionGraphicsItem *, QWidget *)
{
    painter->drawPixmap(0, 0, m_panelTop);
    painter->fillRect(0, m_panelTop.height() - 1, size().width(),
                      size().height() - m_panelTop.height() + 1, QColor(7, 18, 23));
}
