/****************************************************************************
**
** Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights.  These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** Nokia at qt-info@nokia.com.
**
**
**
**
**
**
**
**
** $QT_END_LICENSE$
**
****************************************************************************/
#include "qdebug.h"
#include "qhildoninputcontext_p.h"
#include "qpointer.h"
#include "qapplication.h"
#include "qclipboard.h"
#include "qplaintextedit.h"
#include "qlineedit.h"
#include "qtextedit.h"
#include "qtextbrowser.h"
#include "kernel/qevent_p.h"       //QKeyEventEx
#include "kernel/qapplication_p.h" //QApplicationPrivate::areXInputEventsUsed()
#include "qinputcontext.h"
#include "qtextcodec.h"
#include <private/qkeymapper_p.h>

#include "qgraphicsview.h"
#ifndef QT_NO_GRAPHICSVIEW
#include "qgraphicsitem.h"
#include "qgraphicsproxywidget.h"
#include "qgraphicsscene.h"
#include "qgraphicswidget.h"
#endif

#ifdef Q_WS_MAEMO_5

#define GDK_ISO_ENTER  0xfe34
#define COMPOSE_KEY    Qt::Key_Multi_key   // "Ch" key
#define LEVEL_KEY      Qt::Key_AltGr       //"Fn" key

#define STATE_LEVEL_MASK    1 << 7
#define STATE_CONTROL_MASK  1 << 2
#define STATE_SHIFT_MASK    1 << 0

//Keyboard layout levels
#define NUMERIC_LEVEL 2
#define LOCKABLE_LEVEL 4

#define PRESSURE_THRESHOLD  0.40

/* TODO
   - Cleaning up

   Fremantle:
   - Read Gconf (?) Settings for Auto-Capitalization, Word completion and insert space after word
     Currently these are switched on.
   - TRIGGER STYLUS is sent all the time, this prevent the usage of fullscreen virtual keyboard (Bug tracked, waiting for a tix)
   - sendSurrounding when sendAllContents == true
   - hildon banner integration to show the modifiers status (Sticky/Lock).
   - put Q_WS_MAEMO_5 in configure

   Diablo
   - Fix dead key support
*/

extern bool qt_sendSpontaneousEvent(QObject*, QEvent*); //qapplication_x11.cpp

#define HIM_DEBUG

static inline bool qHimDebugEnabled()
{
    static const bool debug = !qgetenv("QT_HIM_DEBUG").isEmpty();
    return debug;
}

#ifdef HIM_DEBUG
#  define qHimDebug  if (!qHimDebugEnabled()) {} else qDebug
#else
#  define qHimDebug  while (false) qDebug
#endif

#define LOGMESSAGE1(x)       qHimDebug() << x;
#define LOGMESSAGE2(x, y)    qHimDebug() << x << "(" << y << ")";
#define LOGMESSAGE3(x, y, z) qHimDebug() << x << "(" << y << " " << z << ")";


QMap<QWidget *, QHIMProxyWidget *> QHIMProxyWidget::proxies;

QHIMProxyWidget::QHIMProxyWidget(QWidget *widget)
    : QWidget(0), w(widget)
{
    hide();
    setAttribute(Qt::WA_InputMethodEnabled);
    setAttribute(Qt::WA_NativeWindow);
    createWinId();
    connect(w, SIGNAL(destroyed()), this, SLOT(widgetWasDestroyed()));
}

QHIMProxyWidget::~QHIMProxyWidget()
{
}

QWidget *QHIMProxyWidget::widget() const
{
    return w;
}

QHIMProxyWidget *QHIMProxyWidget::proxyFor(QWidget *w)
{
    QHIMProxyWidget *proxy = qobject_cast<QHIMProxyWidget *>(w);

    if (!proxy)
        proxy = proxies.value(w);

    if (!proxy) {
        proxy = new QHIMProxyWidget(w);
        proxies.insert(w, proxy);
    }
    //qWarning() << "Using HIM Proxy widget" << proxy << "for widget" << w << "isnative: " << proxy->testAttribute(Qt::WA_NativeWindow) << " / " << w->testAttribute(Qt::WA_NativeWindow);

    return proxy;
}

void QHIMProxyWidget::widgetWasDestroyed()
{
    proxies.remove(w);
    delete this;
}


/*! XkbLookupKeySym ( X11->display, event->nativeScanCode(), HILDON_IM_SHIFT_STICKY_MASK, &mods_rtrn, sym_rtrn)
 */
static QString translateKeycodeAndState(KeyCode key, uint state, quint32 &keysym){
    uint mods;
    KeySym *ks = reinterpret_cast<KeySym*>(&keysym);
    if ( XkbLookupKeySym ( X11->display, key, state, &mods, ks) )
        return QKeyMapperPrivate::maemo5TranslateKeySym(*ks);
    else
        return QString();
}

static Window findHildonIm()
{
    union
    {
        Window *win;
        unsigned char *val;
    } value;

    Window result = 0;
    ulong n = 0;
    ulong extra = 0;
    int format = 0;
    Atom realType;

    int status = XGetWindowProperty(X11->display, QX11Info::appRootWindow(),
                    ATOM(_HILDON_IM_WINDOW), 0L, 4L, 0,
                    XA_WINDOW, &realType, &format,
                    &n, &extra, (unsigned char **) &value.val);

    if (status == Success && realType == XA_WINDOW
          && format == HILDON_IM_WINDOW_ID_FORMAT && n == 1 && value.win != 0) {
        result = value.win[0];
        XFree(value.val);
    } else {
        qWarning("QHildonInputContext: Unable to get the Hildon IM window id");
    }

    return result;
}



/*! Send a key event to the IM, which makes it available to the plugins
 */
static void sendKeyEvent(QWidget *widget, QEvent::Type type, uint state, uint keyval, quint16 keycode)
{
    int gdkEventType;
    Window w = findHildonIm();

    if (!w)
        return;

    //Translate QEvent::Type in GDK_Event
    switch (type){
        case QEvent::KeyPress:
            gdkEventType = 8;
        break;
        case QEvent::KeyRelease:
            gdkEventType = 9;
        break;
        default:
            qWarning("QHildonInputContext: Event type not allowed");
            return;
    }

    XEvent ev;
    memset(&ev, 0, sizeof(XEvent));

    ev.xclient.type = ClientMessage;
    ev.xclient.window = w;
    ev.xclient.message_type = ATOM(_HILDON_IM_KEY_EVENT);
    ev.xclient.format = HILDON_IM_KEY_EVENT_FORMAT;

    HildonIMKeyEventMessage *msg = reinterpret_cast<HildonIMKeyEventMessage *>(&ev.xclient.data);
    msg->input_window = QHIMProxyWidget::proxyFor(widget)->winId();

    msg->type = gdkEventType;
    msg->state = state;
    msg->keyval = keyval;
    msg->hardware_keycode = keycode;

    XSendEvent(X11->display, w, false, 0, &ev);
    XSync( X11->display, false );
}

static quint32 dead_key_to_unicode_combining_character(int /*qtkeycode*/)
{
  quint32 combining = 0; //Unicode Hex value

#if 0
  //TODO Diablo - Not required in Fremantle
  switch (qtkeycode)
  {
    case Qt::Key_Dead_Grave:            combining = 0x0300; break;
    case Qt::Key_Dead_Acute:            combining = 0x0301; break;
    case Qt::Key_Dead_Circumflex:       combining = 0x0302; break;
    case Qt::Key_Dead_Tilde:            combining = 0x0303; break;
    case Qt::Key_Dead_Macron:           combining = 0x0304; break;
    case Qt::Key_Dead_Breve:            combining = 0x032e; break;
    case Qt::Key_Dead_Abovedot:         combining = 0x0307; break;
    case Qt::Key_Dead_Diaeresis:        combining = 0x0308; break;
    case Qt::Key_Dead_Abovering:        combining = 0x030a; break;
    case Qt::Key_Dead_Doubleacute:      combining = 0x030b; break;
    case Qt::Key_Dead_Caron:            combining = 0x030c; break;
    case Qt::Key_Dead_Cedilla:          combining = 0x0327; break;
    case Qt::Key_Dead_Ogonek:           combining = 0x0328; break;
    case Qt::Key_Dead_Iota:             combining = 0; break; /* Cannot be combined */
    case Qt::Key_Dead_Voiced_Sound:     combining = 0; break; /* Cannot be combined */
    case Qt::Key_Dead_Semivoiced_Sound: combining = 0; break; /* Cannot be combined */
    case Qt::Key_Dead_Belowdot:         combining = 0x0323; break;
    case Qt::Key_Dead_Hook:             combining = 0x0309; break;
    case Qt::Key_Dead_Horn:             combining = 0x031b; break;
    default: combining = 0; break; /* Unknown dead key */
  }
#endif

  return combining;
}

/*! Sends the key as a spontaneous event.
 */
static void sendKey(QWidget *keywidget, int qtCode)
{
    QPointer<QWidget> guard = keywidget;

    KeySym keysym = NoSymbol;
    int keycode;

    switch (qtCode){
        case Qt::Key_Enter:
            keycode = 36;
        break;
        case Qt::Key_Tab:
            keycode = 66;
        break;
        case Qt::Key_Backspace:
            keycode = 22;
        break;
        default:
        qWarning("keycode not allowed");
        return;
    }

    keysym = XKeycodeToKeysym(X11->display, keycode, 0);

    QKeyEventEx click(QEvent::KeyPress, qtCode, Qt::NoModifier , QString(), false, 1, keycode, keysym, 0);
    qt_sendSpontaneousEvent(keywidget, &click);

    // in case the widget was destroyed when the key went down
    if (guard.isNull()){
        return;
    }

    QKeyEventEx release(QEvent::KeyRelease, qtCode, Qt::NoModifier , QString(), false, 1, keycode, keysym, 0);
    qt_sendSpontaneousEvent(keywidget, &release);
}

/*!
 */
static void answerClipboardSelectionQuery(QWidget *widget)
{
    bool hasSelection = !widget->inputMethodQuery(Qt::ImCurrentSelection).toString().isEmpty();

    XEvent xev;
    Window w = findHildonIm();

    memset(&xev, 0, sizeof(xev));
    xev.xclient.type = ClientMessage;
    xev.xclient.window = w;
    xev.xclient.message_type = ATOM(_HILDON_IM_CLIPBOARD_SELECTION_REPLY);
    xev.xclient.format = HILDON_IM_CLIPBOARD_SELECTION_REPLY_FORMAT;
    xev.xclient.data.l[0] = hasSelection;

    XSendEvent(X11->display, w, false, 0, &xev);
}

const char *getNextPacketStart(const char *str)
{
    const char *candidate, *good;

    candidate = good = str;

    while (*candidate != 0) {
        ++candidate;
        if (candidate - str >= ptrdiff_t(HILDON_IM_CLIENT_MESSAGE_BUFFER_SIZE))
            return good;
        good = candidate;
    }

    /* The whole string is small enough */
    return candidate;
}

KeySym getKeySymForLevel(int keycode, int level ){
    XkbDescPtr xkbDesc = XkbGetMap(X11->display, XkbAllClientInfoMask, XkbUseCoreKbd);
    if (!xkbDesc)
        return NoSymbol;

    KeySym keySym = XkbKeySymEntry(xkbDesc, keycode, level, 0);

    //Check for a not repated keysym
    KeySym keySymTest = XkbKeySymEntry(xkbDesc, keycode, 0, 1);
    if (keySym == keySymTest)
        return NoSymbol;

    return keySym;
}

QHildonInputContext::QHildonInputContext(QObject* parent)
    : QInputContext(parent),
      timerId(-1),
      mask(0),
      triggerMode(HILDON_IM_TRIGGER_NONE),
      commitMode(HILDON_IM_COMMIT_REDIRECT),
      inputMode(HILDON_GTK_INPUT_MODE_FULL),
      lastInternalChange(false),
      spaceAfterCommit(false)
{
}

QHildonInputContext::~QHildonInputContext()
{
    sendHildonCommand(HILDON_IM_HIDE);
}

QString QHildonInputContext::identifierName()
{
    return QLatin1String("hildon");
}

QString QHildonInputContext::language()
{
    //TODO GConf /apps/osso/inputmethod/hildon-im-languages
    return QString();
}

/*! \internal
 *  Resolves the focus for a widget inside a QGraphicsView.
 *  Returns the Widget really holding the focus in this case.
 */
QWidget *resolveFocusWidget(QWidget *w)
{
#ifndef QT_NO_GRAPHICSVIEW
    while (QGraphicsView *view = qobject_cast<QGraphicsView *>(w)) {

        QGraphicsScene *scene = view->scene();
        if (scene) {
            QGraphicsItem *item = scene->focusItem();
            QGraphicsProxyWidget *proxy = qgraphicsitem_cast<QGraphicsProxyWidget *>(item);
            if (proxy && proxy->widget() && proxy->widget()->focusWidget() ) {
                w = proxy->widget()->focusWidget();
            } else {
                break;
            }
        } else {
            break;
        }
    }
#endif

    return w;
}

/*!reset the UI state
 */
void QHildonInputContext::reset()
{
    LOGMESSAGE3 ("reset", focusWidget(), QApplication::focusWidget());

    if (realFocus)
        sendHildonCommand(HILDON_IM_CLEAR, realFocus);

    cancelPreedit();

    //Reset internals
    mask = 0;
    lastInternalChange = false;
}

bool QHildonInputContext::isComposing() const
{
    return false;
}

void QHildonInputContext::setFocusWidget(QWidget *w)
{
    // As soon as the virtual keyboard is mapped by the X11 server,
    // it is also activated, which essentially steals our focus.
    // The same happens for the symbol picker.
    // This is a bug in the HIM, that we try to work around here.
    lastFocus = realFocus;

    // Another work around for the GraphicsView.
    // In case of a Widget inside a GraphicsViewProxyWidget we need to remember
    // that it had the focus
    realFocus = resolveFocusWidget(w);

    QInputContext::setFocusWidget(w);

    qHimDebug() << "HIM: setFocusWidget: " << w << " (real: " << realFocus << " / last: " << lastFocus << ")";
}

bool QHildonInputContext::filterEvent(const QEvent *event)
{
     QWidget *w = realFocus;
     if (!w)
         return false;

    switch (event->type()){
    case QEvent::RequestSoftwareInputPanel:{
        //On the device, these events are sent at the same time of the TabletRelease ones
        triggerMode = HILDON_IM_TRIGGER_FINGER;
        inputMode = 0;
        Qt::InputMethodHints hints = w->inputMethodHints();

        // restrictions
        if ((hints & Qt::ImhExclusiveInputMask) == Qt::ImhDialableCharactersOnly) {
            inputMode = HILDON_GTK_INPUT_MODE_TELE;
        } else if (((hints & Qt::ImhExclusiveInputMask) == (Qt::ImhDigitsOnly | Qt::ImhUppercaseOnly)) ||
                   ((hints & Qt::ImhExclusiveInputMask) == (Qt::ImhDigitsOnly | Qt::ImhLowercaseOnly))) {
            inputMode = HILDON_GTK_INPUT_MODE_HEXA;
        } else if ((hints & Qt::ImhExclusiveInputMask) == Qt::ImhDigitsOnly) {
            inputMode = HILDON_GTK_INPUT_MODE_NUMERIC;
        } else if ((hints & Qt::ImhExclusiveInputMask) == Qt::ImhFormattedNumbersOnly) {
            inputMode = HILDON_GTK_INPUT_MODE_NUMERIC | HILDON_GTK_INPUT_MODE_SPECIAL;
        } else {
            inputMode = HILDON_GTK_INPUT_MODE_FULL;
        }

        // behavior flags
        if (hints & Qt::ImhHiddenText)
            inputMode |= HILDON_GTK_INPUT_MODE_INVISIBLE;
        if (!(hints & Qt::ImhNoAutoUppercase))
            inputMode |= HILDON_GTK_INPUT_MODE_AUTOCAP;
        if (!(hints & Qt::ImhNoPredictiveText))
            inputMode |= HILDON_GTK_INPUT_MODE_DICTIONARY;

        qHimDebug("Mapped hint: 0x%x to mode: 0x%x", int(hints), int(inputMode));

        toggleHildonMainIMUi();
        return true;
    }
    case QEvent::KeyPress:
    case QEvent::KeyRelease:
        triggerMode = HILDON_IM_TRIGGER_KEYBOARD;
        return filterKeyPress(w, static_cast<const QKeyEvent *>(event));

    default:
        break;
    }
    return QInputContext::filterEvent(event);
}

//TODO
void QHildonInputContext::update()
{
    LOGMESSAGE1("update");

    if (lastInternalChange) {
        //Autocase update
        checkSentenceStart();
        lastInternalChange = false;
    }
}

/*!  Shows/hides the Hildon Main Input Method Ui Window
 */
void QHildonInputContext::toggleHildonMainIMUi()
{
    LOGMESSAGE1("toggleHildonMainIMUi");

    if (!realFocus)
        return;

    if (!realFocus->inputMethodQuery(Qt::ImCurrentSelection).toString().isEmpty()) {
        sendHildonCommand(HILDON_IM_HIDE);
    } else {
        sendHildonCommand(HILDON_IM_SETCLIENT, realFocus);
        if (timerId != -1)
            killTimer(timerId);
        timerId = startTimer(HILDON_IM_DEFAULT_LAUNCH_DELAY);
    }
}


/*! Shows the IM after a timeout.
 *  GTK implementation use this to improve the possibility
 *  to distinguish a finger poke.
 */
void QHildonInputContext::timerEvent(QTimerEvent *ev)
{
    if (ev->timerId() != timerId)
        return;
    killTimer(timerId);

    if (realFocus)
        sendHildonCommand(HILDON_IM_SETNSHOW, realFocus);
}

/*! Filters spontaneous keyevents then elaborates them and updates the Hildon Main UI
 *  via XMessages. In some cases it creates and posts a new keyevent
 *  as no spontaneous event.
 */
bool QHildonInputContext::filterKeyPress(QWidget *keywidget, const QKeyEvent *event){

    //Avoid to filter events generated by this function.
    // Also ignore non-extended events, since we can't handle those below.
    if (!event->spontaneous() || !event->hasExtendedInfo())
        return false;

    const quint32 state = event->nativeModifiers();
    const quint32 keycode = event->nativeScanCode();
    quint32 keysym= event->nativeVirtualKey();
    const int qtkeycode = event->key();

    qHimDebug("HIM: filterKeyPress Mask: %x state: %x options: %x keycode: %d keysym: %x QtKey: %x",
              mask, state, options, keycode, keysym, qtkeycode);

    //Drop auto repeated keys for COMPOSE_KEY
    if (qtkeycode == COMPOSE_KEY && event->isAutoRepeat()){
        return true;
    }

    //TODO MOVE
    static QWidget* lastKeywidget = 0;
    static int lastQtkeycode = 0;
    static qint32 combiningChar = 0; //Unicode rappresentation of the dead key.

    QString commitString; //String to commit to the Key Widget

    //Reset static vars when the widget change.
    if (keywidget != lastKeywidget){
        mask = 0;
        lastKeywidget = keywidget;
        lastQtkeycode = 0;
        combiningChar = 0;
    }

    if (!qtkeycode)
        return true;

    //1. A dead key will not be immediately commited, but combined with the next key
    if (qtkeycode >= Qt::Key_Dead_Grave && qtkeycode <= Qt::Key_Dead_Horn)
        mask |= HILDON_IM_DEAD_KEY_MASK;
    else
        mask &= ~HILDON_IM_DEAD_KEY_MASK;

    if (mask & HILDON_IM_DEAD_KEY_MASK && combiningChar == 0)
    {
        combiningChar = dead_key_to_unicode_combining_character(qtkeycode);//### WORKS? IMPROVE?
        return true;
    }

    /*2. Pressing any key while the compose key is pressed will keep that
     *   character from being directly submitted to the application. This
     *   allows the IM process to override the interpretation of the key
     */
    if (qtkeycode == COMPOSE_KEY)
    {
        if (event->type() == QEvent::KeyPress)
            mask |= HILDON_IM_COMPOSE_MASK;
        else
            mask &= ~HILDON_IM_COMPOSE_MASK;
    }

    // 3 Sticky and locking keys initialization
    if (event->type() == QEvent::KeyRelease)
    {
        if (qtkeycode == Qt::Key_Shift )
        {
            setMaskState(&mask,
                         HILDON_IM_SHIFT_LOCK_MASK,
                         HILDON_IM_SHIFT_STICKY_MASK,
                         lastQtkeycode == Qt::Key_Shift);
        }else if (qtkeycode == LEVEL_KEY){
            setMaskState(&mask,
                         HILDON_IM_LEVEL_LOCK_MASK,
                         HILDON_IM_LEVEL_STICKY_MASK,
                         lastQtkeycode == LEVEL_KEY);
        }
    }

    //Update lastQtkeycode.
    lastQtkeycode=qtkeycode;

    if (qtkeycode == Qt::Key_Tab){
        commitString = QLatin1String("\t");
    }

    /* 5. When the level key is in sticky or locked state, translate the
     *    keyboard state as if that level key was being held down.
     */
    if ((mask & (HILDON_IM_LEVEL_STICKY_MASK | HILDON_IM_LEVEL_LOCK_MASK)) ||
        (state & STATE_LEVEL_MASK)) {
        commitString = translateKeycodeAndState(keycode, STATE_LEVEL_MASK, keysym);
    }

    /* If the input mode is strictly numeric and the digits are level
     *  shifted on the layout, it's not necessary for the level key to
     *  be pressed at all.
     */
    else if ((options & HILDON_IM_AUTOLEVEL_NUMERIC) &&
             ((inputMode & HILDON_GTK_INPUT_MODE_FULL) == HILDON_GTK_INPUT_MODE_NUMERIC)) {
        KeySym ks = getKeySymForLevel(keycode, NUMERIC_LEVEL);
        QString string = QKeyMapperPrivate::maemo5TranslateKeySym(ks);

        if (!string.isEmpty()) {
            keysym = ks;
            commitString = string;
        }
    }
    /* The input is forced to a predetermined level
     */
    else if (options & HILDON_IM_LOCK_LEVEL)
    {
        KeySym ks = getKeySymForLevel(keycode, LOCKABLE_LEVEL);
        QString string = QKeyMapperPrivate::maemo5TranslateKeySym(ks);

        if (!string.isEmpty()){
            keysym = ks;
            commitString = string;
        }
    }
    /* Hardware keyboard autocapitalization  */
    if (autoUpper && inputMode & HILDON_GTK_INPUT_MODE_AUTOCAP)
    {
        qHimDebug() << "AutoCAP";
        QChar currentChar;
        KeySym lower = NoSymbol;
        KeySym upper = NoSymbol;

        if (commitString.isEmpty()){
            QString ks = QKeyMapperPrivate::maemo5TranslateKeySym(keysym);
            if (!ks.isEmpty())
                currentChar = ks.at(0);
        }else{
            currentChar = commitString.at(0);
        }

        XConvertCase(keysym, &lower, &upper);

        if (currentChar.isPrint()){
            if (state & STATE_SHIFT_MASK){
                currentChar = currentChar.toLower();
                keysym = lower;
            } else {
                currentChar = currentChar.toUpper();
                keysym = upper;
            }
            commitString = QString(currentChar); //sent to the widget
        }
    }

    //6. Shift lock or holding the shift down forces uppercase, ignoring autocap
    if (mask & HILDON_IM_SHIFT_LOCK_MASK || state & STATE_SHIFT_MASK)
    {
        KeySym lower = NoSymbol;
        KeySym upper = NoSymbol;
        XConvertCase(keysym, &lower, &upper);
        QString tempStr = QKeyMapperPrivate::maemo5TranslateKeySym(upper);
        if (!tempStr.isEmpty())
            commitString = tempStr.at(0);
    }else if (mask & HILDON_IM_SHIFT_STICKY_MASK){
        KeySym lower = NoSymbol;
        KeySym upper = NoSymbol;
        QString tempStr = QKeyMapperPrivate::maemo5TranslateKeySym(keysym);
        QChar currentChar;
        if (!tempStr.isEmpty()){
          currentChar = tempStr.at(0);

            /* Simulate shift key being held down in sticky state for non-printables  */
            if ( currentChar.isPrint() ){
                /*  For printable characters sticky shift negates the case,
                 *  including any autocapitalization changes
                 */
                if ( currentChar.isUpper() ){
                    currentChar = currentChar.toLower();
                    lower = lower;
                }else{
                    currentChar = currentChar.toUpper();
                    upper = upper;
                }
                commitString = QString(currentChar); //sent to the widget
            }
        }
    }

    //F. word completion manipulation (for fremantle)
    if (event->type() == QEvent::KeyPress &&
        previousCommitMode == HILDON_IM_COMMIT_PREEDIT &&
        !preEditBuffer.isNull())
    {
        switch (qtkeycode){
            case Qt::Key_Right:{
                //TODO Move this code in commitPreeditData();
                QInputMethodEvent e;

                if (spaceAfterCommit)
                    e.setCommitString(preEditBuffer + QLatin1Char(' '));
                else
                    e.setCommitString(preEditBuffer);

                QApplication::sendEvent(keywidget, &e);
                preEditBuffer.clear();
                return true;
            }
            case Qt::Key_Backspace:
            case Qt::Key_Up:
            case Qt::Key_Down:
            case Qt::Key_Left:{
                cancelPreedit();
                return true;
           }

           case Qt::Key_Return:
           case Qt::Key_Enter: {
                cancelPreedit();
                break;
           }
           default: {
               if (keysym == GDK_ISO_ENTER)
                   cancelPreedit();
               break;
           }
        }
    }

    //7. Sticky and lock state reset
    if (event->type() == QEvent::KeyRelease)
    {
        if (qtkeycode != Qt::Key_Shift )
        {
            /* If not locked, pressing any character resets shift state */
            if ((mask & HILDON_IM_SHIFT_LOCK_MASK) == 0)
            {
                mask &= ~HILDON_IM_SHIFT_STICKY_MASK;
            }
        }
        if (qtkeycode != LEVEL_KEY)
        {
            /* If not locked, pressing any character resets level state */
            if ((mask & HILDON_IM_LEVEL_LOCK_MASK) == 0)
            {
                mask &= ~HILDON_IM_LEVEL_STICKY_MASK;
            }
        }
    }

    if (event->type() == QEvent::KeyRelease || state & STATE_CONTROL_MASK)
    {
        QString debug = QLatin1String("Sending state=0x%1 keysym=0x%2 keycode=%3");
        LOGMESSAGE2(" - ", debug.arg(state,0,16).arg(keysym,0,16).arg(keycode));

        sendKeyEvent(keywidget, event->type(), state, keysym, keycode);
        return false;
    }


    /* 8. Pressing a dead key twice, or if followed by a space, inputs
     *    the dead key's character representation
     */
    if ((mask & HILDON_IM_DEAD_KEY_MASK || qtkeycode == Qt::Key_Space) && combiningChar)
    {
        qint32 last;
        last = dead_key_to_unicode_combining_character (qtkeycode);
        if ((last == combiningChar) || qtkeycode == Qt::Key_Space)
        {
            commitString = QString(combiningChar);
        }else{
            commitString = QString::fromUtf8(XKeysymToString(keysym));
        }
        combiningChar = 0;
    }else{
        /* Regular keypress */
        if (mask & HILDON_IM_COMPOSE_MASK)
        {
            sendKeyEvent(keywidget, event->type(),state, keysym, keycode);
            return true;
        }else{
            if ( commitString.isEmpty() && qtkeycode != Qt::Key_Backspace){
                LOGMESSAGE3(" - ", "text sent to IM", event->text())
                commitString = QString(event->text());
            }
        }
    }

    /* Control keys should not produce commitString, if they do it's a bug
       on keymap side and we have to workaround this here */
    if (qtkeycode == Qt::Key_Return || qtkeycode == Qt::Key_Enter ||
        keysym == GDK_ISO_ENTER || qtkeycode == Qt::Key_Backspace) {
        commitString = QString();
        lastInternalChange = true;
    } else if (qtkeycode == Qt::Key_Shift || qtkeycode == Qt::Key_AltGr ||
               qtkeycode == Qt::Key_Control) {
        commitString = QString();
    }

    if (!commitString.isEmpty()){
        //entering a new character cleans the preedit buffer
        cancelPreedit();

        /* Pressing a dead key followed by a regular key combines to form
         * an accented character
         */
        if (combiningChar){ //FIXME
            commitString.append(combiningChar);//This will be sent to the widget
            const char *charStr = qPrintable(commitString);
            keysym = XStringToKeysym(charStr); //This will be sent to the IM
        }

        //Create the new event with the elaborate information,
        //then it adds the event to the events queue
        {
            QEvent::Type type = event->type();
            Qt::KeyboardModifiers modifiers= event->modifiers();
            //WARNING the qt keycode has not been updated!!
            QKeyEventEx *ke= new QKeyEventEx(type, keycode, modifiers, commitString, false, commitString.size(), keycode, keysym, state);
            QCoreApplication::postEvent(keywidget,ke);
        }

        //Send the new keysym
        sendKeyEvent(keywidget, event->type(), state, keysym, keycode);
#if 0
        /* Non-printable characters invalidate any previous dead keys */
        if (qtkeycode != Qt::Key_Shift)
            combiningChar=0;
#endif
        lastInternalChange = true;
        return true;
    } else {
        //Send the new keysym
        sendKeyEvent(keywidget, event->type(), state, keysym, keycode);
        return false;
    }
}

/*! Filters the XClientMessages sent by QApplication_x11
 */
bool QHildonInputContext::x11FilterEvent(QWidget *keywidget, XEvent *event)
{
    LOGMESSAGE3("x11FilterEvent", keywidget, event);

    if (QHIMProxyWidget *proxy = qobject_cast<QHIMProxyWidget *>(keywidget))
        keywidget = proxy->widget();

    if (event->xclient.message_type == ATOM(_HILDON_IM_INSERT_UTF8)
            && event->xclient.format == HILDON_IM_INSERT_UTF8_FORMAT) {
        HildonIMInsertUtf8Message *msg = (HildonIMInsertUtf8Message *)&event->xclient.data;
        insertUtf8(msg->msg_flag, QString::fromUtf8(msg->utf8_str));
        return true;

    } else if (event->xclient.message_type == ATOM(_HILDON_IM_COM)) {
        HildonIMComMessage *msg = (HildonIMComMessage *)&event->xclient.data;
        options = msg->options;

        LOGMESSAGE3 (" - ", "Options=", options)

        switch (msg->type) {
        //Handle Keys msgs
        case HILDON_IM_CONTEXT_HANDLE_ENTER: {
            sendKey(keywidget, Qt::Key_Enter);
            return true; }
        case HILDON_IM_CONTEXT_HANDLE_TAB: {
            sendKey(keywidget, Qt::Key_Tab);
            return true; }
        case HILDON_IM_CONTEXT_HANDLE_BACKSPACE: {
            sendKey(keywidget, Qt::Key_Backspace);
            return true; }
        case HILDON_IM_CONTEXT_HANDLE_SPACE: {
            insertUtf8(HILDON_IM_MSG_CONTINUE, QChar(Qt::Key_Space));
            commitPreeditData();
            return true; }

        //Handle Clipboard msgs
        case HILDON_IM_CONTEXT_CLIPBOARD_SELECTION_QUERY: {
            answerClipboardSelectionQuery(keywidget);
            return true; }
        case HILDON_IM_CONTEXT_CLIPBOARD_PASTE:
            if (QClipboard *clipboard = QApplication::clipboard()){
                QInputMethodEvent e;
                e.setCommitString(clipboard->text());
                sendEvent(e);
            }
            return true;
        case HILDON_IM_CONTEXT_CLIPBOARD_COPY: {
            if (QClipboard *clipboard = QApplication::clipboard())
                clipboard->setText(keywidget->inputMethodQuery(Qt::ImCurrentSelection).toString());
            return true; }
        case HILDON_IM_CONTEXT_CLIPBOARD_CUT: {
            if (QClipboard *clipboard = QApplication::clipboard())
                clipboard->setText(keywidget->inputMethodQuery(Qt::ImCurrentSelection).toString());
            QInputMethodEvent ev;
            sendEvent(ev);
            return true; }

        //Handle commit mode msgs
        case HILDON_IM_CONTEXT_DIRECT_MODE: { //default mode
            LOGMESSAGE3 (" - ", "Commit Mode=", "Direct")
            preEditBuffer.clear();
            commitMode = HILDON_IM_COMMIT_DIRECT;
            return true; }
        case HILDON_IM_CONTEXT_BUFFERED_MODE: {
            LOGMESSAGE3 (" - ", "Commit Mode=", "Buffered")
            if (commitMode != HILDON_IM_COMMIT_BUFFERED){
                preEditBuffer = QLatin1String("");
                commitMode = HILDON_IM_COMMIT_BUFFERED;
            }
            return true; }
        case HILDON_IM_CONTEXT_REDIRECT_MODE: {
            LOGMESSAGE3 (" - ", "Commit Mode=", "Redirect")
            preEditBuffer.clear();
            commitMode = HILDON_IM_COMMIT_REDIRECT;
            checkCommitMode();
            clearSelection();
            return true; }
        case HILDON_IM_CONTEXT_SURROUNDING_MODE: {
            LOGMESSAGE3 (" - ", "Commit Mode=", "Surrounding")
            preEditBuffer.clear();
            commitMode = HILDON_IM_COMMIT_SURROUNDING;
            return true; }

        //Handle context
        case HILDON_IM_CONTEXT_CONFIRM_SENTENCE_START: {
            LOGMESSAGE2(" - ", "XMessage: Sentence start")
            checkSentenceStart();
            return true; }
        case HILDON_IM_CONTEXT_FLUSH_PREEDIT: {
            LOGMESSAGE2(" - ", "XMessage: Flush preedit")
            commitPreeditData();
            return true; }
        case HILDON_IM_CONTEXT_REQUEST_SURROUNDING: {
            LOGMESSAGE2(" - ", "XMessage: Request surrounding")
            checkCommitMode();
            sendSurrounding();
            //if (self->is_url_entry)
            //  hildon_im_context_send_command(self, HILDON_IM_SELECT_ALL);
            return true; }
        case HILDON_IM_CONTEXT_OPTION_CHANGED: {
            //Nothing to do
            return true; }
        case HILDON_IM_CONTEXT_CLEAR_STICKY: {
            mask &= ~(HILDON_IM_SHIFT_STICKY_MASK |
                      HILDON_IM_SHIFT_LOCK_MASK |
                      HILDON_IM_LEVEL_STICKY_MASK |
                      HILDON_IM_LEVEL_LOCK_MASK);
            return true; }

        //Unused
        case HILDON_IM_CONTEXT_NUM_COM: {
            return true; }
        case HILDON_IM_CONTEXT_ENTER_ON_FOCUS: {
            return true; }
        case HILDON_IM_CONTEXT_WIDGET_CHANGED: {
            //Do we really need to set the mask=0? See reset();
            return true; }

#ifdef Q_WS_MAEMO_5
        case HILDON_IM_CONTEXT_CANCEL_PREEDIT: {
            LOGMESSAGE3(" - ", "XMessage: Cancel preedit=", preEditBuffer)
            cancelPreedit();
            return true; }
        case HILDON_IM_CONTEXT_PREEDIT_MODE: {
            LOGMESSAGE3 (" - ", "Commit Mode=", "Preedit")
            preEditBuffer.clear();
            previousCommitMode = commitMode;
            commitMode = HILDON_IM_COMMIT_PREEDIT;
            return true; }
        case HILDON_IM_CONTEXT_REQUEST_SURROUNDING_FULL: {
            LOGMESSAGE2(" - ", "XMessage: request surrounding full")
            checkCommitMode();
            sendSurrounding(true);
            //if (self->is_url_entry)
            //  hildon_im_context_send_command(self, HILDON_IM_SELECT_ALL);
            return true; }
        case HILDON_IM_CONTEXT_SPACE_AFTER_COMMIT: {
            LOGMESSAGE2(" - ", "XMessage: put a space after commit")
            spaceAfterCommit = true;
            return true; }
        case HILDON_IM_CONTEXT_NO_SPACE_AFTER_COMMIT: {
            LOGMESSAGE2(" - ", "XMessage: no space after commit")
            spaceAfterCommit = false;
            return true; }
#endif
        default:
            qWarning() << "HIM: (warning) Message not handled:" << msg->type;
        }
    }else if (event->xclient.message_type == ATOM(_HILDON_IM_SURROUNDING_CONTENT) &&
              event->xclient.format == HILDON_IM_SURROUNDING_CONTENT_FORMAT) {
        HildonIMSurroundingContentMessage *msg = reinterpret_cast<HildonIMSurroundingContentMessage*>(&event->xclient.data);
        LOGMESSAGE3(" - ", "SurroundingContentMessage=", "msg->surrounding")

        if (!surrounding.isNull()) {
            if (msg->msg_flag == HILDON_IM_MSG_START) {
                surrounding.clear();
            }else if (msg->msg_flag == HILDON_IM_MSG_END) {
                //TODO? commitSurrounding();
                qWarning("commitSurrounding() NOT IMPLEMENTED YET");
                return true;
            }
        }

        surrounding += QString::fromUtf8(msg->surrounding);
        return true;
    }else if (event->xclient.message_type == ATOM(_HILDON_IM_SURROUNDING) &&
                  event->xclient.format == HILDON_IM_SURROUNDING_FORMAT) {
        LOGMESSAGE2(" - ", "XMessage: IM Surrounding")
        HildonIMSurroundingMessage *msg = reinterpret_cast<HildonIMSurroundingMessage*>(&event->xclient.data);
        setClientCursorLocation(msg->offset_is_relative, msg->cursor_offset );
        return true;
    }
    return false;
}

/*! Ask the client widget to insert the specified text at the cursor
 *  position, by triggering the commit signal on the context
 */
void QHildonInputContext::insertUtf8(int flag, const QString& text)
{
    LOGMESSAGE3("insertUtf8", flag, text)

    qHimDebug() << "HIM: insertUtf8: " << text << " (real: " << realFocus << " / last: " << lastFocus << ")";
    QWidget *w = realFocus;
    if (!w)
        w = lastFocus;
    if (!w)
        return;

    QString cleanText = text;
    if (mask & HILDON_IM_SHIFT_LOCK_MASK)
            cleanText = cleanText.toUpper();

    lastInternalChange = true;

    //TODO HILDON_IM_AUTOCORRECT is used by the hadwriting plugin
    //Writing CiAo in the plugin add Ciao in the widget.
    if (options & HILDON_IM_AUTOCORRECT){
        qWarning() << "HILDON_IM_AUTOCORRECT Not Implemented Yet";
    }


    //Delete suroundings when we are using the preeditbuffer.
    // Eg: For the HandWriting plugin
    if (!preEditBuffer.isNull()) {
        //Updates preEditBuffer
        if (flag != HILDON_IM_MSG_START) {
            preEditBuffer.append(cleanText);
            cleanText = preEditBuffer;
        }
     }

    //Adds the actual text
    switch (commitMode) {
        case HILDON_IM_COMMIT_PREEDIT: { //Fremantle
            if (preEditBuffer.isNull())
                preEditBuffer = cleanText;

            //Creating attribute list
            QList<QInputMethodEvent::Attribute> attributes;

            QTextCharFormat textCharFormat;
            textCharFormat.setFontUnderline(true);
            textCharFormat.setBackground(w->palette().highlight());
            textCharFormat.setForeground(w->palette().base());
            attributes << QInputMethodEvent::Attribute(QInputMethodEvent::TextFormat, 0, cleanText.length(), textCharFormat);
            attributes << QInputMethodEvent::Attribute(QInputMethodEvent::Cursor, 0, cleanText.length(), QVariant());
            QInputMethodEvent e(cleanText, attributes);
            QApplication::sendEvent(w, &e);

            //Reset commit mode
            if (flag == HILDON_IM_MSG_END){
                commitMode = previousCommitMode;
                previousCommitMode = HILDON_IM_COMMIT_PREEDIT;
            }

        }break;
        case HILDON_IM_COMMIT_BUFFERED: //Diablo Handwriting
        case HILDON_IM_COMMIT_DIRECT:
        case HILDON_IM_COMMIT_REDIRECT:{
            QInputMethodEvent e;
            e.setCommitString(cleanText);
            QApplication::sendEvent(w, &e);
        }break;
        default:
            qWarning() << "Commit mode " << commitMode << " not handled by InsertText method";
    }
}

void QHildonInputContext::clearSelection()
{
    LOGMESSAGE1("clearSelection");

    QWidget *w = realFocus;
    if (!w)
        w = lastFocus;
    if (!w)
        return;

    int textCursorPos = w->inputMethodQuery(Qt::ImCursorPosition).toInt();
    QString selection = w->inputMethodQuery(Qt::ImCurrentSelection).toString();

    if (selection.isEmpty())
        return;

    //Remove the selection
    QInputMethodEvent e;
    e.setCommitString(selection);
    QApplication::sendEvent(w, &e);

    //Move the cursor backward if the text has been selected from right to left
    if (textCursorPos < textCursorPosOnPress){
        QInputMethodEvent e;
        e.setCommitString(QString(), -selection.length(),0);
        QApplication::sendEvent(w, &e);
    }
}

void QHildonInputContext::cancelPreedit()
{
    LOGMESSAGE1("cancelPreedit");

    QWidget *w = realFocus;
    if (!w)
        w = lastFocus;
    if (!w)
        return;

    if (preEditBuffer.isEmpty())
        return;

    preEditBuffer.clear();

    QInputMethodEvent e;
    QApplication::sendEvent(w, &e);
}

void QHildonInputContext::sendHildonCommand(HildonIMCommand cmd, QWidget *widget)
{
    LOGMESSAGE3("sendHildonCommand", cmd, widget);

    Window w = findHildonIm();

    if (!w)
        return;

    XEvent ev;
    memset(&ev, 0, sizeof(XEvent));

    ev.xclient.type = ClientMessage;
    ev.xclient.window = w;
    ev.xclient.message_type = ATOM(_HILDON_IM_ACTIVATE);
    ev.xclient.format = HILDON_IM_ACTIVATE_FORMAT;

    HildonIMActivateMessage *msg = reinterpret_cast<HildonIMActivateMessage *>(&ev.xclient.data);

    if (widget) {
        msg->input_window = QHIMProxyWidget::proxyFor(widget)->winId();
        msg->app_window = widget->window()->winId();
    } else if (cmd != HILDON_IM_HIDE) {
        qWarning() << "Invalid Hildon Command:" << cmd;
        return;
    }

    if (cmd == HILDON_IM_HIDE && timerId != -1)
        killTimer(timerId);

    if (cmd == HILDON_IM_SETCLIENT || cmd == HILDON_IM_SETNSHOW)
        sendInputMode();

    msg->cmd = cmd;
    msg->trigger = triggerMode;

    XSendEvent(X11->display, w, false, 0, &ev);
    XSync(X11->display, False);
}


/*!
 */
void QHildonInputContext::sendX11Event(XEvent *event)
{
    LOGMESSAGE2("sendX11Event", event);

    Window w = findHildonIm();

    if (!w)
        return;

    event->xclient.type = ClientMessage;
    event->xclient.window = w;

    XSendEvent(X11->display, w, false, 0, event);
    XSync(X11->display, False);
}

//CONTEXT
/*! Updates the IM with the autocap state at the active cursor position
 */
void QHildonInputContext::checkSentenceStart()
{
    LOGMESSAGE1("checkSentenceStart");

    if (!realFocus)
        return;

    QWidget *w = realFocus;

    if ((inputMode & (HILDON_GTK_INPUT_MODE_ALPHA | HILDON_GTK_INPUT_MODE_AUTOCAP)) !=
            (HILDON_GTK_INPUT_MODE_ALPHA | HILDON_GTK_INPUT_MODE_AUTOCAP)) {
        /* If autocap is off, but the mode contains alpha, send autocap message.
         * The important part is that when entering a numerical entry the autocap
         * is not defined, and the plugin sets the mode appropriate for the language */
        if (inputMode & HILDON_GTK_INPUT_MODE_ALPHA) {
            autoUpper = false;
            sendHildonCommand(HILDON_IM_LOW, w);
        }
        return;
    }

    QString surrounding = w->inputMethodQuery(Qt::ImSurroundingText).toString();
    int cpos = w->inputMethodQuery(Qt::ImCursorPosition).toInt();

    if (surrounding.isEmpty()) {
        autoUpper = true;
        sendHildonCommand(HILDON_IM_UPP, w);
        return;
    }

    QRegExp r(QLatin1String(".*[!?.][\\s]+"));
    QRegExp c(QLatin1String("[\xbf\xa1]"));  // this is "[¿¡]": mirrored question- and exclamation mark
    //Improve performance: Don't make sense analyzing more than N chars before the cursor
    QString left = surrounding.left(cpos).right(10);
    QString right = left.right(1);
    QString notRemoved = left.remove(r);

    if (!notRemoved.count() || right.contains(c)) {
        autoUpper = options & HILDON_IM_AUTOCASE;
        sendHildonCommand(HILDON_IM_UPP, w);
    } else {
        autoUpper = false;
        sendHildonCommand(HILDON_IM_LOW, w);
    }
}

void QHildonInputContext::commitPreeditData()
{
    LOGMESSAGE1("commitPreeditData")

    if (!preEditBuffer.isNull())
        preEditBuffer = QLatin1String("");
}

void QHildonInputContext::checkCommitMode() //### REMOVE?
{
    LOGMESSAGE1("checkCommitMode")

    //if (m_commitMode == HILDON_IM_COMMIT_REDIRECT)
    //    m_commitMode = HILDON_IM_COMMIT_SURROUNDING;
}

/*! Send the text of the client widget surrounding the active cursor position,
 *  as well as the cursor position in the surrounding, to the IM
 */
void QHildonInputContext::sendSurrounding(bool sendAllContents)
{
    LOGMESSAGE2("sendSurrounding", sendAllContents )

    if (!realFocus)
        return;

    QWidget *w = realFocus;

    QString surrounding;
    int cpos;
    if (sendAllContents){

         //Try to detect the kind of widget
        QTextEdit *te = qobject_cast<QTextEdit*>(w);
        QPlainTextEdit *pte = qobject_cast<QPlainTextEdit*>(w);

        if (te){
            surrounding = te->toPlainText();
            cpos = te->textCursor().position();
        }else if (pte){
            surrounding = pte->toPlainText();
            cpos = pte->textCursor().position();
        }else{
            surrounding = w->inputMethodQuery(Qt::ImSurroundingText).toString();
            cpos = w->inputMethodQuery(Qt::ImCursorPosition).toInt();
        }
    }else{
        surrounding = w->inputMethodQuery(Qt::ImSurroundingText).toString();
        cpos = w->inputMethodQuery(Qt::ImCursorPosition).toInt();
    }

    XEvent xev;
    HildonIMSurroundingContentMessage *surroundingContentMsg = 0;
    int flag = HILDON_IM_MSG_START;

    //Sending a null to clean the plugin.
    if (surrounding.isEmpty()) {
        memset(&xev, 0, sizeof(XEvent));
        xev.xclient.message_type = ATOM(_HILDON_IM_SURROUNDING_CONTENT);
        xev.xclient.format = HILDON_IM_SURROUNDING_CONTENT_FORMAT;

        surroundingContentMsg = reinterpret_cast<HildonIMSurroundingContentMessage*>(&xev.xclient.data);
        surroundingContentMsg->msg_flag = flag;
        surroundingContentMsg->surrounding[0] = '\0';

        sendX11Event(&xev);

        sendSurroundingHeader(0);
        return;
    }

    // Split surrounding context into pieces that are small enough
    // to send in a x message
    QByteArray ba = surrounding.toUtf8();
    const char *utf8 = ba.data();
    while (*utf8){
        const char *nextStart = getNextPacketStart(utf8);
        unsigned int len = nextStart - utf8;

        //this call will take care of adding the null terminator
        memset(&xev, 0, sizeof(XEvent));
        xev.xclient.message_type = ATOM(_HILDON_IM_SURROUNDING_CONTENT);
        xev.xclient.format = HILDON_IM_SURROUNDING_CONTENT_FORMAT;

        surroundingContentMsg = reinterpret_cast<HildonIMSurroundingContentMessage*>(&xev.xclient.data);
        surroundingContentMsg->msg_flag = flag;
        memcpy(surroundingContentMsg->surrounding, utf8, len);

        sendX11Event(&xev);

        utf8 = nextStart;
        flag = HILDON_IM_MSG_CONTINUE;
    }
    sendSurroundingHeader(cpos);
}

void QHildonInputContext::sendSurroundingHeader(int offset)
{
    LOGMESSAGE2("sendSurroundingHeader",offset);

    XEvent xev;
    /* Send the cursor offset in the surrounding */
    memset(&xev, 0, sizeof(XEvent));
    xev.xclient.message_type = ATOM(_HILDON_IM_SURROUNDING);
    xev.xclient.format = HILDON_IM_SURROUNDING_FORMAT;

    HildonIMSurroundingMessage *surroundingMsg = reinterpret_cast<HildonIMSurroundingMessage *>(&xev.xclient.data);
    surroundingMsg->commit_mode = commitMode;
    surroundingMsg->cursor_offset = offset;

    sendX11Event(&xev);
}

/*! Notify IM of any input mode changes
 */
void QHildonInputContext::inputModeChanged(){
    LOGMESSAGE1("inputModeChanged");

#if 0
  //TODO
  if ((input_mode & HILDON_GTK_INPUT_MODE_ALPHA) == 0  &&
      (input_mode & HILDON_GTK_INPUT_MODE_HEXA)  == 0  &&
      ( (input_mode & HILDON_GTK_INPUT_MODE_NUMERIC) != 0 ||
        (input_mode & HILDON_GTK_INPUT_MODE_TELE)    != 0))
  {
    self->mask = HILDON_IM_LEVEL_LOCK_MASK | HILDON_IM_LEVEL_STICKY_MASK;
  }
  else
  {
    self->mask &= ~HILDON_IM_LEVEL_LOCK_MASK;
    self->mask &= ~HILDON_IM_LEVEL_STICKY_MASK;
  }
#endif
  /* Notify IM of any input mode changes in cases where the UI is
     already visible. */
  sendInputMode();
}

void QHildonInputContext::sendInputMode(){
    LOGMESSAGE1("sendInputMode")

    Window w = findHildonIm();

    if (!w){
        return;
    }

    XEvent ev;
    memset(&ev, 0, sizeof(XEvent));

    ev.xclient.type = ClientMessage;
    ev.xclient.window = w;
    ev.xclient.message_type = ATOM(_HILDON_IM_INPUT_MODE);
    ev.xclient.format = HILDON_IM_INPUT_MODE_FORMAT;

    HildonIMInputModeMessage *msg = reinterpret_cast<HildonIMInputModeMessage *>(&ev.xclient.data);
    msg->input_mode = static_cast<HildonGtkInputMode>(inputMode);
    msg->default_input_mode = static_cast<HildonGtkInputMode>(HILDON_GTK_INPUT_MODE_FULL);

    XSendEvent(X11->display, w, false, 0, &ev);
    XSync(X11->display, False);
}

/*! In redirect mode we use a proxy widget (fullscreen vkb). When the cursor position
 *  changes there, the HIM update the cursor position in the client (Qt application)
 */
void QHildonInputContext::setClientCursorLocation(int offsetIsRelative, int cursorOffset)
{
    LOGMESSAGE3("setClientCursorLocation", offsetIsRelative, cursorOffset)

    if (!offsetIsRelative){
        qWarning("setClientCursorLocation can't manage absolute cursor Offsets");
        return;
    }

    //Move the cursor
    //NOTE: To move the cursor changes in customWidget::inputMethodEvent(QInputMethodEvent *e) are needed.
    QInputMethodEvent e;
    e.setCommitString(QString(), cursorOffset,0);
    sendEvent(e);
}

void QHildonInputContext::setMaskState(int *mask,
                                              HildonIMInternalModifierMask lock_mask,
                                              HildonIMInternalModifierMask sticky_mask,
                                              bool was_press_and_release)
{
    LOGMESSAGE3("setMaskState", lock_mask, sticky_mask)
    LOGMESSAGE3(" - ", "mask=", *mask)

   /* Locking Fn is disabled in TELE and NUMERIC */
    if (!(inputMode & HILDON_GTK_INPUT_MODE_ALPHA) &&
        !(inputMode & HILDON_GTK_INPUT_MODE_HEXA)  &&
        ((inputMode & HILDON_GTK_INPUT_MODE_TELE) ||
         (inputMode & HILDON_GTK_INPUT_MODE_NUMERIC))
       ) {
        if (*mask & lock_mask){
            /* already locked, remove lock and set it to sticky */
            *mask &= ~(lock_mask | sticky_mask);
            *mask |= sticky_mask;
        }else if (*mask & sticky_mask){
            /* the key is already sticky, it's fine */
        }else if (was_press_and_release){
            /* Pressing the key for the first time stickies the key for one character,
             * but only if no characters were entered while holding the key down */
            *mask |= sticky_mask;
        }
        return;
    }

    if (*mask & lock_mask)
    {
        /* Pressing the key while already locked clears the state */
        if (lock_mask & HILDON_IM_SHIFT_LOCK_MASK)
            sendHildonCommand(HILDON_IM_SHIFT_UNLOCKED, realFocus);
        else if (lock_mask & HILDON_IM_LEVEL_LOCK_MASK)
            sendHildonCommand(HILDON_IM_MOD_UNLOCKED, realFocus);

        *mask &= ~(lock_mask | sticky_mask);
    } else if (*mask & sticky_mask) {
        /* When the key is already sticky, a second press locks the key */
        *mask |= lock_mask;

        if (lock_mask & HILDON_IM_SHIFT_LOCK_MASK)
            sendHildonCommand(HILDON_IM_SHIFT_LOCKED, realFocus);
        else if (lock_mask & HILDON_IM_LEVEL_LOCK_MASK)
            sendHildonCommand(HILDON_IM_MOD_LOCKED, realFocus);
    }else if (was_press_and_release){
        /* Pressing the key for the first time stickies the key for one character,
         * but only if no characters were entered while holding the key down */
        *mask |= sticky_mask;
    }

}

#endif
