//OssoWrapper.cpp: Implementation of the C++ OSSO / libplayback wrapper class.

#include "OssoWrapper.h"

//libplayback callbacks

static void libplayback_state_request_handler(pb_playback_t   *pb,
                                              enum pb_state_e req_state,
                                              pb_req_t        *req,
                                              void            *data)
{
    //playback state change request received via libplayback
    pb_playback_req_completed(pb, req);
    OssoWrapper::m_pStaticOssoWrapper->playbackStateRequestHandlerCallback(req_state);
}

static void libplayback_state_reply(pb_playback_t *pb, enum pb_state_e granted_state, const char *reason,
                                    pb_req_t* req, void *data)
{ //reply to playback state change request received via libplayback

    pb_playback_req_completed(pb, req); //force state change
    OssoWrapper::m_pStaticOssoWrapper->playbackStateReplyCallback(granted_state, reason);
}

OssoWrapper* OssoWrapper::m_pStaticOssoWrapper; //static pointer to this (current) OssoWrapper instance. Used for callback to signal
                                          //handling

OssoWrapper::OssoWrapper(const QString& strAppName, const QString& strAppVersion, QObject *parent)
    : QObject(parent)
{
    //default contructor, init the libosso context

    QByteArray appName = strAppName.toLatin1();
    QByteArray appVer = strAppVersion.toLatin1();

    m_pOssoContext = NULL;
    m_pPlayback = NULL;

    OssoWrapper::m_pStaticOssoWrapper = this;

    m_pOssoContext = osso_initialize(appName.data(),appVer.data(),FALSE,NULL);
}

OssoWrapper::~OssoWrapper()
{
    //deinitialise / free the osso context
    if(m_pPlayback)
        pb_playback_destroy(m_pPlayback);
    if(m_pOssoContext)
        osso_deinitialize(m_pOssoContext);
}

bool OssoWrapper::executePlugin(const QString &strPluginName)
{
    //load and execute the specfied shared object (so) plugin (e.g., maemo control panel plugins)

    QByteArray pluginName = strPluginName.toLatin1();

    if(m_pOssoContext)
    {
        if(osso_cp_plugin_execute(m_pOssoContext,pluginName.data(),this->parent(),TRUE)==OSSO_OK)
            return true;
        else
            return false;
    }
    else
    {
        return false;
    }
}

bool OssoWrapper::initOssoPlayback(const QString& audioStream)
{
    QByteArray streamID;

    if(m_pOssoContext)
    {
        m_pPlayback = pb_playback_new_2((DBusConnection*)osso_get_dbus_connection(m_pOssoContext),
                                        PB_CLASS_MEDIA,
                                        PB_FLAG_AUDIO,
                                        PB_STATE_STOP,
                                        (PBStateRequest)libplayback_state_request_handler,
                                        NULL);
        if(m_pPlayback)
        { //set the id of stream that handles the playback (e.g., id of pulseaudio stream)
            streamID = audioStream.toLatin1();
            pb_playback_set_stream(m_pPlayback, streamID.data());
            return true;
        }
        else
            return false;
    }
    else
        return false;
}

void OssoWrapper::requestPlaybackStateChange(enum pb_state_e pb_new_state)
{
    //request to change playbackstate to pb_new_state
    if(m_pPlayback)
    {
        pb_playback_req_state(m_pPlayback, pb_new_state, (PBStateReply)libplayback_state_reply, NULL);
    }
}

void OssoWrapper::playbackStateRequestHandlerCallback(enum pb_state_e requestedState)
{
    //handler called when state request received via libplayback
    emit playbackStateChangeRequest(requestedState); //emit corresponding signal
}

void OssoWrapper::playbackStateReplyCallback(enum pb_state_e grantedState, const char* reason)
{
    //handler called when state change reply received via libplayback
    QString strReason = ""; //reason why state change request denied (if set)
    if(reason)
    {
        strReason = reason;
    }

    emit playbackStateReply(grantedState, strReason);
}

